﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Config.h>

#if !NN_BUILD_CONFIG_OS_SUPPORTS_WIN32
    #error  "OS 種別として Win32 が指定されていません。"
#endif

#include <nn/nn_Windows.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_MemoryHeapCommon.h>

//--------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------
// Windows 環境において、メモリヒープ領域の初回獲得時に予約するサイズ。
//
// Windows 環境ではヒープ直後の領域も他の用途ですぐに使用されてしまうため、
// メモリヒープの伸長に失敗することが多くなる。そのため、初回獲得時に
// ある程度の大きさの空間を予約しておき、この予約サイズまでであれば
// いつでもメモリヒープを伸長できるようにしておく。
//
// 32bit 環境ではとりあえず 1GB の空間を予約しダメなら減らしていく。
// 64bit 環境では今後の拡張を見据え 32GB の空間を予約しておく。
//
// CLI 版の場合は、PC ツールが os の管理するヒープ外でより多くのメモリを
// 利用できるように、デフォルトの予約サイズを控えめにしておく。（暫定対処）
//
#if defined(NN_BUILD_CONFIG_ADDRESS_32)
#if !defined(NN_OS_FOR_NONE_CLI)
const size_t DefaultReserveHeapSize     =    1u * 1024u * 1024u * 1024u;
const size_t DefaultReserveHeapSizeLAA  =    2u * 1024u * 1024u * 1024u;
#else
const size_t DefaultReserveHeapSize     =          256u * 1024u * 1024u;
const size_t DefaultReserveHeapSizeLAA  =          256u * 1024u * 1024u;
#endif
#elif defined(NN_BUILD_CONFIG_ADDRESS_64)
const size_t DefaultReserveHeapSize     = 32ull * 1024u * 1024u * 1024u;
#else
    #error "Unexpected address space size is defined."
#endif

//--------------------------------------------------------------------------
//  MemoryHeapManagerImplByWin32 クラス
//
class MemoryHeapManagerImplByWin32
{
public:
    MemoryHeapManagerImplByWin32() NN_NOEXCEPT;

    Result SetHeapSize(uintptr_t* pOutAddress, size_t size) NN_NOEXCEPT;
    size_t      GetReservedSize() const NN_NOEXCEPT
    {
        return m_AlignedReservedHeapSize;
    }

private:
    bool    ReserveVirtualSpace(uintptr_t address, size_t size) NN_NOEXCEPT;
    void    ReleaseVirtualSpace() NN_NOEXCEPT;
    bool    CommitMemory(size_t size) NN_NOEXCEPT;
    void    DecommitMemory(uintptr_t address, size_t size) NN_NOEXCEPT;

private:
    LPVOID  m_RealReservedAddress;          // VirtualAlloc() で予約した本当のアドレス
    size_t  m_RealReservedSize;             // VirtualAlloc() で予約した本当のサイズ
                                            // 本当のサイズは解放時に不要なので保持せず
    LPVOID  m_AlignedReservedHeapAddress;   // ヒープ領域のアライメント済み先頭アドレス
    size_t  m_AlignedReservedHeapSize;      // ヒープ領域のアライメント済み予約サイズ
    size_t  m_CommitedSize;                 // ↑この予約済みの中でコミット済みサイズ
};

typedef MemoryHeapManagerImplByWin32    MemoryHeapManagerImpl;

}   // namespace detail
}}  // namespace nn::os

