﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/diag/text/diag_SdkTextOs.h>

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_MemoryAttribute.h>

#include "os_Diag.h"
#include "os_Common.h"
#include "os_MemoryHeapManager.h"

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>


//--------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {

//----------------------------------------------------------------------------
// メモリ属性の変更
void SetMemoryAttributeImpl(uintptr_t beginAddress, size_t regionSize, MemoryAttribute attribute) NN_NOEXCEPT
{
    Bit32 mask = svc::MemoryAttribute_Uncached;
    Bit32 attr = 0;

    switch (attribute)
    {
    case nn::os::MemoryAttribute_Normal:
        {
            attr = 0;
        }
        break;
    case nn::os::MemoryAttribute_Uncached:
        {
            attr = svc::MemoryAttribute_Uncached;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    // カーネルのメモリ管理単位に分割して svc::SetMemoryAttribute() を発行する
    auto address  = beginAddress;
    auto leftSize = regionSize;
    while (leftSize > 0)
    {
        // メモリ情報を取得
        svc::MemoryInfo  memoryInfo;
        svc::PageInfo    pageInfo;
        auto result = svc::QueryMemory(&memoryInfo, &pageInfo, address);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // ページ属性変更対象領域のサイズ計算
        size_t size = (memoryInfo.baseAddress + memoryInfo.size) - address;
               size = (size > leftSize) ? leftSize : size;

        // メモリ属性の変更
        if (memoryInfo.attribute != attr)
        {
            auto result = svc::SetMemoryAttribute(address, size, mask, attr);
            if (result.IsFailure())
            {
                NN_SDK_LOG(NN_TEXT_OS("nn::os::SetMemoryAttribute(): メモリ属性の変更に失敗しました\n"));
                NN_SDK_LOG("   Address    = 0x%p\n", address);
                NN_SDK_LOG("   Size       = 0x%zx\n", size);
                NN_SDK_LOG("   State      = 0x%x\n", memoryInfo.state);
                NN_SDK_LOG("   Permission = 0x%x\n", memoryInfo.permission);
                NN_SDK_LOG("   Attribute  = 0x%x -> 0x%x\n", memoryInfo.attribute, attr);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }

        address  += size;
        leftSize -= size;
    }
}

//----------------------------------------------------------------------------

}   // namespace detail
}}  // namespace nn::os

