﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "os_Common.h"

#include <nn/os/os_Config.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_Windows.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_EventCommon.h>
#include <nn/os/os_InterruptEventCommon.h>

#include "os_Diag.h"
#include "os_Utility.h"
#include "os_InterruptEventImpl.h"
#include "os_TimeoutHelper.h"
#include "os_GiantLock.h"


namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------
// Win32 Event をアトミックにクリアする。クリア前の状態が返値となる。
bool InterruptEventImplByWin32::ResetEventSignal() NN_NOEXCEPT
{
    std::lock_guard<GiantLock> lock(*GetGiantLockInstance());

    DWORD ret = ::WaitForSingleObject(m_Handle, 0);
    if (ret == WAIT_OBJECT_0)
    {
        // シグナル状態だった場合はクリアして true を返す
        ::ResetEvent( m_Handle );
        return true;
    }

    return false;
}

//--------------------------------------------------------------------------
// コンストラクタ
InterruptEventImplByWin32::InterruptEventImplByWin32(InterruptName name, EventClearMode clearMode) NN_NOEXCEPT
{
    HANDLE handle = ::OpenEvent( EVENT_MODIFY_STATE | SYNCHRONIZE, FALSE,
                                 reinterpret_cast<LPCTSTR>(name) );

    NN_SDK_ASSERT( handle != NULL, NN_TEXT_OS("nn::os::InitializeInterruptEvent(): イベントのオープンに失敗しました。\n指定された名前の Win32 イベントが見つかりません。\n（EventName=\x22%s\x22）"), name );

    m_Handle            = handle;
    m_ManualReset       = (clearMode == EventClearMode_ManualClear);
}

//--------------------------------------------------------------------------
// デストラクタ
InterruptEventImplByWin32::~InterruptEventImplByWin32() NN_NOEXCEPT
{
    BOOL ret = ::CloseHandle( m_Handle );
    NN_SDK_ASSERT( ret != 0, NN_TEXT_OS("nn::os::FinalizeInterruptEvent(): イベントの削除に失敗しました。"));
    NN_UNUSED(ret);
}

//--------------------------------------------------------------------------
// クリア
void InterruptEventImplByWin32::Clear() NN_NOEXCEPT
{
    BOOL ret = ::ResetEvent( m_Handle );
    NN_SDK_ASSERT( ret != 0, NN_TEXT_OS("nn::os::ClearInterruptEvent(): イベントのクリアに失敗しました。"));
    NN_UNUSED(ret);
}

//--------------------------------------------------------------------------
// 割込みイベント待機
void InterruptEventImplByWin32::Wait() NN_NOEXCEPT
{
    for (;;)
    {
        DWORD ret = ::WaitForSingleObject( m_Handle, INFINITE );
        NN_SDK_ASSERT( ret == WAIT_OBJECT_0, NN_TEXT_OS("nn::os::WaitInterruptEvent(): イベントの待機でエラーが発生しました。"));
        NN_UNUSED(ret);

        if (m_ManualReset)
        {
            return;
        }

        bool result = ResetEventSignal();
        if (result == true)
        {
            return;
        }
    }
}

//--------------------------------------------------------------------------
// 割込みイベントのポーリング
bool InterruptEventImplByWin32::TryWait() NN_NOEXCEPT
{
    if (!m_ManualReset)
    {
        return ResetEventSignal();
    }

    DWORD result = ::WaitForSingleObject( m_Handle, 0 );

    NN_SDK_ASSERT( (result == WAIT_OBJECT_0) || (result == WAIT_TIMEOUT), NN_TEXT_OS("nn::os::TryWaitInterruptEvet(): イベントのポーリングでエラーが発生しました。"));

    return result == WAIT_OBJECT_0;
}

//--------------------------------------------------------------------------
// 割込みイベント待機（タイムアウト付き）
bool InterruptEventImplByWin32::TimedWait(TimeSpan timeout) NN_NOEXCEPT
{
    detail::TimeoutHelper   tmout( timeout );

    do
    {
        DWORD result = ::WaitForSingleObject( m_Handle,
                              tmout.GetLeftTimeOnTarget() );

        if (result == WAIT_OBJECT_0)
        {
            if (m_ManualReset)
            {
                return true;
            }

            bool ret = ResetEventSignal();
            if (ret == true)
            {
                return true;
            }
        }

    } while ( !tmout.IsTimedout() );

    return false;
}


}  // namespace detail
}} // namespace nn::os

