﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "os_Common.h"

#include <nn/os/os_Config.h>

#if !defined(NN_BUILD_CONFIG_OS_SUPPORTS_WIN32)
    #error "OS 種別として Win32 が指定されていません。"
#endif

#include <new>
#include <mutex>
#include <chrono>
#include <condition_variable>

#include <nn/nn_Windows.h>
#include <nn/nn_Macro.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/os/os_ConditionVariableCommon.h>
#include <nn/os/detail/os_InternalConditionVariable-os.win32.h>

#include "os_TimeoutHelper.h"
#include "os_InternalCriticalSectionTypes-os.win32.h"
#include "os_InternalConditionVariableTypes-os.win32.h"

//--------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------
InternalConditionVariableImplByWin32::InternalConditionVariableImplByWin32() NN_NOEXCEPT
{
    this->Initialize();
}

//--------------------------------------------------------------------------
void InternalConditionVariableImplByWin32::Initialize() NN_NOEXCEPT
{
#if defined(NN_OS_USE_STD_CONDITION_VARIABLE_CLASS)
    new ( &(Get(m_conditionVariableWin32).conditionVariable) ) std::condition_variable_any;
#else
    // Win32 の条件変数の初期化
    ::InitializeConditionVariable( &Get(m_conditionVariableWin32).conditionVariable );
#endif
}

//--------------------------------------------------------------------------
InternalConditionVariableImplByWin32::~InternalConditionVariableImplByWin32() NN_NOEXCEPT
{
#if defined(NN_OS_USE_STD_CONDITION_VARIABLE_CLASS)
    Get(m_conditionVariableWin32).conditionVariable.~condition_variable_any();
#else
    // Win32 の条件変数の削除
    // Do nothing
#endif
}

//--------------------------------------------------------------------------
void InternalConditionVariableImplByWin32::Wait(InternalCriticalSection* p) NN_NOEXCEPT
{
#if defined(NN_OS_USE_STD_CONDITION_VARIABLE_CLASS)
    // すでにロック済みのはずなので std::adopt_lock を指定する。
    std::unique_lock<InternalCriticalSection> lock( *p, std::adopt_lock );
    Get(m_conditionVariableWin32).conditionVariable.wait( lock );

    // ここではアンロックしないため release() を発行しておく。
    lock.release();
#else
    InternalCriticalSectionImplByWin32* cs = static_cast<InternalCriticalSectionImplByWin32*>( p->Get() );

    // Win32 の条件変数を待機
    SleepConditionVariableCS( &Get(m_conditionVariableWin32).conditionVariable,
                              &Get(cs->m_criticalSectionWin32).criticalSection,
                              INFINITE );
#endif
}

//--------------------------------------------------------------------------
ConditionVariableStatus InternalConditionVariableImplByWin32::TimedWait(InternalCriticalSection* p, const TimeoutHelper& timeout) NN_NOEXCEPT
{
#if defined(NN_OS_USE_STD_CONDITION_VARIABLE_CLASS)
    // すでにロック済みのはずなので std::adopt_lock を指定する。
    std::unique_lock<InternalCriticalSection> lock( *p, std::adopt_lock );

    const std::chrono::milliseconds diffTime(timeout.GetLeftTimeOnTarget());
    auto absTime  = std::chrono::system_clock::now() + diffTime;
    auto cvStatus = Get(m_conditionVariableWin32).conditionVariable.wait_until( lock, absTime );

    // ここではアンロックしないため release() を発行しておく。
    lock.release();
    return cvStatus == std::cv_status::timeout
                    ? ConditionVariableStatus_Timeout
                    : ConditionVariableStatus_NoTimeout;
#else
    InternalCriticalSectionImplByWin32* cs = static_cast<InternalCriticalSectionImplByWin32*>( p->Get() );

    // Win32 の条件変数を待機（タイムアウト付き）
    BOOL result = SleepConditionVariableCS(
                              &Get(m_conditionVariableWin32).conditionVariable,
                              &Get(cs->m_criticalSectionWin32).criticalSection,
                              timeout.GetLeftTimeOnTarget() );

    // 成功 or タイムアウト成立ならリターン
    if (result == 0)
    {
        DWORD errorCode = GetLastError();
        if (errorCode == ERROR_TIMEOUT)
        {
            return ConditionVariableStatus_Timeout;
        }
    }

    // この返値には spurious なども含む
    return ConditionVariableStatus_NoTimeout;
#endif
}

//--------------------------------------------------------------------------
void InternalConditionVariableImplByWin32::Signal() NN_NOEXCEPT
{
#if defined(NN_OS_USE_STD_CONDITION_VARIABLE_CLASS)
    Get(m_conditionVariableWin32).conditionVariable.notify_one();
#else
    // Win32 の条件変数へ通知（単一スレッド）
    ::WakeConditionVariable( &Get(m_conditionVariableWin32).conditionVariable );
#endif
}

//--------------------------------------------------------------------------
void InternalConditionVariableImplByWin32::Broadcast() NN_NOEXCEPT
{
#if defined(NN_OS_USE_STD_CONDITION_VARIABLE_CLASS)
    Get(m_conditionVariableWin32).conditionVariable.notify_all();
#else
    // Win32 の条件変数へ通知（全スレッド）
    ::WakeAllConditionVariable( &Get(m_conditionVariableWin32).conditionVariable );
#endif
}

}   // namespace detail
}}  // namespace nn::os

