﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Config.h>

#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/detail/os_InternalCriticalSection.h>
#include <nn/util/util_TinyMt.h>


#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include "os_AddressSpaceAllocatorImpl-os.win32.h"
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include "os_AddressSpaceAllocatorImpl-os.horizon.h"
#else
    #error   "未サポートの OS 種別が指定されています。"
#endif


namespace nn { namespace os {

namespace detail {

//--------------------------------------------------------------------------
// 仮想アドレス空間の管理用クラス
//
//  本クラスでは、指定した仮想アドレス空間を ASLR 対象領域として管理します。
//  必要なサイズ分の空間を、管理領域中のランダムなアドレスから確保できます。
//  確保される空間の後方にはガード空間が設けられます。
//
//  管理対象の領域はコンストラクタで指定します。
//  このとき、以下の図のように、指定する空間の前後にガードサイズ以上の未使用空間が
//  あることを期待します。
//
//    |   未使用空間   | ) guardSize 以上
//    +----------------+ <- beginAddress
//    |                |
//    |    管理空間    |
//    |                |
//    +----------------+ <- endAddress
//    |   未使用空間   | ) guardSize 以上
//
//  管理空間の前後の未使用空間をガード空間とみなし、管理空間全体から空間を確保します。
//
class AddressSpaceAllocator
{
public:
    //-----------------------------------------------------------------
    // コンストラクタ
    AddressSpaceAllocator(uint64_t beginAddress, uint64_t endAddress, size_t guardSize) NN_NOEXCEPT;

    //-----------------------------------------------------------------
    // 仮想アドレス空間から指定サイズのアドレス空間が収まる空き空間を獲得する。
    void* AllocateSpace(size_t size, size_t align) NN_NOEXCEPT;

    //-----------------------------------------------------------------
    // マップした領域の前後にガード空間があるか確認する
    bool CheckGuardSpace(uintptr_t address, size_t size, size_t guardSize) NN_NOEXCEPT;

private:
    //-----------------------------------------------------------------
    // page と pageNum の示す範囲が既に確保済みな領域と重なっているか調べる
    bool GetNextNonOverlappedNodeUnsafe(uintptr_t page, size_t pageNum) NN_NOEXCEPT;

private:
    InternalCriticalSection   m_CriticalSection;  // クリティカルセクション
    uintptr_t                 m_BeginPage;        // 先頭ノード
    uintptr_t                 m_EndPage;          // 末尾ノード
    size_t                    m_GuardPageCount;   // ガード用ページ数

};

//--------------------------------------------------------------------------

}   // namespace detail
}}  // namespace nn::os

