﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Result.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_Tick.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/util/util_BitUtil.h>

#include "os_Diag.h"
#include "os_Common.h"
#include "os_AddressSpaceAllocatorTypes.h"
#include "os_AddressSpaceAllocatorImpl.h"
#include "os_ThreadManager.h"
#include "os_RngManager.h"


//-----------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {

//-----------------------------------------------------------------------------
// コンストラクタ
AddressSpaceAllocator::AddressSpaceAllocator(uint64_t beginAddress, uint64_t endAddress, size_t guardSize) NN_NOEXCEPT
{
    // 空間の前方・後方にガードサイズ以上の空きが必要
    NN_SDK_ASSERT(beginAddress >= guardSize);
    NN_SDK_ASSERT(endAddress + guardSize >= endAddress); // ガードサイズを足してもオーバーフローしなければ OK

    // ガードのページ数
    m_GuardPageCount = nn::util::DivideUp(guardSize, MemoryPageSize);

    // 先頭ノードと末端ノードのアドレスを設定する。
    m_BeginPage = static_cast<uintptr_t>(beginAddress / MemoryPageSize);
    m_EndPage = static_cast<uintptr_t>(endAddress / MemoryPageSize) + m_GuardPageCount;  // ガード込みでピッタリ確保できるように
}

//-----------------------------------------------------------------
// page と pageCount の示す範囲が既に確保済みな領域と重なっているか調べる
bool AddressSpaceAllocator::GetNextNonOverlappedNodeUnsafe(uintptr_t page, size_t pageCount) NN_NOEXCEPT
{
    // 桁あふれが起きないことをチェック
    NN_SDK_ASSERT( page < page + pageCount );

    return CheckFreeSpace(page * MemoryPageSize, pageCount * MemoryPageSize);
}

//-----------------------------------------------------------------------------
// 仮想アドレス空間から指定サイズのアドレス空間が収まる空き空間を獲得する。
// 確保するアドレス空間の後方にガード用空間を設ける。
// size にはガードを含まないサイズ（実際に利用したいサイズ）を指定する。
void* AddressSpaceAllocator::AllocateSpace(size_t size, size_t align) NN_NOEXCEPT
{
    // ガード空間を含めて必要なページ数
    // ガード空間のために +m_GuardPageCount*2 する（前後それぞれにガードページ分の空きがあることを確認する）
    size_t pageCount = nn::util::DivideUp(size, MemoryPageSize) + m_GuardPageCount * 2;

    // 必要なアラインをページ数に換算
    NN_SDK_ASSERT(align % MemoryPageSize == 0);
    size_t alignPageCount = align / MemoryPageSize;

    // m_BeginNode と m_EndNode から乱数の範囲を計算する
    uintptr_t randRangeBegin = m_BeginPage;
    uintptr_t randRangeEnd   = m_EndPage - pageCount;

    if (randRangeBegin > randRangeEnd)
    {
        return nullptr;
    }

    // TORIAEZU:
    // 確保可能なアドレスが見つかるまで繰り返す
    // 512 回やって見つからなければ nullptr を返す
    // 空間が混雑してきたら別のアプローチを取った方がよい
    for (int loop=0; loop<512; ++loop)
    {
        {   // クリティカルセクション
            std::lock_guard<InternalCriticalSection> lock(m_CriticalSection);

            // randRangeBegin～randRangeEnd の範囲で乱数を生成
            uintptr_t page = (GetRngManagerInstance()->GenerateRandomU64() % ( randRangeEnd - randRangeBegin )) + randRangeBegin;
            page = nn::util::align_down(page, alignPageCount);

            if (GetNextNonOverlappedNodeUnsafe(page, pageCount))
            {
                return reinterpret_cast<void*>((page + m_GuardPageCount) * MemoryPageSize);
            }
        }
    }
    return nullptr;
}

bool AddressSpaceAllocator::CheckGuardSpace(uintptr_t address, size_t size, size_t guardSize) NN_NOEXCEPT
{
    if (CheckFreeSpace(address - guardSize, guardSize) && CheckFreeSpace(address + size, guardSize))
    {
        return true;
    }
    return false;
}

}   // namespace detail
}}  // namespace nn::os
