﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Nintendo;
using Nintendo.ServiceFramework;
using Nintendo.ServiceFramework.CppCode;
using Nintendo.ServiceFramework.Hipc;
using nn;
using nn.sf;
using std;

namespace nn.ncm
{
    [CppRefPath(@"<nn/ncm/ncm_ContentStorage.h>")]
    [ExternalStruct(16, 8)]
    public struct PlaceHolderId {}

    [CppRefPath(@"<nn/ncm/ncm_ContentId.h>")]
    [ExternalStruct(16, 4)]
    public struct ContentId{}

    [CppRefPath(@"<nn/ncm/ncm_ContentInfo.h>")]
    [ExternalStruct(24, 8)]
    public struct ContentInfo{}

    [CppRefPath(@"<nn/ncm/ncm_ContentMeta.h>")]
    [ExternalStruct(16, 8)]
    public struct ContentMetaInfo { }

    [CppRefPath(@"<nn/ncm/ncm_ContentStorage.h>")]
    [ExternalStruct(768, 4)]
    [LargeData]
    public struct Path{}

    [CppRefPath(@"<nn/fs/fs_RightsId.h>")]
    [ExternalStruct(24, 8)]
    public struct RightsId{}

    [CppRefPath(@"<nn/ncm/ncm_IContentStorage.h>")]
    public interface IContentStorage : IServiceObject
    {
        [MethodId(0)]
        Result GeneratePlaceHolderId(Out<PlaceHolderId> outValue);

        [MethodId(1)]
        Result CreatePlaceHolder(PlaceHolderId placeHolderId, ContentId contentId, int64_t size);

        [MethodId(2)]
        Result DeletePlaceHolder(PlaceHolderId id);

        [MethodId(3)]
        Result HasPlaceHolder(Out<bool> outValue, PlaceHolderId id);

        [MethodId(4)]
        Result WritePlaceHolder(PlaceHolderId id, int64_t offset, InBuffer buffer);

        [MethodId(5)]
        Result Register(PlaceHolderId placeHolderId, ContentId contentId);

        [MethodId(6)]
        Result Delete(ContentId id);

        [MethodId(7)]
        Result Has(Out<bool> outValue, ContentId id);

        [MethodId(8)]
        Result GetPath(Out<Path> outValue, ContentId id);

        [MethodId(9)]
        Result GetPlaceHolderPath(Out<Path> outValue, PlaceHolderId id);

        [MethodId(10)]
        Result CleanupAllPlaceHolder();

        [MethodId(11)]
        Result ListPlaceHolder(Out<int32_t> outCount, [BufferTransferMode(BufferTransferMode.MapAlias)] OutArray<PlaceHolderId> outList);

        [MethodId(12)]
        Result GetContentCount(Out<int32_t> outCount);

        [MethodId(13)]
        Result ListContentId(Out<int32_t> outCount, [BufferTransferMode(BufferTransferMode.MapAlias)] OutArray<ContentId> outList, int32_t offset);

        [MethodId(14)]
        Result GetSizeFromContentId(Out<int64_t> outValue, ContentId id);

        [MethodId(15)]
        Result DisableForcibly();

        [MethodId(16)]
        Result RevertToPlaceHolder(PlaceHolderId placeHolderId, ContentId contentId, ContentId postContentId);

        [MethodId(17)]
        Result SetPlaceHolderSize(PlaceHolderId placeHolderId, int64_t size);

        [MethodId(18)]
        Result ReadContentIdFile(OutBuffer buffer, ContentId contentId, int64_t offset);

        [MethodId(19)]
        Result GetRightsIdFromPlaceHolderId(Out<RightsId> outValue, PlaceHolderId id);

        [MethodId(20)]
        Result GetRightsIdFromContentId(Out<RightsId> outValue, ContentId id);

        [MethodId(21)]
        Result WriteContentForDebug(ContentId id, int64_t offset, InBuffer buffer);

        [MethodId(22)]
        Result GetFreeSpaceSize(Out<int64_t> outValue);

        [MethodId(23)]
        Result GetTotalSpaceSize(Out<int64_t> outValue);

        [MethodId(24)]
        Result FlushPlaceHolder();

        [MethodId(25)]
        Result GetSizeFromPlaceHolderId(Out<int64_t> outValue, PlaceHolderId id);

        [MethodId(26)]
        Result RepairInvalidFileAttribute();
    }

    [CppRefPath(@"<nn/ncm/ncm_ContentMetaKey.h>")]
    [ExternalStruct(16, 8)]
    public struct ContentMetaKey{}

    [CppRefPath(@"<nn/ncm/ncm_ContentMetaKey.h>")]
    [ExternalStruct(24, 8)]
    public struct ApplicationContentMetaKey{}

    [CppRefPath(@"<nn/ncm/ncm_ProgramId.h>")]
    [ExternalStruct(8, 8)]
    public struct ProgramId{}

    [CppRefPath(@"<nn/ncm/ncm_ContentMetaId.h>")]
    [ExternalStruct(8, 8)]
    public struct ApplicationId { }

    [CppRefPath(@"<nn/ncm/ncm_ContentMetaId.h>")]
    [ExternalStruct(8, 8)]
    public struct PatchId{}

    [CppRefPath(@"<nn/ncm/ncm_ContentType.h>")]
    [ExternalStruct(1, 1)]
    public struct ContentType{}

    [CppRefPath(@"<nn/ncm/ncm_ContentMetaKey.h>")]
    [ExternalStruct(1, 1)]
    public struct ContentMetaType{}

    [CppRefPath(@"<nn/ncm/ncm_ContentMetaKey.h>")]
    [ExternalStruct(1, 1)]
    public struct ContentInstallType{}

    [CppRefPath(@"<nn/ncm/ncm_IContentMetaDatabase.h>")]
    public interface IContentMetaDatabase : IServiceObject
    {
        [MethodId(0)]
        Result Set(ContentMetaKey key, InBuffer buffer);

        [MethodId(1)]
        Result Get(Out<uint64_t> outValue, ContentMetaKey key, OutBuffer buffer);

        [MethodId(2)]
        Result Remove(ContentMetaKey key);

        [MethodId(3)]
        Result GetContentIdByType(Out<ContentId> outValue, ContentMetaKey key, ContentType type);

        [MethodId(4)]
        Result ListContentInfo(Out<int32_t> outCount, [BufferTransferMode(BufferTransferMode.MapAlias)] OutArray<ContentInfo> outArray, ContentMetaKey key, int offset);

        [MethodId(5)]
        Result List(Out<int32_t> outTotal, Out<int32_t> outCount, [BufferTransferMode(BufferTransferMode.MapAlias)] OutArray<ContentMetaKey> outArray, ContentMetaType type, ApplicationId appId, Bit64 min, Bit64 max, ContentInstallType installType);

        [MethodId(6)]
        Result GetLatestContentMetaKey(Out<ContentMetaKey> outValue, nn.Bit64 id);

        [MethodId(7)]
        Result ListApplication(Out<int32_t> outTotal, Out<int32_t> outCount, [BufferTransferMode(BufferTransferMode.MapAlias)] OutArray<ApplicationContentMetaKey> outArray, ContentMetaType type);

        [MethodId(8)]
        Result Has(Out<bool> outValue, ContentMetaKey key);

        [MethodId(9)]
        Result HasAll(Out<bool> outValue, [BufferTransferMode(BufferTransferMode.MapAlias)] InArray<ContentMetaKey> inArray);

        [MethodId(10)]
        Result GetSize(Out<uint64_t> outValue, ContentMetaKey key);

        [MethodId(11)]
        Result GetRequiredSystemVersion(Out<uint32_t> outValue, ContentMetaKey key);

        [MethodId(12)]
        Result GetPatchId(Out<PatchId> outValue, ContentMetaKey key);

        [MethodId(13)]
        Result DisableForcibly();

        [MethodId(14)]
        Result LookupOrphanContent([BufferTransferMode(BufferTransferMode.MapAlias)] OutArray<bool> outList, [BufferTransferMode(BufferTransferMode.MapAlias)] InArray<ContentId> contentList);

        [MethodId(15)]
        Result Commit();

        [MethodId(16)]
        Result HasContent(Out<bool> outValue, ContentMetaKey key, ContentId contentId);

        [MethodId(17)]
        Result ListContentMetaInfo(Out<int32_t> outCount, [BufferTransferMode(BufferTransferMode.MapAlias)] OutArray<ContentMetaInfo> outArray, ContentMetaKey key, int offset);

        [MethodId(18)]
        Result GetAttributes(Out<Bit8> outAttributes, ContentMetaKey key);

        [MethodId(19)]
        Result GetRequiredApplicationVersion(Out<uint32_t> outValue, ContentMetaKey key);

        [MethodId(20)]
        Result GetContentIdByTypeAndIdOffset(Out<ContentId> outValue, ContentMetaKey key, ContentType type, byte idOffset);
    }

    [CppRefPath(@"<nn/ncm/ncm_StorageId.h>")]
    [ExternalStruct(1, 1)]
    public struct StorageId { }

    [CppRefPath(@"<nn/ncm/ncm_IContentManager.h>")]
    public interface IContentManager : IServiceObject
    {
        [MethodId(0)]
        Result CreateContentStorage(StorageId storageId);

        [MethodId(1)]
        Result CreateContentMetaDatabase(StorageId storageId);

        [MethodId(2)]
        Result VerifyContentStorage(StorageId storageId);

        [MethodId(3)]
        Result VerifyContentMetaDatabase(StorageId storageId);

        [MethodId(4)]
        Result OpenContentStorage(Out<IContentStorage> outValue, StorageId storageId);

        [MethodId(5)]
        Result OpenContentMetaDatabase(Out<IContentMetaDatabase> outValue, StorageId storageId);

        [MethodId(8)]
        Result CleanupContentMetaDatabase(StorageId storageId);

        [MethodId(9)]
        Result ActivateContentStorage(StorageId storageId);

        [MethodId(10)]
        Result InactivateContentStorage(StorageId storageId);

        [MethodId(11)]
        Result ActivateContentMetaDatabase(StorageId storageId);

        [MethodId(12)]
        Result InactivateContentMetaDatabase(StorageId storageId);
    }
}
