﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_StorageUtil.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/detail/ncm_Log.h>

namespace nn { namespace ncm {

StorageList GetStorageList(StorageId storage) NN_NOEXCEPT
{
    StorageList list;
    switch (storage)
    {
    case StorageId::BuildInSystem:
    case StorageId::BuildInUser:
    case StorageId::SdCard:
        {
            list.Push(storage);
        }
        break;
    case StorageId::Any:
        {
            list.Push(StorageId::SdCard);
            list.Push(StorageId::BuildInUser);
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    return list;
}


Result SelectDownloadableStorage(StorageId* outValue, StorageId storageId, int64_t requiredSize) NN_NOEXCEPT
{
    auto list = GetStorageList(storageId);
    for (int i = 0; i < list.Count(); i++)
    {
        auto candidate = list[i];

        // storage は開けるけれども db は開けないというパターンがあるので、
        // どちらもチェックする
        ContentMetaDatabase db;
        if (OpenContentMetaDatabase(&db, candidate).IsFailure())
        {
            continue;
        }
        ContentStorage storage;
        if (OpenContentStorage(&storage, candidate).IsFailure())
        {
            continue;
        }
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        int64_t freeSpace;
        NN_RESULT_DO(storage.GetFreeSpaceSize(&freeSpace));
        if (requiredSize < freeSpace)
        {
            *outValue = candidate;
            NN_RESULT_SUCCESS;
        }
#else
        // horizon 以外ではサイズチェックができない
        NN_UNUSED(requiredSize);
        *outValue = list[i];
        NN_RESULT_SUCCESS;
#endif
    }

    NN_RESULT_THROW(ResultNotEnoughInstallSpace());
}

Result SelectPatchStorage(StorageId* outValue, StorageId storageId, PatchId id) NN_NOEXCEPT
{
    auto list = GetStorageList(storageId);
    uint32_t version = 0;  // パッチのバージョンが 0 のことは無いので、0 は未定と扱う
    *outValue = storageId; // 初期値は、候補ストレージと同じとする

    for (int i = 0; i < list.Count(); i++)
    {
        auto candidate = list[i];

        ContentMetaDatabase db;
        if (OpenContentMetaDatabase(&db, candidate).IsFailure())
        {
            continue;
        }
        ContentMetaKey key;
        NN_RESULT_TRY(db.GetLatest(&key, id.value))
            NN_RESULT_CATCH(ResultContentMetaNotFound) { continue; }
        NN_RESULT_END_TRY

        if (key.version > version)
        {
            version = key.version;
            *outValue = candidate;
        }
    }

    NN_RESULT_SUCCESS;
}

const char* GetStorageIdString(StorageId storageId) NN_NOEXCEPT
{
    switch( storageId )
    {
    case StorageId::BuiltInSystem:
        return "BuiltInSystem";
    case StorageId::BuiltInUser:
        return "BuiltInUser";
    case StorageId::GameCard:
        return "GameCard";
    case StorageId::Host:
        return "Host";
    case StorageId::None:
        return "None";
    case StorageId::SdCard:
        return "SdCard";
    default:
        return "(unknown)";
    }
}

const char* GetStorageIdStringForPlayReport(StorageId storageId) NN_NOEXCEPT
{
    // prepo はすでに Buil'd'In .. で送っているので、そちらで送り続ける必要がある
    switch( storageId )
    {
    case StorageId::BuildInSystem:
        return "BuildInSystem";
    case StorageId::BuildInUser:
        return "BuildInUser";
    case StorageId::Card:
        return "Card";
    case StorageId::Host:
        return "Host";
    case StorageId::None:
        return "None";
    case StorageId::SdCard:
        return "SdCard";
    default:
        return "(unknown)";
    }
}


}}
