﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/fs/fs_Bis.h>
#include <nn/util/util_StringUtil.h>

#include <nn/ncm/ncm_Config.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentManagerImpl.h>
#include <nn/ncm/ncm_ServiceName.h>

#include <nn/ncm/ncm_ContentManagerFactory.h>

namespace nn { namespace ncm {

    namespace {

        sf::ProxyObjectAllocator<64> g_ServiceObjectAllocator = NN_SF_PROXY_OBJECT_ALLOCATOR_INITIALIZER;
        class MyAllocatorInitializer
        {
        public:
            MyAllocatorInitializer() NN_NOEXCEPT
            {
                g_ServiceObjectAllocator.Initialize();
            }
        } g_MyAllocatorInitializer;

        Result GetSharedContentManagerFromServiceManager(sf::SharedPointer<IContentManager> *outValue) NN_NOEXCEPT
        {
#if defined( NN_BUILD_CONFIG_OS_WIN32 )
            ContentManagerConfig config = {};
            *outValue = CreateDefaultContentManager(config);
            NN_RESULT_SUCCESS;
#else
            return sf::CreateHipcProxyByName<IContentManager>(outValue, g_ServiceObjectAllocator.GetMemoryResource(), ContentManagerServiceName);
#endif
        }

        sf::ShimLibraryObjectHolder<IContentManager> g_ObjectHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
    }

    void Initialize() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.InitializeHolder(GetSharedContentManagerFromServiceManager));
    }

    void InitializeWithContentManagerObject(sf::SharedPointer<IContentManager> contentManager) NN_NOEXCEPT
    {
        g_ObjectHolder.InitializeHolderDirectly(contentManager);
    }

    void Finalize() NN_NOEXCEPT
    {
        g_ObjectHolder.FinalizeHolder();
    }

    Result CreateContentStorage(StorageId storageId) NN_NOEXCEPT
    {
        return g_ObjectHolder->CreateContentStorage(storageId);
    }

    Result CreateContentMetaDatabase(StorageId storageId) NN_NOEXCEPT
    {
        return g_ObjectHolder->CreateContentMetaDatabase(storageId);
    }

    Result VerifyContentStorage(StorageId storageId) NN_NOEXCEPT
    {
        return g_ObjectHolder->VerifyContentStorage(storageId);
    }

    Result VerifyContentMetaDatabase(StorageId storageId) NN_NOEXCEPT
    {
        return g_ObjectHolder->VerifyContentMetaDatabase(storageId);
    }

    Result OpenContentStorage(ContentStorage* outValue, StorageId storageId) NN_NOEXCEPT
    {
        sf::SharedPointer<::nn::ncm::IContentStorage> contentStorage;
        NN_RESULT_DO(g_ObjectHolder->OpenContentStorage(&contentStorage, storageId));

        *outValue = ContentStorage(std::move(contentStorage));

        NN_RESULT_SUCCESS;
    }

    Result OpenContentMetaDatabase(ContentMetaDatabase* outValue, StorageId storageId) NN_NOEXCEPT
    {
        sf::SharedPointer<::nn::ncm::IContentMetaDatabase> contentMetaDatabase;
        NN_RESULT_DO(g_ObjectHolder->OpenContentMetaDatabase(&contentMetaDatabase, storageId));

        *outValue = ContentMetaDatabase(std::move(contentMetaDatabase));

        NN_RESULT_SUCCESS;
    }

    Result CleanupContentMetaDatabase(StorageId storageId) NN_NOEXCEPT
    {
        return g_ObjectHolder->CleanupContentMetaDatabase(storageId);
    }

    Result ActivateContentStorage(StorageId storageId) NN_NOEXCEPT
    {
        return g_ObjectHolder->ActivateContentStorage(storageId);
    }

    Result InactivateContentStorage(StorageId storageId) NN_NOEXCEPT
    {
        return g_ObjectHolder->InactivateContentStorage(storageId);
    }

    Result ActivateContentMetaDatabase(StorageId storageId) NN_NOEXCEPT
    {
        return g_ObjectHolder->ActivateContentMetaDatabase(storageId);
    }

    Result InactivateContentMetaDatabase(StorageId storageId) NN_NOEXCEPT
    {
        return g_ObjectHolder->InactivateContentMetaDatabase(storageId);
    }

}}
