﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <string>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentStorageImpl.h>
#include <nn/ncm/ncm_ReadOnlyContentStorageImpl.h>
#include <nn/fs/fs_MountPrivate.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_File.h>
#include <nn/util/util_ScopeExit.h>

#include "ncm_FileSystemUtility.h"

namespace nn { namespace ncm {
namespace {
    void MakeContentPath(PathString* outValue, ContentId id, MakeContentPathFunction func, const char* rootPath) NN_NOEXCEPT
    {
        return func(outValue, id, rootPath);
    }

    void MakeGameCardContentMetaPath(PathString* outValue, ContentId id, MakeContentPathFunction func, const char* rootPath) NN_NOEXCEPT
    {
        PathString path;

        func(&path, id, rootPath);
        *outValue = path.MakeSubString(0, path.GetLength() - 4);
        outValue->Append(".cnmt.nca");
    }

    Result OpenContentIdFileImpl(fs::FileHandle* outValue, ContentId id, MakeContentPathFunction func, const char* rootPath) NN_NOEXCEPT
    {
        PathString path;
        MakeContentPath(&path, id, func, rootPath);

        NN_RESULT_TRY(nn::fs::OpenFile(outValue, path, nn::fs::OpenMode_Read))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                MakeGameCardContentMetaPath(&path, id, func, rootPath);
                NN_RESULT_DO(nn::fs::OpenFile(outValue, path, nn::fs::OpenMode_Read));
            }
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }
}

    ReadOnlyContentStorageImpl::ReadOnlyContentStorageImpl() NN_NOEXCEPT : m_MakeContentPathFunction(), m_IsDisabled(false) {}

    Result ReadOnlyContentStorageImpl::Initialize(const char* rootPath, MakeContentPathFunction func) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_MakeContentPathFunction);
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        m_RootPath.Assign(rootPath);
        m_MakeContentPathFunction = func;

        NN_RESULT_SUCCESS;
    }

    nn::Result ReadOnlyContentStorageImpl::GeneratePlaceHolderId(nn::sf::Out<nn::ncm::PlaceHolderId> outValue) NN_NOEXCEPT
    {
        NN_UNUSED(outValue);

        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    nn::Result ReadOnlyContentStorageImpl::CreatePlaceHolder(PlaceHolderId placeHolderId, ContentId contentId, int64_t size) NN_NOEXCEPT
    {
        NN_UNUSED(placeHolderId);
        NN_UNUSED(contentId);
        NN_UNUSED(size);

        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    nn::Result ReadOnlyContentStorageImpl::SetPlaceHolderSize(PlaceHolderId placeHolderId, int64_t size) NN_NOEXCEPT
    {
        NN_UNUSED(placeHolderId);
        NN_UNUSED(size);

        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    nn::Result ReadOnlyContentStorageImpl::DeletePlaceHolder(PlaceHolderId id) NN_NOEXCEPT
    {
        NN_UNUSED(id);

        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    nn::Result ReadOnlyContentStorageImpl::HasPlaceHolder(sf::Out<bool> outValue, PlaceHolderId id) const NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_UNUSED(id);

        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    nn::Result ReadOnlyContentStorageImpl::WritePlaceHolder(PlaceHolderId id, int64_t offset, sf::InBuffer buffer) NN_NOEXCEPT
    {
        NN_UNUSED(id);
        NN_UNUSED(offset);
        NN_UNUSED(buffer);

        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    nn::Result ReadOnlyContentStorageImpl::Register(PlaceHolderId placeHolderId, ContentId contentId) NN_NOEXCEPT
    {
        NN_UNUSED(placeHolderId);
        NN_UNUSED(contentId);

        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    nn::Result ReadOnlyContentStorageImpl::RevertToPlaceHolder(PlaceHolderId placeHolderId, ContentId contentId, ContentId postContentId) NN_NOEXCEPT
    {
        NN_UNUSED(placeHolderId);
        NN_UNUSED(contentId);
        NN_UNUSED(postContentId);

        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    nn::Result ReadOnlyContentStorageImpl::Delete(ContentId id) NN_NOEXCEPT
    {
        NN_UNUSED(id);

        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    Result ReadOnlyContentStorageImpl::Has(sf::Out<bool> outValue, ContentId id) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_MakeContentPathFunction);
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        PathString rawPath;
        MakeContentPath(&rawPath, id, m_MakeContentPathFunction, m_RootPath);

        bool hasFile;
        NN_RESULT_DO(detail::HasFile(&hasFile, rawPath));
        if (!hasFile)
        {
            MakeGameCardContentMetaPath(&rawPath, id, m_MakeContentPathFunction, m_RootPath);
            NN_RESULT_DO(detail::HasFile(&hasFile, rawPath));
        }
        outValue.Set(hasFile);

        NN_RESULT_SUCCESS;
    }

    Result ReadOnlyContentStorageImpl::GetPath(sf::Out<Path> outValue, ContentId id) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_MakeContentPathFunction);
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        // 最終的な出力を通常の ContentPath の方にするため、ContentMetaPath の方で存在確認をする
        PathString rawPath;
        MakeGameCardContentMetaPath(&rawPath, id, m_MakeContentPathFunction, m_RootPath);

        bool hasFile;
        NN_RESULT_DO(detail::HasFile(&hasFile, rawPath));
        if (!hasFile)
        {
            MakeContentPath(&rawPath, id, m_MakeContentPathFunction, m_RootPath);
        }

        Path commonPath;
        NN_RESULT_DO(fs::ConvertToFsCommonPath(commonPath.string, sizeof(commonPath.string), rawPath));

        outValue.Set(commonPath);

        NN_RESULT_SUCCESS;
    }
    Result ReadOnlyContentStorageImpl::GetFreeSpaceSize(sf::Out<std::int64_t> outValue) const NN_NOEXCEPT
    {
        outValue.Set(0);
        NN_RESULT_SUCCESS;
    }

    Result ReadOnlyContentStorageImpl::GetTotalSpaceSize(sf::Out<std::int64_t> outValue) const NN_NOEXCEPT
    {
        outValue.Set(0);
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadOnlyContentStorageImpl::GetPlaceHolderPath(sf::Out<Path> outValue, PlaceHolderId id) const NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_UNUSED(id);

        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    nn::Result ReadOnlyContentStorageImpl::CleanupAllPlaceHolder() NN_NOEXCEPT
    {
        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    nn::Result ReadOnlyContentStorageImpl::ListPlaceHolder(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::ncm::PlaceHolderId>& outList) const NN_NOEXCEPT
    {
        NN_UNUSED(outCount);
        NN_UNUSED(outList);

        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    nn::Result ReadOnlyContentStorageImpl::GetContentCount(nn::sf::Out<std::int32_t> outCount) const NN_NOEXCEPT
    {
        NN_UNUSED(outCount);
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    nn::Result ReadOnlyContentStorageImpl::ListContentId(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::ncm::ContentId>& outList, std::int32_t offset) const NN_NOEXCEPT
    {
        NN_UNUSED(outCount);
        NN_UNUSED(outList);
        NN_UNUSED(offset);
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    nn::Result ReadOnlyContentStorageImpl::GetSizeFromContentId(nn::sf::Out<std::int64_t> outValue, ContentId id) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        fs::FileHandle handle;
        NN_RESULT_DO(OpenContentIdFileImpl(&handle, id, m_MakeContentPathFunction, m_RootPath));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(handle); };

        int64_t fileSize;
        NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));
        outValue.Set(fileSize);

        NN_RESULT_SUCCESS;
    }

    nn::Result ReadOnlyContentStorageImpl::GetSizeFromPlaceHolderId(nn::sf::Out<std::int64_t> outValue, PlaceHolderId id) const NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_UNUSED(id);

        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    Result ReadOnlyContentStorageImpl::DisableForcibly() NN_NOEXCEPT
    {
        m_IsDisabled = true;
        NN_RESULT_SUCCESS;
    }

    Result ReadOnlyContentStorageImpl::ReadContentIdFile(sf::OutBuffer buffer, ContentId id, int64_t offset) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        fs::FileHandle handle;
        NN_RESULT_DO(OpenContentIdFileImpl(&handle, id, m_MakeContentPathFunction, m_RootPath));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(handle); };

        NN_RESULT_DO(fs::ReadFile(handle, offset, buffer.GetPointerUnsafe(), buffer.GetSize()));

        NN_RESULT_SUCCESS;
    }

    Result ReadOnlyContentStorageImpl::GetRightsIdFromPlaceHolderId(sf::Out<RightsId> outValue, PlaceHolderId id) NN_NOEXCEPT
    {
        NN_UNUSED(outValue);
        NN_UNUSED(id);

        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    Result ReadOnlyContentStorageImpl::GetRightsIdFromContentId(sf::Out<RightsId> outValue, ContentId id) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        Path path;
        NN_RESULT_DO(this->GetPath(&path, id));

        fs::RightsId rightsId;
        uint8_t keyGeneration;
        NN_RESULT_DO(nn::fs::GetRightsId(&rightsId, &keyGeneration, path.string));

        *outValue = { rightsId, keyGeneration };
        NN_RESULT_SUCCESS;
    }

    Result ReadOnlyContentStorageImpl::WriteContentForDebug(ContentId id, int64_t offset, sf::InBuffer buffer) NN_NOEXCEPT
    {
        NN_UNUSED(id);
        NN_UNUSED(offset);
        NN_UNUSED(buffer);

        //読み込み専用のため、利用不可
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    Result ReadOnlyContentStorageImpl::FlushPlaceHolder() NN_NOEXCEPT
    {
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

    Result ReadOnlyContentStorageImpl::RepairInvalidFileAttribute() NN_NOEXCEPT
    {
        NN_RESULT_THROW(nn::ncm::ResultWriteToReadOnlyContentStorage());
    }

}}
