﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/ncm/ncm_PackageSystemUpdateTask.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_MaxCount.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/detail/ncm_Log.h>

namespace nn { namespace ncm {

PackageSystemUpdateTask::~PackageSystemUpdateTask() NN_NOEXCEPT
{
    if (m_ContextPath.GetLength() > 0)
    {
        fs::DeleteFile(m_ContextPath);
    }
    // Result は無視
    InactivateContentMetaDatabase(ncm::StorageId::Card);
}

Result PackageSystemUpdateTask::Initialize(const char* packageRoot, const char* contextPath, void* buffer, size_t bufferSize, bool requiresExFatDriver) NN_NOEXCEPT
{
    bool isSuccess = false;
    NN_RESULT_DO(ActivateContentMetaDatabase(StorageId::Card));
    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            // Result は無視
            InactivateContentMetaDatabase(ncm::StorageId::Card);
        }
    };

    NN_RESULT_DO(OpenContentMetaDatabase(&m_PackageDb, StorageId::Card));
    ContentMetaDatabaseBuilder builder(&m_PackageDb);
    NN_RESULT_DO(builder.Cleanup());
    NN_RESULT_DO(builder.BuildFromPackage(packageRoot));

    fs::DeleteFile(contextPath);
    NN_RESULT_DO(ncm::FileInstallTaskData::Create(contextPath, CardMaxContentMetaCount)); // TODO: SIGLO-33388 後に Max 定義を参照
    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            fs::DeleteFile(contextPath);
        }
    };

    auto config = requiresExFatDriver ? InstallConfig_SystemUpdateRecursive | InstallConfig_RequiresExFatDriver :
                                        InstallConfig_SystemUpdateRecursive;
    NN_RESULT_DO(m_Data.Initialize(contextPath));
    NN_RESULT_DO(PackageInstallTaskBase::Initialize(packageRoot, buffer, bufferSize, StorageId::BuildInSystem, &m_Data, config));
    isSuccess = true;

    m_ContextPath.Assign(contextPath);

    NN_RESULT_SUCCESS;
}

util::optional<ncm::ContentMetaKey> PackageSystemUpdateTask::GetSystemUpdateMetaKey() NN_NOEXCEPT
{
    int count;
    ncm::StorageContentMetaKey keyList[16];
    int keyListCount = sizeof(keyList) / sizeof(keyList[0]);
    int offset = 0;

    do
    {
        NN_RESULT_TRY(ListContentMetaKey(&count, keyList, keyListCount, offset))
            NN_RESULT_CATCH_ALL { return util::nullopt; }
        NN_RESULT_END_TRY

        offset += count;

        for (int i = 0; i < count; i++)
        {
            if (keyList[i].key.type == ncm::ContentMetaType::SystemUpdate)
            {
                return keyList[i].key;
            }
        }

    } while (count > 0);

    return util::nullopt;
}

Result PackageSystemUpdateTask::GetInstallContentMetaInfo(ncm::InstallContentMetaInfo* outValue, const ncm::ContentMetaKey& key) NN_NOEXCEPT
{
    ContentInfo info;
    NN_RESULT_DO(GetContentInfoOfContentMeta(&info, key));

    *outValue = InstallContentMetaInfo::MakeUnverifiable(info.GetId(), info.GetSize(), key.type);
    NN_RESULT_SUCCESS;
}

Result PackageSystemUpdateTask::PrepareInstallContentMetaData() NN_NOEXCEPT
{
    ContentMetaKey updateMetaKey;
    auto listCount = m_PackageDb.ListContentMeta(&updateMetaKey, 1, ContentMetaType::SystemUpdate);
    NN_RESULT_THROW_UNLESS(listCount.listed > 0, ResultSystemUpdateNotFoundInPackage());

    ContentInfo info;
    NN_RESULT_DO(GetContentInfoOfContentMeta(&info, updateMetaKey));

    auto metaInfo = InstallContentMetaInfo::MakeUnverifiable(info.GetId(), info.GetSize(), ContentMetaType::SystemUpdate);
    NN_RESULT_DO(PrepareContentMeta(metaInfo, updateMetaKey));

    NN_RESULT_SUCCESS;
}
Result PackageSystemUpdateTask::PrepareDependency() NN_NOEXCEPT
{
    NN_RESULT_DO(PrepareSystemUpdateDependency());
    NN_RESULT_SUCCESS;
}

Result PackageSystemUpdateTask::GetContentInfoOfContentMeta(ContentInfo* outValue, const ContentMetaKey& key) NN_NOEXCEPT
{
    for (int i = 0;; i++)
    {
        int count;
        ContentInfo info;
        NN_RESULT_DO(m_PackageDb.ListContentInfo(&count, &info, 1, key, i));
        if (count == 0)
        {
            break;;
        }

        if (info.GetType() == ContentType::Meta)
        {
            *outValue = info;
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultContentInfoNotFound());
}

}}
