﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>

#include <nn/ncm/ncm_PackageInstallTask.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_Result.h>

#include "ncm_FileSystemUtility.h"

namespace nn { namespace ncm {

bool PackageInstallTask::IsContentMetaContentName(const char* name) NN_NOEXCEPT
{
    return detail::PathView(name).HasSuffix(".cnmt.nca");
}

void PackageInstallTask::Initialize(const char* directoryPath, StorageId storage, void* buffer, size_t bufferSize, bool ignoreTicket) NN_NOEXCEPT
{
    PackageInstallTaskBase::Initialize(directoryPath, buffer, bufferSize, storage, &m_Data, ignoreTicket ? InstallConfig_IgnoreTicket : 0);
}

Result PackageInstallTask::GetInstallContentMetaInfo(ncm::InstallContentMetaInfo* outValue, const ncm::ContentMetaKey& key) NN_NOEXCEPT
{
    NN_UNUSED(outValue);
    NN_UNUSED(key);

    NN_RESULT_THROW(ResultContentNotFound());
}

Result PackageInstallTask::PrepareInstallContentMetaData() NN_NOEXCEPT
{
    fs::DirectoryHandle directory;
    NN_RESULT_DO(fs::OpenDirectory(&directory, GetPackageRootPath(), fs::OpenDirectoryMode_File));
    NN_UTIL_SCOPE_EXIT { fs::CloseDirectory(directory); };

    while (NN_STATIC_CONDITION(true))
    {
        fs::DirectoryEntry entry;
        int64_t outRead;
        NN_RESULT_DO(fs::ReadDirectory(&outRead, &entry, directory, 1));
        if (outRead == 0)
        {
            break;
        }

        if (IsContentMetaContentName(entry.name))
        {
            auto contentId = GetContentIdFromString(entry.name, util::Strnlen(entry.name, sizeof(entry.name)));
            NN_RESULT_THROW_UNLESS(contentId, ResultInvalidPackageFormat());

            NN_RESULT_DO(PrepareContentMeta(InstallContentMetaInfo::MakeUnverifiable(*contentId, entry.fileSize, ContentMetaType::Unknown), util::nullopt));
        }
    }

    NN_RESULT_SUCCESS;
}

}}
