﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/ncm/ncm_HostPackageInstallTask.h>

#include <nn/ncm/ncm_PackageInstallTask.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_ContentMeta.h>

#include "ncm_FileSystemUtility.h"

namespace nn { namespace ncm {
    Result HostPackageInstallTask::FindContentMeta(ContentId contentId, fs::DirectoryEntry* entry) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(entry);

        fs::FileHandle file;
        char contentIdString[ContentIdStringLength + 1] = {};
        GetStringFromContentId(contentIdString, sizeof(contentIdString), contentId);

        nn::kvdb::BoundedString<256> path(GetPackageRootPath());
        path.Append(contentIdString);
        path.Append(".cnmt");

        //prod の場合は、contentId + .cnmt ファイルが存在する
        if (fs::OpenFile(&file, path.Get(), fs::OpenMode_Read).IsSuccess())
        {
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

            int64_t fileSize;
            NN_RESULT_DO(fs::GetFileSize(&fileSize, file));
            size_t metaSize = static_cast<size_t>(fileSize);

            AutoBuffer meta;
            NN_RESULT_DO(meta.Initialize(metaSize));

            size_t readSize;
            NN_RESULT_DO(fs::ReadFile(&readSize, file, 0, meta.Get(), metaSize));
            NN_RESULT_DO(PrepareContentMeta(contentId, (*entry).fileSize, ContentMetaType::Unknown, &meta));
        }
        //存在しない場合は通常の処理
        else
        {
            NN_RESULT_DO(PrepareContentMeta(InstallContentMetaInfo::MakeUnverifiable(contentId, entry->fileSize, ContentMetaType::Unknown), util::nullopt));
        }

        NN_RESULT_SUCCESS;
    }

    Result HostPackageInstallTask::PrepareInstallContentMetaData() NN_NOEXCEPT
    {
        fs::DirectoryHandle directory;
        NN_RESULT_DO(fs::OpenDirectory(&directory, GetPackageRootPath(), fs::OpenDirectoryMode_File));
        NN_UTIL_SCOPE_EXIT{ fs::CloseDirectory(directory); };

        while (NN_STATIC_CONDITION(true))
        {
            fs::DirectoryEntry entry;
            int64_t outRead;
            NN_RESULT_DO(fs::ReadDirectory(&outRead, &entry, directory, 1));
            if (outRead == 0)
            {
                break;
            }

            if (IsContentMetaContentName(entry.name))
            {
                auto contentId = GetContentIdFromString(entry.name, util::Strnlen(entry.name, sizeof(entry.name)));
                NN_RESULT_THROW_UNLESS(contentId, ResultInvalidPackageFormat());

                NN_RESULT_DO(FindContentMeta(*contentId, &entry));
            }
        }

        NN_RESULT_SUCCESS;
    }

}}
