﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <atomic>
#include <cstring>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/util/util_StringView.h>
#include <nn/util/util_FormatString.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/os/os_Mutex.h>
#include <mutex>
#include "ncm_FileSystemUtility.h"
#include <nn/ncm/ncm_AutoBuffer.h>

namespace nn { namespace ncm { namespace detail {

    namespace{

        Result EnsureDirectory(const char* path) NN_NOEXCEPT
        {
            NN_RESULT_TRY(fs::CreateDirectory(path))
                NN_RESULT_CATCH(fs::ResultPathAlreadyExists){}
            NN_RESULT_END_TRY

            NN_RESULT_SUCCESS;
        }

        Result EnsureDirectoryRecursivelyImpl(const char* path, bool createTail) NN_NOEXCEPT
        {
            const size_t length = std::strlen(path);
            char pathBuffer[768];

            if (length + 1 > sizeof(pathBuffer))
            {
                NN_RESULT_THROW(ResultAllocationMemoryFailed());
            }

            std::strncpy(pathBuffer, path, sizeof(pathBuffer));

            for (size_t i = 0; i < length; i++)
            {
                if (i > 0 && pathBuffer[i] == '/' && pathBuffer[i - 1] != ':')
                {
                    pathBuffer[i] = '\0';
                    NN_RESULT_DO(EnsureDirectory(pathBuffer));
                    pathBuffer[i] = '/';
                }
            }

            if (createTail)
            {
                NN_RESULT_DO(EnsureDirectory(path));
            }

            NN_RESULT_SUCCESS;
        }

        Result HasEntry(bool* outValue, const char* path, fs::DirectoryEntryType expectedEntryType)
        {
            fs::DirectoryEntryType entryType;
            NN_RESULT_TRY(fs::GetEntryType(&entryType, path))
                NN_RESULT_CATCH(fs::ResultPathNotFound)
                {
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY

            *outValue = entryType == expectedEntryType;

            NN_RESULT_SUCCESS;
        }
    }

    Result EnsureDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        return EnsureDirectoryRecursivelyImpl(path, true);
    }

    Result EnsureParentDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        return EnsureDirectoryRecursivelyImpl(path, false);
    }

    Result HasDirectory(bool* outValue, const char* path) NN_NOEXCEPT
    {
        return HasEntry(outValue, path, fs::DirectoryEntryType_Directory);
    }

    Result HasFile(bool* outValue, const char* path) NN_NOEXCEPT
    {
        return HasEntry(outValue, path, fs::DirectoryEntryType_File);
    }

    bool PathView::HasSuffix(util::string_view suffix) const NN_NOEXCEPT
    {
        return m_Path.substr(m_Path.size() - suffix.size()) == suffix;
    }

    bool PathView::HasPrefix(util::string_view prefix) const NN_NOEXCEPT
    {
        return GetFileName().substr(0, prefix.size()) == prefix;
    }

    util::string_view PathView::GetFileName() const NN_NOEXCEPT
    {
        auto delimiterPos = m_Path.find_last_of("/");
        return delimiterPos != util::string_view::npos ? m_Path.substr(delimiterPos + 1) : m_Path;
    }

    MountName CreateUniqueMountName() NN_NOEXCEPT
    {
        static std::atomic<uint32_t> g_Count(0);
        MountName name = {};
        util::SNPrintf(name.string, sizeof(name.string), "@ncm%08x", g_Count.fetch_add(1));

        return name;
    }

    RootDirectoryPath GetRootDirectoryPath(const MountName& mountName) NN_NOEXCEPT
    {
        RootDirectoryPath path = {};
        util::SNPrintf(path.string, sizeof(path.string), "%s:/", mountName.string);

        return path;
    }

    Result CopyFile(const char* destinationPath, const char* sourcePath) NN_NOEXCEPT
    {
        nn::fs::FileHandle sourceHandle;
        NN_RESULT_DO(
            nn::fs::OpenFile(&sourceHandle, sourcePath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(sourceHandle);
        };

        int64_t fileSize;
        NN_RESULT_DO(
            nn::fs::GetFileSize(&fileSize, sourceHandle));

        NN_RESULT_DO(
            nn::fs::CreateFile(destinationPath, fileSize));

        nn::fs::FileHandle destinationHandle;
        NN_RESULT_DO(
            nn::fs::OpenFile(&destinationHandle, destinationPath, nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(destinationHandle);
        };

        const int BufferSize = 4 * 1024;
        AutoBuffer buffer;
        NN_RESULT_DO(buffer.Initialize(BufferSize));

        int64_t readOffset = 0;
        while (readOffset < fileSize)
        {
            size_t readSize = std::min(static_cast<size_t>(fileSize - readOffset), buffer.GetSize());
            NN_RESULT_DO(nn::fs::ReadFile(sourceHandle, readOffset, buffer.Get(), readSize));
            NN_RESULT_DO(nn::fs::WriteFile(destinationHandle, readOffset, buffer.Get(), readSize, nn::fs::WriteOption()));
            readOffset += readSize;
        }

        NN_RESULT_DO(
            nn::fs::FlushFile(destinationHandle));

        NN_RESULT_SUCCESS;
    }
}}}
