﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Content.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaExtendedData.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include "ncm_FileSystemUtility.h"

namespace nn { namespace ncm {
    namespace {
        typedef kvdb::BoundedString<64> BoundedPath;

        bool IsContentMetaName(const char* name) NN_NOEXCEPT
        {
            return detail::PathView(name).HasSuffix(".cnmt");
        }
    }

    Result ReadContentMetaPath(AutoBuffer* outValue, const char* path) NN_NOEXCEPT
    {
        auto mountName = detail::CreateUniqueMountName();
        NN_RESULT_DO(fs::MountContent(mountName.string, path, fs::ContentType_Meta));
        NN_UTIL_SCOPE_EXIT{ fs::Unmount(mountName.string); };

        auto rootDirectoryPath = detail::GetRootDirectoryPath(mountName);
        fs::DirectoryHandle directory;
        NN_RESULT_DO(fs::OpenDirectory(&directory, rootDirectoryPath.string, fs::OpenDirectoryMode_File));
        NN_UTIL_SCOPE_EXIT{ fs::CloseDirectory(directory); };

        while (NN_STATIC_CONDITION(true))
        {
            fs::DirectoryEntry entry;
            int64_t outRead;
            NN_RESULT_DO(fs::ReadDirectory(&outRead, &entry, directory, 1));
            if (outRead == 0)
            {
                break;
            }

            if (IsContentMetaName(entry.name))
            {
                BoundedPath filePath(rootDirectoryPath.string);
                filePath.Append(entry.name);

                fs::FileHandle file;
                NN_RESULT_DO(fs::OpenFile(&file, filePath, fs::OpenMode_Read));
                NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

                int64_t fileSize;
                NN_RESULT_DO(fs::GetFileSize(&fileSize, file));
                size_t metaSize = static_cast<size_t>(fileSize);

                AutoBuffer meta;
                NN_RESULT_DO(meta.Initialize(metaSize));

                size_t readSize;
                NN_RESULT_DO(fs::ReadFile(&readSize, file, 0, meta.Get(), metaSize));

                *outValue = std::move(meta);
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_THROW(ResultContentMetaNotFound());
    }
    Result GetContentMetaPath(Path* outPath, const StorageContentMetaKey storageKey) NN_NOEXCEPT
    {
        ContentMetaDatabase db;
        ContentStorage storage;
        NN_RESULT_DO(OpenContentMetaDatabase(&db, storageKey.storageId));
        NN_RESULT_DO(OpenContentStorage(&storage, storageKey.storageId));

        int numContent;
        ContentInfo contentInfo;
        for(int i = 0; i < 16; ++i)
        {
            NN_RESULT_DO(db.ListContentInfo(&numContent, &contentInfo, 1, storageKey.key, i));
            if(numContent == 0)
            {
                break;
            }
            if(contentInfo.type == ContentType::Meta)
            {
                storage.GetPath(outPath, contentInfo.id);
                NN_RESULT_SUCCESS;
            }
        }
        NN_RESULT_THROW(ResultContentMetaNotFound());
    }
    Result VerifyPatchHistory(bool* outFound, ContentMetaDatabase* db, const ContentMetaKey& key, const PackagedContentMetaReader& reader) NN_NOEXCEPT
    {
        *outFound = false;
        // reader 内の history 上に、key とそこに紐づくコンテンツが存在しているか確認
        auto exDataSize = reader.GetExtendedDataSize();
        if (exDataSize == 0)
        {
            // 見つからなかった
            NN_RESULT_SUCCESS;
        }
        PatchMetaExtendedDataReader exReader(reader.GetExtendedData(), exDataSize);

        auto header = exReader.GetHeader();
        for (int i = 0; i < static_cast<int>(header->historyCount); ++i)
        {
            auto historyHeader = exReader.GetPatchHistoryHeader(i);
            if (key == historyHeader->key)
            {
                // コンテンツが一致しているか確認
                std::unique_ptr<ContentInfo[]> contentInfos(new ContentInfo[historyHeader->contentCount]);
                int installContentCount;
                NN_RESULT_DO(db->ListContentInfo(&installContentCount, contentInfos.get(), historyHeader->contentCount, key, 0));
                for (int j = 0; j < historyHeader->contentCount; ++j)
                {
                    auto historyContentInfo = exReader.GetPatchHistoryContentInfo(i, j);

                    // 三重ループだが、contentCount は現時点で高々 5 とかなので、許容とする
                    // やりたいことは下記。配列をイテレータに変換する妥当な方法がみつからずあきらめている
                    // if (contentInfos.Any(p => p.id == historyContentInfo->id) == false)
                    // {
                    //      *outFound = false; NN_RESULT_SUCCESS;
                    // }
                    bool found = false;
                    for (int k = 0; k < installContentCount; ++k)
                    {
                        if (historyContentInfo->id == contentInfos.get()[k].id)
                        {
                            found = true;
                            break;
                        }
                    }
                    if (found == false)
                    {
                        // 対応するものが見つからなかったので、*outFound = false のまま返す
                        NN_RESULT_SUCCESS;
                    }
                }
                // ここに来るということは、一致するものがすべて見つかったということ
                *outFound = true;
                NN_RESULT_SUCCESS;
            }
        }
        // history に一致する key が見つからない場合も、見つからないで成功扱い
        NN_RESULT_SUCCESS;
    }
}}
