﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <utility>

#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_AutoBuffer.h>

namespace nn { namespace ncm {

    namespace {

        class Accessor
        {
            NN_DISALLOW_COPY(Accessor);
            NN_DISALLOW_MOVE(Accessor);

        public:
            explicit Accessor(ContentMetaDatabaseShared* db) NN_NOEXCEPT : m_Db(db){}

            template<typename Id, typename Meta>
            Result Set(Id id, const Meta& meta) NN_NOEXCEPT
            {
                return (*m_Db)->Set(ContentMetaKey::Make(id, meta.version), sf::InBuffer(reinterpret_cast<const char*>(&meta), sizeof(meta)));
            }

            template<typename Id, typename Meta>
            Result Get(Meta* outValue, Id id, uint32_t version) NN_NOEXCEPT
            {
                uint64_t outSize;
                NN_RESULT_DO((*m_Db)->Get(&outSize, ContentMetaKey::Make(id, version), sf::OutBuffer(reinterpret_cast<char*>(outValue), sizeof(Meta))));
                NN_RESULT_THROW_UNLESS(outSize == sizeof(Meta), ResultInvalidKeyValueFormat());

                NN_RESULT_SUCCESS;
            }

            template<typename Id>
            Result Remove(Id id, uint32_t version) NN_NOEXCEPT
            {
                return (*m_Db)->Remove(ContentMetaKey::Make(id, version));
            }

        private:
            ContentMetaDatabaseShared* m_Db;
        };

    }

    const ApplicationId ContentMetaDatabase::AnyApplicationId = {0};

    ContentMetaDatabase::ContentMetaDatabase() NN_NOEXCEPT{}

    ContentMetaDatabase::ContentMetaDatabase(sf::SharedPointer<IContentMetaDatabase> interfac) NN_NOEXCEPT : m_Interface(interfac){}

    ContentMetaDatabase::ContentMetaDatabase(ContentMetaDatabase&& rvalue) NN_NOEXCEPT
    {
        m_Interface = std::move(rvalue.m_Interface);
    }

    ContentMetaDatabase& ContentMetaDatabase::operator=(ContentMetaDatabase&& rvalue) NN_NOEXCEPT
    {
        ContentMetaDatabase(std::move(rvalue)).swap(*this);

        return *this;
    }

    void ContentMetaDatabase::swap(ContentMetaDatabase& other) NN_NOEXCEPT
    {
        std::swap(m_Interface, other.m_Interface);
    }

    Result ContentMetaDatabase::Set(const ContentMetaKey& key, const void* buffer, size_t size) NN_NOEXCEPT
    {
        return m_Interface->Set(key, sf::InBuffer(reinterpret_cast<const char*>(buffer), size));
    }

    Result ContentMetaDatabase::Remove(SystemProgramId id, uint32_t version) NN_NOEXCEPT
    {
        return Accessor(&m_Interface).Remove(id, version);
    }

    Result ContentMetaDatabase::Remove(SystemDataId id, uint32_t version) NN_NOEXCEPT
    {
        return Accessor(&m_Interface).Remove(id, version);
    }

    Result ContentMetaDatabase::Remove(ApplicationId id, uint32_t version) NN_NOEXCEPT
    {
        return Accessor(&m_Interface).Remove(id, version);
    }

    Result ContentMetaDatabase::GetProgram(ContentId* outValue, ProgramId id, uint32_t version) const NN_NOEXCEPT
    {
        return m_Interface->GetContentIdByType(outValue, ContentMetaKey::MakeUnknownType(id.value, version), ContentType::Program);
    }

    Result ContentMetaDatabase::GetLatestProgram(ContentId* outValue, ProgramId id) const NN_NOEXCEPT
    {
        ContentMetaKey foundKey;
        NN_RESULT_DO(m_Interface->GetLatestContentMetaKey(&foundKey, id.value));
        return m_Interface->GetContentIdByType(outValue, foundKey, ContentType::Program);
    }

    Result ContentMetaDatabase::GetLatestData(ContentId* outValue, DataId id) const NN_NOEXCEPT
    {
        ContentMetaKey foundKey;
        NN_RESULT_DO(m_Interface->GetLatestContentMetaKey(&foundKey, id.value));
        return m_Interface->GetContentIdByType(outValue, foundKey, ContentType::Data);
    }

    Result ContentMetaDatabase::GetControl(ContentId* outValue, ApplicationId id, uint32_t version) const NN_NOEXCEPT
    {
        return m_Interface->GetContentIdByType(outValue, ContentMetaKey::MakeUnknownType(id.value, version), ContentType::Control);
    }

    Result ContentMetaDatabase::GetLatestControl(ContentId* outValue, ApplicationId id) const NN_NOEXCEPT
    {
        ContentMetaKey foundKey;
        NN_RESULT_DO(m_Interface->GetLatestContentMetaKey(&foundKey, id.value));
        return m_Interface->GetContentIdByType(outValue, foundKey, ContentType::Control);
    }

    Result ContentMetaDatabase::GetHtmlDocument(ContentId* outValue, ApplicationId id, uint32_t version) const NN_NOEXCEPT
    {
        return m_Interface->GetContentIdByType(outValue, ContentMetaKey::MakeUnknownType(id.value, version), ContentType::HtmlDocument);
    }

    Result ContentMetaDatabase::GetLatestHtmlDocument(ContentId* outValue, ApplicationId id) const NN_NOEXCEPT
    {
        ContentMetaKey foundKey;
        NN_RESULT_DO(m_Interface->GetLatestContentMetaKey(&foundKey, id.value));
        return m_Interface->GetContentIdByType(outValue, foundKey, ContentType::HtmlDocument);
    }

    Result ContentMetaDatabase::GetLegalInformation(ContentId* outValue, ApplicationId id, uint32_t version) const NN_NOEXCEPT
    {
        return m_Interface->GetContentIdByType(outValue, ContentMetaKey::MakeUnknownType(id.value, version), ContentType::LegalInformation);
    }

    Result ContentMetaDatabase::GetLatestLegalInformation(ContentId* outValue, ApplicationId id) const NN_NOEXCEPT
    {
        ContentMetaKey foundKey;
        NN_RESULT_DO(m_Interface->GetLatestContentMetaKey(&foundKey, id.value));
        return m_Interface->GetContentIdByType(outValue, foundKey, ContentType::LegalInformation);
    }

    Result ContentMetaDatabase::GetContentIdByType(ContentId* outValue, const ContentMetaKey& key, ContentType type) const NN_NOEXCEPT
    {
        return m_Interface->GetContentIdByType(outValue, key, type);
    }

    Result ContentMetaDatabase::GetContentIdByTypeAndIdOffset(ContentId* outValue, const ContentMetaKey& key, ContentType type, uint8_t idOffset) const NN_NOEXCEPT
    {
        return m_Interface->GetContentIdByTypeAndIdOffset(outValue, key, type, idOffset);
    }

    ListCount ContentMetaDatabase::ListApplication(ApplicationContentMetaKey outValue[], int count) const NN_NOEXCEPT
    {
        ListCount listCount = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->ListApplication(&listCount.total, &listCount.listed, sf::OutArray<ApplicationContentMetaKey>(outValue, count), ContentMetaType::Unknown));

        return listCount;
    }

    ListCount ContentMetaDatabase::ListContentMeta(ContentMetaKey outValue[], int count, ContentMetaType type, ApplicationId appId, Bit64 min, Bit64 max, ContentInstallType installType) const NN_NOEXCEPT
    {
        ListCount listCount = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface->List(&listCount.total, &listCount.listed, sf::OutArray<ContentMetaKey>(outValue, count), type, appId, min, max, installType));

        return listCount;
    }

    Result ContentMetaDatabase::GetLatest(ContentMetaKey* outValue, Bit64 id) const NN_NOEXCEPT
    {
        return m_Interface->GetLatestContentMetaKey(outValue, id);
    }

    Result ContentMetaDatabase::Remove(const ContentMetaKey key) NN_NOEXCEPT
    {
        return m_Interface->Remove(key);
    }

    Result ContentMetaDatabase::ListContentInfo(int* outCount, ContentInfo outList[], int count, const ContentMetaKey& key, int offset) const NN_NOEXCEPT
    {
        return m_Interface->ListContentInfo(outCount, sf::OutArray<ContentInfo>(outList, count), key, offset);
    }

    Result ContentMetaDatabase::ListContentMetaInfo(int* outCount, ContentMetaInfo outList[], int count, const ContentMetaKey& key, int offset) const NN_NOEXCEPT
    {
        return m_Interface->ListContentMetaInfo(outCount, sf::OutArray<ContentMetaInfo>(outList, count), key, offset);
    }

    Result ContentMetaDatabase::Has(bool* outValue, const ContentMetaKey& key) const NN_NOEXCEPT
    {
        return m_Interface->Has(outValue, key);
    }

    Result ContentMetaDatabase::HasAll(bool* outValue, const ContentMetaKey list[], int count) const NN_NOEXCEPT
    {
        return m_Interface->HasAll(outValue, sf::InArray<ContentMetaKey>(list, count));
    }

    Result ContentMetaDatabase::HasContent(bool* outValue, const ContentMetaKey& key, const ContentId& contentId) const NN_NOEXCEPT
    {
        return m_Interface->HasContent(outValue, key, contentId);
    }

    Result ContentMetaDatabase::Get(size_t* outSize, void* outValue, size_t bufferSize, const ContentMetaKey& contentMetaKey) const NN_NOEXCEPT
    {
        sf::OutBuffer sfOutBuffer(reinterpret_cast<char*>(outValue), bufferSize);
        uint64_t size;
        NN_RESULT_DO(m_Interface->Get(&size, contentMetaKey, sfOutBuffer));

        *outSize = static_cast<size_t>(size);
        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabase::GetSize(size_t* outSize, const ContentMetaKey& key) const NN_NOEXCEPT
    {
        uint64_t size;
        NN_RESULT_DO(m_Interface->GetSize(&size, key));

        *outSize = static_cast<size_t>(size);
        NN_RESULT_SUCCESS;
    }

    Result ContentMetaDatabase::GetRequiredSystemVersion(uint32_t* out, const ContentMetaKey& key) const NN_NOEXCEPT
    {
        return m_Interface->GetRequiredSystemVersion(out, key);
    }

    Result ContentMetaDatabase::GetPatchId(PatchId* out, const ContentMetaKey& key) const NN_NOEXCEPT
    {
        return m_Interface->GetPatchId(out, key);
    }

    Result ContentMetaDatabase::GetAttributes(Bit8* outAttributes, const ContentMetaKey& key) const NN_NOEXCEPT
    {
        return m_Interface->GetAttributes(outAttributes, key);
    }

    Result ContentMetaDatabase::LookupOrphanContent(bool* outList, const ContentId contentList[], int count) const NN_NOEXCEPT
    {
        return m_Interface->LookupOrphanContent(sf::OutArray<bool>(outList, static_cast<size_t>(count)), sf::InArray<ContentId>(contentList, static_cast<size_t>(count)));
    }

    Result ContentMetaDatabase::Commit() NN_NOEXCEPT
    {
        return m_Interface->Commit();
    }

    Result ContentMetaDatabase::DisableForcibly() NN_NOEXCEPT
    {
        return m_Interface->DisableForcibly();
    }

    Result ContentMetaDatabase::GetRequiredApplicationVersion(uint32_t* outValue, const ContentMetaKey& key) const NN_NOEXCEPT
    {
        return m_Interface->GetRequiredApplicationVersion(outValue, key);
    }
}}
