﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <cstddef>

#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nn/os/os_SdkThreadLocalStorage.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/mem/detail/mem_Log.h>
#include <nn/nlib/heap/TlsHeapCache.h>
#include <nn/nlib/heap/TlsHeapCentral.h>
#include <nn/nlib/heap/TlsHeapStatic.h>
#include <nn/nlib/heap/CentralHeap.h>
#include <nn/nlib/heap/CachedHeap.h>

// Hash 計算用
#include <nn/crypto/crypto_Sha1Generator.h>

namespace nn { namespace mem {

    // デフォルトアライメントは max_align_t にあわせる
    const size_t DefaultAlignment = NN_ALIGNOF(std::max_align_t);

    // アロケータに最小限必要なサイズ（管理領域のサイズ）
    const size_t MinimumAllocatorSize = 16 * 1024;

namespace {

    // TLS デストラクタ関数
    // 外に見せるなら無名名前空間から出す
    static void ThreadDestroy(uintptr_t value)
    {
        if (value)
        {
            nn::nlibsdk::heap::TlsHeapCache* p = reinterpret_cast<nn::nlibsdk::heap::TlsHeapCache*>(value);
            p->Finalize();
        }
    }

    // CentralHeap* にキャスト
    static inline nn::nlibsdk::heap::CentralHeap* GetCentral ( const detail::InternalCentralHeapStorage* ptr )
    {
        return reinterpret_cast<nn::nlibsdk::heap::CentralHeap*>(const_cast<detail::InternalCentralHeapStorage*>(ptr));
    }

    // スレッドキャッシュを取得する。なければ作成する。
    static inline void GetCache( nn::nlibsdk::heap::CentralHeap* pCentral, nn::os::TlsSlot slot )
    {
        nn::nlibsdk::heap::CachedHeap tmp;

        if (pCentral->MakeCache(&tmp))
        {
            nn::nlibsdk::heap::TlsHeapCache* cache = tmp.Release();
            nn::os::SetTlsValue(slot, reinterpret_cast<uintptr_t>(cache));
        }
    }

    struct InternalHash
    {
        size_t allocCount;
        size_t allocSize;
        nn::crypto::Sha1Generator sha1;
    };

    // ハッシュ値計算用 Walk callback
    static int CalcSha1Callback( void *addr, size_t size, void *userPtr )
    {
        auto hash = reinterpret_cast<InternalHash*>(userPtr);
        hash->sha1.Update(reinterpret_cast<void*>(&addr), sizeof(void*));
        hash->sha1.Update(reinterpret_cast<void*>(&size), sizeof(size));
        hash->allocCount++;
        hash->allocSize += size;
        return 1;
    }
}

StandardAllocator::StandardAllocator() NN_NOEXCEPT : m_Initialized(false), m_EnableThreadCache(false), m_CentralAllocatorAddr(0)
{
    new(&m_CentralHeap) nn::nlibsdk::heap::CentralHeap;
}

StandardAllocator::StandardAllocator(void* addr, size_t size) NN_NOEXCEPT : m_Initialized(false), m_EnableThreadCache(false), m_CentralAllocatorAddr(0)
{
    new(&m_CentralHeap) nn::nlibsdk::heap::CentralHeap;
    Initialize(addr, size);
}

StandardAllocator::StandardAllocator(void* addr, size_t size, bool isCacheEnable) NN_NOEXCEPT : m_Initialized(false), m_EnableThreadCache(false), m_CentralAllocatorAddr(0)
{
    new(&m_CentralHeap) nn::nlibsdk::heap::CentralHeap;
    Initialize(addr, size, isCacheEnable);
}

void StandardAllocator::Initialize( void* addr, size_t size, bool isCacheEnable ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(!m_Initialized);

    uintptr_t alignedStart = reinterpret_cast<uintptr_t>(addr);
    uintptr_t alignedEnd = reinterpret_cast<uintptr_t>(addr) + size;
    alignedStart = (alignedStart + nn::nlibsdk::heap::TlsHeapStatic::kPageSize - 1) & ~(nn::nlibsdk::heap::TlsHeapStatic::kPageSize - 1);
    alignedEnd = alignedEnd & ~(nn::nlibsdk::heap::TlsHeapStatic::kPageSize - 1);
    size_t alignedSize = alignedEnd - alignedStart;

    if (addr == nullptr)
    {
        //
        // Virtual Address Memory を利用する
        //
#if defined(NN_OS_CPU_ARM_AARCH64_ARMV8A) || defined(NN_BUILD_CONFIG_CPU_X64)
        // 仮想アドレスメモリが有効な場合のみ利用可能
        NN_ABORT_UNLESS(nn::os::IsVirtualAddressMemoryEnabled() == true);

        errno_t e = GetCentral(&m_CentralHeap)->Init(nullptr, size, 0);
        NN_ABORT_UNLESS(e == 0);
#else
        NN_ABORT_UNLESS(!addr, NN_TEXT("32bit 環境では Virtual Address Memory は使用できません。"));
#endif
    }
    else
    {
        //
        // Virtual Address Memory を利用しない
        //

        // size が kPageSize でメモリが kPageSize アライメントでない場合
        NN_ABORT_UNLESS(alignedStart < alignedEnd);

        // alignedSize が 16KB より小さい場合、管理領域が不足する
        NN_ABORT_UNLESS(alignedSize >= MinimumAllocatorSize);

        errno_t e = GetCentral(&m_CentralHeap)->Init(reinterpret_cast<void*>(alignedStart), alignedSize, 0);
        NN_ABORT_UNLESS(e == 0);
    }

    m_EnableThreadCache = isCacheEnable;
    if(m_EnableThreadCache)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateTlsSlot(&m_TlsSlot, ThreadDestroy));
    }

    // 初期化済みフラグ
    m_Initialized = true;
}

void StandardAllocator::Initialize(void* addr, size_t size) NN_NOEXCEPT
{
    Initialize(addr, size, false);
}

void StandardAllocator::Finalize() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Initialized);

    // TLS スロットを解放
    if (m_EnableThreadCache)
    {
        nn::os::FreeTlsSlot(m_TlsSlot);
    }

    GetCentral(&m_CentralHeap)->Finalize();

    // 初期化済みフラグを落とす
    m_Initialized = false;
}

void* StandardAllocator::Allocate( size_t size, size_t align ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);

    // TLS からスレッドキャッシュを取得
    nn::nlibsdk::heap::TlsHeapCache* p = NULL;
    if (m_EnableThreadCache)
    {
        p = reinterpret_cast<nn::nlibsdk::heap::TlsHeapCache*>(nn::os::GetTlsValue(m_TlsSlot));
    }

    void* ptr = NULL;
    if (!p && m_EnableThreadCache)
    {
        // スレッドキャッシュをまだ作成していなかった場合
        GetCache(GetCentral(&m_CentralHeap), m_TlsSlot);
        // TLS からスレッドキャッシュを再取得
        p = reinterpret_cast<nn::nlibsdk::heap::TlsHeapCache*>(nn::os::GetTlsValue(m_TlsSlot));
    }

    // スレッドキャッシュからアロケート
    if (p)
    {
        ptr = p->Alloc(size, align);
        if (ptr)
        {
            return ptr;
        }

        // 確保できなかった場合、TLS 領域を解放して再度アロケートを試みる
        nn::nlibsdk::heap::CachedHeap cache;
        cache.Reset(p);
        cache.Query(kNmallocQueryFinalizeCache);
        nn::os::SetTlsValue(m_TlsSlot, 0);
    }

    // 直接中央ヒープに取りに行く
    ptr = GetCentral(&m_CentralHeap)->Alloc(size, align);
    if (ptr)
    {
        return ptr;
    }
    else
    {
        return NULL;
    }
}

void* StandardAllocator::Allocate( size_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);
    return Allocate(size, DefaultAlignment);
}

void StandardAllocator::Free( void* addr ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);

    if (!addr)
    {
        return;
    }

    if (m_EnableThreadCache)
    {
        // TLS からスレッドキャッシュを取得
        nn::nlibsdk::heap::TlsHeapCache* p = reinterpret_cast<nn::nlibsdk::heap::TlsHeapCache*>(nn::os::GetTlsValue(m_TlsSlot));
        if (p)
        {
            p->Free(addr);
            return;
        }

        // [nlib の NMalloc.cpp より]
        // キャッシュがない場合というのは、スレッド終了時に
        // nmalloc_finalize_tls()（*1）を呼び出した後の解放であることが多い。
        // この場合に再度キャッシュを確保してしまうと、キャッシュのリークが避けられない。
        // この問題を避けるためにキャッシュを介せずダイレクトに中央ヒープに返す
        //
        // *1 nmalloc_finalize_tls() はスレッド終了時に呼ばれるスレッドキャッシュ終了関数。
        //    本実装では ThreadDestroy() がこれに相当する。
    }

    errno_t e = GetCentral(&m_CentralHeap)->Free(addr);
    if (e != 0)
    {
        NN_DETAIL_MEM_ERROR("StandardAllocator::Free failed: error=%d, addr=%p\n", e, addr);
        NN_SDK_ASSERT(e == 0);
    }
}

void* StandardAllocator::Reallocate( void* addr, size_t newSize ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);

    if(newSize > RSIZE_MAX)
    {
        return NULL;
    }
    void* p = NULL;
    // 第1引数が NULL の場合、Allocate 関数と同じ動作になる
    if (!addr)
    {
        return Allocate(newSize);
    }

    // 第2引数が 0 の場合、Free 関数と同じ動作になる
    if (newSize == 0)
    {
        Free(addr);
        return NULL;
    }

    // NOTE:
    // このヒープではサイズがy * 2^xならばアライメントが2^nになる性質がある。
    // nをalgnに適合するようにする
    size_t alignedNewSize = (newSize + DefaultAlignment - 1) & ~(DefaultAlignment - 1);

    nn::nlibsdk::heap::TlsHeapCache* cache = NULL;
    if (m_EnableThreadCache)
    {
        cache = reinterpret_cast<nn::nlibsdk::heap::TlsHeapCache*>(nn::os::GetTlsValue(m_TlsSlot));
    }
    errno_t e;
    if (!cache)
    {
        if (m_EnableThreadCache)
        {
            GetCache(GetCentral(&m_CentralHeap), m_TlsSlot);
            cache = reinterpret_cast<nn::nlibsdk::heap::TlsHeapCache*>(nn::os::GetTlsValue(m_TlsSlot));
        }
        if (!cache)
        {
            e = GetCentral(&m_CentralHeap)->Realloc(addr, alignedNewSize, &p);
            if (e == 0)
            {
                return p;
            }
            else
            {
                return NULL;
            }
        }
    }

    e = cache->Realloc(addr, alignedNewSize, &p);
    if (e == 0)
    {
        return p;
    }
    else
    {
        return NULL;
    }
}

size_t StandardAllocator::GetSizeOf( const void* addr ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);

    // 少なくともデフォルトアライメントは取られていることの確認
    if ((reinterpret_cast<uintptr_t>(addr) % DefaultAlignment) != 0)
    {
        nn::nlibsdk::heap::NotNMallocPtrError(addr);
        return 0;
    }

    nn::nlibsdk::heap::TlsHeapCache* p = NULL;
    if (m_EnableThreadCache)
    {
        p = reinterpret_cast<nn::nlibsdk::heap::TlsHeapCache*>(nn::os::GetTlsValue(m_TlsSlot));
    }
    if (!p)
    {
        if (m_EnableThreadCache)
        {
            GetCache(GetCentral(&m_CentralHeap), m_TlsSlot);
            p = reinterpret_cast<nn::nlibsdk::heap::TlsHeapCache*>(nn::os::GetTlsValue(m_TlsSlot));
        }
        if (!p)
        {
            return GetCentral(&m_CentralHeap)->GetAllocSize(addr);
        }
    }

    return p->GetAllocSize(addr);
}

size_t StandardAllocator::GetTotalFreeSize() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);

    size_t retVal;
    errno_t e = GetCentral(&m_CentralHeap)->Query(kNmallocQueryFreeSizeNx, &retVal);
    // 仮想メモリ機能が有効でない
    if (e != 0)
    {
        e = GetCentral(&m_CentralHeap)->Query(kNmallocQueryFreeSize, &retVal);
        NN_SDK_ASSERT(e == 0);
    }

    return retVal;
}

size_t StandardAllocator::GetAllocatableSize() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);

    size_t retVal;
    errno_t e = GetCentral(&m_CentralHeap)->Query(kNmallocQueryMaxAllocatableSizeNx, &retVal);
    // 仮想メモリ機能が有効でない
    if (e != 0)
    {
        e = GetCentral(&m_CentralHeap)->Query(kNmallocQueryMaxAllocatableSize, &retVal);
        NN_SDK_ASSERT(e == 0);
    }

    return retVal;
}

void StandardAllocator::ClearThreadCache() const NN_NOEXCEPT
{
    if (m_EnableThreadCache)
    {
        nn::nlibsdk::heap::TlsHeapCache* p = reinterpret_cast<nn::nlibsdk::heap::TlsHeapCache*>(nn::os::GetTlsValue(m_TlsSlot));
        nn::nlibsdk::heap::CachedHeap cache;
        cache.Reset(p);
        cache.Query(kNmallocQueryClearCache);
        cache.Release();
    }
}

void StandardAllocator::CleanUpManagementArea() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);

    errno_t e = GetCentral(&m_CentralHeap)->Query(kNmallocQueryUnifyFreelist);
    NN_UNUSED(e);
    NN_SDK_ASSERT(e == 0);
}

void StandardAllocator::WalkAllocatedBlocks( WalkCallback callback, void* userPtr ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);

    ClearThreadCache();
    GetCentral(&m_CentralHeap)->WalkAllocatedPtrs(callback, userPtr);
}

void StandardAllocator::Dump() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);

    size_t retVal;
    NN_UNUSED(retVal);
    errno_t e = GetCentral(&m_CentralHeap)->Query(kNmallocQueryMaxAllocatableSizeNx, &retVal);
    // 仮想メモリ機能が有効な場合は page_summary を省略する
    // 巨大な仮想メモリ領域を与えた際に page_summary が出力の大半を占めるため
    if (e == 0)
    {
        GetCentral(&m_CentralHeap)->Query(kNmallocQueryDump, kNmallocDumpSpans | kNmallocDumpPointers, 1);
    }
    else
    {
        GetCentral(&m_CentralHeap)->Query(kNmallocQueryDump, kNmallocDumpAll, 1);
    }
}

StandardAllocator::AllocatorHash StandardAllocator::Hash() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Initialized);

    // Walk しつつ SHA1 を計算する
    char sha1val[nn::crypto::Sha1Generator::HashSize];
    AllocatorHash allocatorHash;
    InternalHash hash;
    hash.allocCount = 0;
    hash.allocSize = 0;
    hash.sha1.Initialize();
    WalkAllocatedBlocks(CalcSha1Callback, reinterpret_cast<void*>(&hash));
    hash.sha1.GetHash(sha1val, sizeof(sha1val));

    // 結果をユーザーに返す
    allocatorHash.allocCount = hash.allocCount;
    allocatorHash.allocSize  = hash.allocSize;
    memcpy(&allocatorHash.hash, sha1val, sizeof(allocatorHash.hash));
    return allocatorHash;
}

}}  // namespace nn::mem
