﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <cerrno>
#include <cstdio>
#include <algorithm>
#include <cstring>
#include <nn/mem/detail/mem_Log.h>
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <pthread.h>
#endif

// 64bit 対応
// TODO: Juno a57 は ILP32 だが max_align_t が 16 となる。
//       このままだと NLIB_STATIC_ASSERT に引っかかるため、暫定的に juno-a57 が
//       NLIB_64BIT マクロに引っかかるように NN_OS_CPU_ARM_AARCH64_ARMV8A を定義しておく。
//       ILP32 で AArch64 なのは、今のところ Juno a57 のみ。
//       同データモデル、 CPU アーキテクチャで max_align_t が違うターゲット対応の際は要修正
#if defined(NN_BUILD_CONFIG_ABI_LP64) || defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
# define NLIB_64BIT
#endif

// nlib のマクロを Siglo のマクロに置き換え
#define NLIB_NO_TSAN

#define NLIB_THREAD_AA_(x)
#define NLIB_CAPABILITY(x) NLIB_THREAD_AA_(capability(x))
#define NLIB_SCOPED_CAPABILITY NLIB_THREAD_AA_(scoped_lockable)
#define NLIB_PT_GUARDED_BY(x) NLIB_THREAD_AA_(pt_guarded_by(x))
#define NLIB_ACQUIRED_BEFORE(...) NLIB_THREAD_AA_(acquired_before(__VA_ARGS__))
#define NLIB_ACQUIRED_AFTER(...) NLIB_THREAD_AA_(acquired_after(__VA_ARGS__))
#define NLIB_REQUIRES(...) NLIB_THREAD_AA_(requires_capability(__VA_ARGS__))
#define NLIB_REQUIRES_SHARED(...) NLIB_THREAD_AA_(requires_shared_capability(__VA_ARGS__))
#define NLIB_ACQUIRE(...) NLIB_THREAD_AA_(acquire_capability(__VA_ARGS__))
#define NLIB_ACQUIRE_SHARED(...) NLIB_THREAD_AA_(acquire_shared_capability(__VA_ARGS__))
#define NLIB_RELEASE(...) NLIB_THREAD_AA_(release_capability(__VA_ARGS__))
#define NLIB_RELEASE_SHARED(...) NLIB_THREAD_AA_(release_shared_capability(__VA_ARGS__))
#define NLIB_TRY_ACQUIRE(...) NLIB_THREAD_AA_(try_acquire_capability(__VA_ARGS__))
#define NLIB_TRY_ACQUIRE_SHARED(...) NLIB_THREAD_AA_(try_acquire_shared_capability(__VA_ARGS__))
#define NLIB_EXCLUDES(...) NLIB_THREAD_AA_(locks_excluded(__VA_ARGS__))
#define NLIB_ASSERT_CAPABILITY(x) NLIB_THREAD_AA_(assert_capability(x))
#define NLIB_ASSERT_SHARED_CAPABILITY(x) NLIB_THREAD_AA_(assert_shared_capability(x))
#define NLIB_RETURN_CAPABILITY(x) NLIB_THREAD_AA_(lock_returned(x))

#define NLIB_NOEXCEPT NN_NOEXCEPT
#define NLIB_FINAL final
#define NLIB_VIS_PUBLIC
#define NLIB_VIS_PUBLIC_ALT NLIB_VIS_PUBLIC
#define NLIB_VIS_HIDDEN
#define NLIB_ALWAYS_INLINE NN_FORCEINLINE
#define NLIB_NEVER_INLINE __attribute__((__noinline__))
#define NLIB_ASSERT NN_SDK_ASSERT
#define NLIB_ASSERT_NOERR(e) NLIB_ASSERT((e) == 0)
#define NLIB_STATIC_ASSERT(x) NN_STATIC_ASSERT(x)
#define NLIB_DISALLOW_COPY_AND_ASSIGN NN_DISALLOW_COPY
#define NLIB_UNUSED NN_UNUSED
#define NLIB_DEPRECATED
#define NLIB_LIKELY(x) (x)
#define NLIB_UNLIKELY(x) (x)
#define nlib_printf NN_DETAIL_MEM_TRACE_V1
#define nlib_debug_break(...) NN_ABORT("")
#define __PRIS_PREFIX "z"
#define PRIuS __PRIS_PREFIX "u"
#define PRIxPTR "p"
#define PRIuPTR "u"
#define NLIB_CHECK_RESULT
#define NLIB_ATTRIBUTE_MALLOC
#define NLIB_ATTRIBUTE_ALLOC_SIZE1(n)
#define NLIB_ATTRIBUTE_ALLOC_SIZE2(n0, n1)
#define NLIB_ATTRIBUTE_ALLOC_ALIGN(algn)
#define NLIB_ATTRIBUTE_ASSUME_ALIGNED(n)
#define NLIB_EINVAL_IFNULL(p)

#ifdef __cpp_constexpr
# define NLIB_CEXPR constexpr
# if __cpp_constexpr >= 201304L
#  define NLIB_CEXPR14 constexpr
# else
#  define NLIB_CEXPR14
# endif
#else
# define NLIB_CEXPR
# define NLIB_CEXPR14
#endif


#define NLIB_NAMESPACE_END      }}

#if defined(__ARM_NEON__) || defined(__aarch64__)
# include <arm_neon.h>  // NOLINT
#elif defined(__SSE4_1__)
# include <smmintrin.h>  // NOLINT
#elif defined(__SSE4_2__)
# include <nmmintrin.h>  // NOLINT
#endif

#ifndef __PRIS_PREFIX
# define __PRIS_PREFIX "z"
#endif
#ifndef PRIxS
# define PRIxS __PRIS_PREFIX "x"
#endif
#define PRIx32       "x"

// 一旦何もしない
#define nlib_debug_backtrace_gettext(...)
#define NLIB_MOVE_MEMBER_HELPER_1(...)

// Lock 関連マクロ (UNIX でのみ有効なもの。Siglo では何もしない)
#define NLIB_NO_THREAD_SAFETY_ANALYSIS
#define NLIB_LOCKABLE
#define NLIB_LOCK_FUNC(...)
#define NLIB_TRYLOCK_FUNC(...)
#define NLIB_UNLOCK_FUNC(...)
#define NLIB_LOCK_REQUIRED(...)
#define NLIB_GUARDED_BY(x)

NLIB_NAMESPACE_BEGIN
struct move_tag {};
NLIB_NAMESPACE_END

#ifdef __cpp_rvalue_references
#define NLIB_DEFMOVE_PIMPL(tp) \
    NLIB_ALWAYS_INLINE tp(tp&& rhs) NLIB_NOEXCEPT : prv_(rhs.prv_) { \
        rhs.prv_ = nullptr;                                          \
    }                                                                \
    NLIB_ALWAYS_INLINE tp& operator=(tp&& rhs) NLIB_NOEXCEPT {       \
        Reset();                                                     \
        prv_ = rhs.prv_;                                             \
        rhs.prv_ = nullptr;                                          \
        return *this;                                                \
    }                                                                \
    NLIB_ALWAYS_INLINE                                               \
    tp(tp& rhs, ::nlib_ns::move_tag) NLIB_NOEXCEPT : prv_(rhs.prv_) /* NOLINT */ { \
        rhs.prv_ = nullptr;                                             \
    }                                                                \
    NLIB_ALWAYS_INLINE                                               \
    tp& assign(tp& rhs, ::nlib_ns::move_tag) NLIB_NOEXCEPT { /* NOLINT */  \
        Reset();                                                     \
        prv_ = rhs.prv_;                                             \
        rhs.prv_ = nullptr;                                             \
        return *this;                                                \
    }
#else
#define NLIB_DEFMOVE_PIMPL(tp) \
    NLIB_ALWAYS_INLINE                                               \
    tp(tp& rhs, ::nlib_ns::move_tag) NLIB_NOEXCEPT : prv_(rhs.prv_) /* NOLINT */ { \
        rhs.prv_ = nullptr;                                             \
    }                                                                \
    NLIB_ALWAYS_INLINE                                               \
    tp& assign(tp& rhs, ::nlib_ns::move_tag) NLIB_NOEXCEPT { /* NOLINT */  \
        Reset();                                                     \
        prv_ = rhs.prv_;                                             \
        rhs.prv_ = nullptr;                                             \
        return *this;                                                \
    }
#endif

typedef int nlib_thread_id;

// 環境毎の対応が必要
#if defined(NN_BUILD_CONFIG_OS_WIN32)

//
// Win32
//

#ifdef _WIN64
# define NLIB_64BIT
#endif

#include <nn/nn_Windows.h>

#define NLIB_ASSUME(cond) __assume(cond)

#if _MSC_VER >= 1500
# define NLIB_HAS_NATIVE_TYPETRAITS
# if _MSC_VER <= 1600
#  define NLIB_HAS_TR1_TYPETRAITS
# endif
#endif

#elif defined(NN_BUILD_CONFIG_OS_HORIZON)

//
// Horizon
//

#include <type_traits>  // NOLINT

typedef int errno_t;
#define NLIB_ASSUME(cond) switch (0) case 0: default: if (cond) ; else NN_ABORT("")

#else
    #error   "未サポートの OS 種別が指定されています。"
#endif

#ifndef RSIZE_MAX
# ifndef NLIB_64BIT
#  define RSIZE_MAX 0x7FFFFFFFL             // NOLINT(preprocessor/const)
# else
#  define RSIZE_MAX 0x7FFFFFFFFFFFFFFFLL    // NOLINT(preprocessor/const)
# endif
#endif

// C++11 必須
#define NLIB_SAFE_BOOL(class_name, exp) \
public:                                 \
    inline operator bool() const NLIB_NOEXCEPT { return exp; }
/*
#define NLIB_SAFE_BOOL(class_name, exp) \
public:                                 \
    explicit NLIB_ALWAYS_INLINE operator bool() const NLIB_NOEXCEPT { return (exp); }
*/

#define NLIB_ATOMIC_RELAXED (0)     // NOLINT(preprocessor/const)
#define NLIB_ATOMIC_ACQUIRE (1)     // NOLINT(preprocessor/const)
#define NLIB_ATOMIC_RELEASE (2)     // NOLINT(preprocessor/const)
#define NLIB_ATOMIC_ACQ_REL (3)     // NOLINT(preprocessor/const)
#define NLIB_ATOMIC_SEQ_CST (7)     // NOLINT(preprocessor/const)

// メモリバリア
#define NLIB_MEMORY_ORDER_RELEASE nn::os::FenceMemoryAnyStore()
#define NLIB_MEMORY_ORDER_ACQUIRE nn::os::FenceMemoryLoadAny()
#define NLIB_MEMORY_ORDER_ACQ_REL nn::os::FenceMemoryAnyAny()

// Mutex
#define nlib_mutex nn::os::MutexType

typedef struct NLIB_VIS_PUBLIC {
    size_t alloc_count;
    size_t alloc_size;
    size_t hash;
} HeapHash;

typedef struct NLIB_VIS_PUBLIC {
    void* addr;
    size_t size;
    unsigned int heap_option;
} NMallocSettings;

typedef int (*nmalloc_heapwalk_callback)(void* allocatedPtr, size_t size, void* userPtr);

typedef int64_t nlib_offset;
typedef int nlib_fd;
#define NLIB_FD_INVALID (-1)  // NOLINT(readability/define)

#define NMALLOC_HEAPOPTION_CACHE_DISABLE (0x00000004)  // NOLINT(readability/define)
#define NMALLOC_HEAPOPTION_CHECK_0 (0x00000000)  // NOLINT(readability/define)
#define NMALLOC_HEAPOPTION_CHECK_1 (0x00000008)  // NOLINT(readability/define)

typedef enum {
    kNmallocDumpBasic = 0,
    kNmallocDumpSpans = 1,
    kNmallocDumpPointers = 2,
    kNmallocDumpPageSummary = 4,
    kNmallocDumpAll = kNmallocDumpSpans | kNmallocDumpPointers | kNmallocDumpPageSummary,
    NMALLOC_DUMP_BASIC = kNmallocDumpBasic,
    NMALLOC_DUMP_SPANS = kNmallocDumpSpans,
    NMALLOC_DUMP_POINTERS = kNmallocDumpPointers,
    NMALLOC_DUMP_PAGESUMMARY = kNmallocDumpPageSummary,
    NMALLOC_DUMP_ALL = kNmallocDumpAll
} NMallocDumpMode;

typedef enum {
    kNmallocQueryDump = 0,
    kNmallocQueryPageSize,
    kNmallocQueryAllocatedSize,
    kNmallocQueryFreeSize,
    kNmallocQuerySystemSize,
    kNmallocQueryMaxAllocatableSize,
    kNmallocQueryIsClean,
    kNmallocQueryHeapHash,
    kNmallocQueryUnifyFreelist,
    kNmallocQuerySetColor,
    kNmallocQueryGetColor,
    kNmallocQuerySetName,
    kNmallocQueryGetName,
    kNmallocQueryCacheMin_,
    kNmallocQueryCheckCache,
    kNmallocQueryClearCache,
    kNmallocQueryFinalizeCache,
    kNmallocQueryFreeSizeNx,
    kNmallocQueryMaxAllocatableSizeNx,
    NMALLOC_QUERY_DUMP = kNmallocQueryDump,
    NMALLOC_QUERY_PAGE_SIZE = kNmallocQueryPageSize,
    NMALLOC_QUERY_ALLOCATED_SIZE = kNmallocQueryAllocatedSize,
    NMALLOC_QUERY_FREE_SIZE = kNmallocQueryFreeSize,
    NMALLOC_QUERY_SYSTEM_SIZE = kNmallocQuerySystemSize,
    NMALLOC_QUERY_MAX_ALLOCATABLE_SIZE = kNmallocQueryMaxAllocatableSize,
    NMALLOC_QUERY_IS_CLEAN = kNmallocQueryIsClean,
    NMALLOC_QUERY_HEAP_HASH = kNmallocQueryHeapHash,
    NMALLOC_QUERY_UNIFY_FREELIST = kNmallocQueryUnifyFreelist,
    NMALLOC_QUERY_SET_COLOR = kNmallocQuerySetColor,
    NMALLOC_QUERY_GET_COLOR = kNmallocQueryGetColor,
    NMALLOC_QUERY_SET_NAME = kNmallocQuerySetName,
    NMALLOC_QUERY_GET_NAME = kNmallocQueryGetName,
    NMALLOC_QUERY_CACHE_MIN_ = kNmallocQueryCacheMin_,
    NMALLOC_QUERY_CHECK_CACHE = kNmallocQueryCheckCache,
    NMALLOC_QUERY_CLEAR_CACHE = kNmallocQueryClearCache,
    NMALLOC_QUERY_FINALIZE_CACHE = kNmallocQueryFinalizeCache,
    NMALLOC_QUERY_FREE_SIZE_NX = kNmallocQueryFreeSizeNx,
    NMALLOC_QUERY_MAX_ALLOCATABLE_SIZE_NX = kNmallocQueryMaxAllocatableSizeNx
} NMallocQuery;

// for std::integral_constant
template<class T, T v>
struct IntegralConstant
{
    static const T value = v;
    typedef T value_type;
    typedef IntegralConstant type;
    NN_EXPLICIT_OPERATOR value_type() const
    {
        return value;
    }
};

// for std::true_type, std::false_type
typedef IntegralConstant<bool, true> TrueType;
typedef IntegralConstant<bool, false> FalseType;

NLIB_NAMESPACE_BEGIN
namespace detail {

template<class CH>
class TmpBuf {
    public:
        TmpBuf() NLIB_NOEXCEPT : buf_(NULL) {}
        ~TmpBuf() NLIB_NOEXCEPT {
            if (buf_) free(buf_);
        }
        bool alloc(size_t size) {
            buf_ = reinterpret_cast<CH*>(malloc(sizeof(*buf_) * size));
            return buf_ != NULL;
        }
        CH* get() NLIB_NOEXCEPT { return buf_; }

    private:
        CH* buf_;
};

}  // namespace detail
NLIB_NAMESPACE_END

// 名前空間にいれる
namespace nn {

// nlib_ctz64
#if defined(NN_BUILD_CONFIG_OS_WIN)
static NLIB_ALWAYS_INLINE int nlib_ctz64(uint64_t x) {
#ifdef NLIB_64BIT
    DWORD cnt;
    return _BitScanForward64(&cnt, x) ? cnt : 64;
#else
    DWORD cnt;
    DWORD dw = (DWORD)(x);
    if (_BitScanForward(&cnt, dw)) {
        return (int)(cnt);  // NOLINT
    }
    else {
        dw = (DWORD)(x >> 32);
        return _BitScanForward(&cnt, dw) ?
            (int)(32 + cnt) : 64;  // NOLINT
    }
#endif
}
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
static inline int nlib_ctz64(uint64_t x) {
    return x != 0 ? __builtin_ctzll(x) : 64;
}
#endif

//
// atomic
//
// nlib_atomic_compare_exchange32
#if defined(NN_BUILD_CONFIG_OS_WIN)
static __inline int nlib_atomic_compare_exchange32(int32_t* ptr, int32_t* expected,
                                                   int32_t desired, int weak,
                                                   int success_memorder, int failure_memorder) {
    int32_t result;
    _ReadWriteBarrier();
    result = _InterlockedCompareExchange((volatile long*)ptr, desired, *expected);  // NOLINT
    _ReadWriteBarrier();
    (void)weak;
    (void)success_memorder;
    (void)failure_memorder;
    if (result == *expected) {
        return 1;
    } else {
        *expected = result;
        return 0;
    }
}

static __inline void nlib_atomic_store32(int32_t* ptr, int32_t val, int memorder) {
    _ReadWriteBarrier();
    if (memorder != NLIB_ATOMIC_SEQ_CST) {
        // if (memorder & NLIB_ATOMIC_RELEASE)
        //    NLIB_MEMORY_ORDER_RELEASE;
        *ptr = val;
    } else {
        _InterlockedExchange((volatile long*)ptr, val);  // NOLINT
    }
    _ReadWriteBarrier();
}

#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
static __inline int nlib_atomic_compare_exchange32(int32_t* ptr, int32_t* expected,
                                                   int32_t desired, int weak,
                                                   int success_memorder, int failure_memorder) {
    return __atomic_compare_exchange_n(ptr, expected, desired, weak,
                                       success_memorder, failure_memorder);
}

static __inline void nlib_atomic_store32(int32_t* ptr, int32_t val, int memorder) {
    __atomic_store_n(ptr, val, memorder);
}
#endif

inline void nlib_atomic_thread_fence(int memorder)
{
    switch (memorder) {
    case NLIB_ATOMIC_RELAXED:
        break;
    case NLIB_ATOMIC_ACQUIRE:
        NLIB_MEMORY_ORDER_ACQUIRE;
        break;
    case NLIB_ATOMIC_RELEASE:
        NLIB_MEMORY_ORDER_RELEASE;
        break;
    case NLIB_ATOMIC_ACQ_REL:
        NLIB_MEMORY_ORDER_ACQ_REL;
        break;
    default:
        NLIB_MEMORY_ORDER_ACQ_REL;
        break;
    }
}

//
// thread
//
inline errno_t nlib_thread_getid(nlib_thread_id* id) {
    NLIB_EINVAL_IFNULL(id);
#ifdef NLIB_64BIT
    uintptr_t tmp = reinterpret_cast<uintptr_t>(nn::os::GetCurrentThread());
    *id = static_cast<nlib_thread_id>(tmp);
#else
    *id = reinterpret_cast<nlib_thread_id>(nn::os::GetCurrentThread());
#endif
    return 0;
}

inline errno_t nlib_thread_getcpu(int* cpuid) {
    NLIB_EINVAL_IFNULL(cpuid);
    *cpuid = nn::os::GetCurrentCoreNumber();
    return 0;
}

inline errno_t nlib_mutex_init(nlib_mutex* mutex)
{
    if (NLIB_UNLIKELY(!mutex))
    {
        return EINVAL;
    }
    nn::os::InitializeMutex(mutex, true, 0);
    return 0;
}

inline errno_t nlib_mutex_destroy(nlib_mutex* mutex)
{
    if (NLIB_UNLIKELY(!mutex))
    {
        return EINVAL;
    }
    nn::os::FinalizeMutex(mutex);
    return 0;
}

inline errno_t nlib_mutex_lock(nlib_mutex* mutex)
{
    if (NLIB_UNLIKELY(!mutex))
    {
        return EINVAL;
    }
    nn::os::LockMutex(mutex);
    return 0;
}
inline errno_t nlib_mutex_unlock(nlib_mutex* mutex)
{
    if (NLIB_UNLIKELY(!mutex))
    {
        return EINVAL;
    }
    nn::os::UnlockMutex(mutex);
    return 0;
}

// debug
inline errno_t nlib_debug_backtrace(size_t* __restrict result, void** __restrict buffer, size_t count)
{
    if (!result || !buffer || count == 0)
    {
        return EINVAL;
    }
    memset(buffer, 0, count * sizeof(*buffer));
    return ENOTSUP;
}

inline void nmalloc_dumpex(NMallocDumpMode mode)
{
    NN_UNUSED(mode);
}

// util
inline errno_t nlib_memset(void* buf, int ch, size_t n)
{
    if (!buf)
    {
        return EINVAL;
    }
    memset(buf, ch, n);
    return 0;
}

// memcpy_s
inline errno_t nlib_memcpy(void* s1, size_t s1max, const void* s2, size_t n)
{
    if (!s1)
    {
        return ERANGE;
    }
    if (!s2 || s1max < n)
    {
        nlib_memset(s1, 0, s1max);
        return ERANGE;
    }
    memcpy(s1, s2, n);
    return 0;
}

inline errno_t nlib_gen_random(void* buf, size_t size)
{
    NLIB_UNUSED(buf);
    NLIB_UNUSED(size);
    return ENOTSUP;
}

inline char* nlib_strcpy(char *s1, const char *s2)
{
    return strcpy(s1, s2);
}

inline size_t nlib_strlen(const char* s)
{
    return strlen(s);
}

#ifndef NLIB_MEMCPY
# define NLIB_MEMCPY(a, b, c) memcpy((a), (b), (c))
#endif

inline size_t nlib_strlcpy(char* __restrict s1, const char* __restrict s2, size_t s1max)
{
    size_t len = nlib_strlen(s2);
    if (NLIB_LIKELY(len < s1max)) {
        NLIB_MEMCPY(s1, s2, len + 1);
    } else if (NLIB_LIKELY(s1max > 0)) {
        NLIB_MEMCPY(s1, s2, s1max - 1);
        s1[s1max - 1] = '\0';
    }
    return len;
}

template<size_t N>
NLIB_ALWAYS_INLINE size_t nlib_strlcpy(char (&s1)[N], const char* s2) NLIB_NOEXCEPT {
    // CTR armcc needs cast
    return nlib_strlcpy((char*)&s1[0], (const char*)s2, N);  // NOLINT
}

inline errno_t nlib_yield() {
    nn::os::YieldThread();
    return 0;
}

// 出力関連
inline errno_t nlib_write_stdxxx(int fd, size_t* __restrict result, const void* __restrict buf, size_t count)
{
    NLIB_UNUSED(fd);
    const char* p = reinterpret_cast<const char*>(buf);
    NN_UNUSED(p);
    if (count <= 1024)
    {
        NLIB_MY_PUT(p, count);
        if (result)
        {
            *result = count;
        }
    }
    else
    {
        NLIB_MY_PUT(p, 1024);
        if (result)
        {
            *result = 1024;
        }
    }
    return 0;
}

inline errno_t nlib_write_stdout(size_t* __restrict result, const void* __restrict buf, size_t count)
{
    NLIB_EINVAL_IFNULL(result);
    NLIB_EINVAL_IFNULL(buf);
    return nlib_write_stdxxx(1, result, buf, count);
}

inline errno_t nlib_write_stderr(size_t* __restrict result, const void* __restrict buf, size_t count)
{
    NLIB_EINVAL_IFNULL(result);
    NLIB_EINVAL_IFNULL(buf);
    return nlib_write_stdxxx(2, result, buf, count);
}

inline errno_t nlib_vsnprintf(size_t* __restrict count, char* __restrict buf, size_t size, const char* __restrict fmt, va_list args)
{
    errno = 0;
    int result = vsnprintf(buf, size, fmt, args);
    if (NLIB_UNLIKELY(result < 0))
    {
        errno_t e = errno;
        return e;
    }
    if (count)
    {
        *count = static_cast<size_t>(result);
    }
    if (NLIB_UNLIKELY(static_cast<size_t>(result) >= size))
    {
        return ERANGE;
    }
    return 0;
}

inline errno_t nlib_vdprintf(nlib_fd fd, size_t* __restrict count, const char* __restrict fmt, va_list args)
{
    if (fd == 1 || fd == 2)
    {
        int result = vfprintf(fd == 1 ? stdout : stderr, fmt, args);
        if (result < 0)
        {
            return errno;
        }
        if (count)
        {
            *count = static_cast<size_t>(result);
        }
    }
    else
    {
        // nn::mem ではここに到達しない
        NN_ABORT("invalid fd");
    }
    return 0;
}

inline errno_t nlib_dprintf(nlib_fd fd, size_t* __restrict count, const char* __restrict fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    errno_t e = nlib_vdprintf(fd, count, fmt, args);
    va_end(args);
    return e;
}

// popcnt64
static int nlib_popcnt64(uint64_t x);
#if defined(__SSE4_2__)
static NLIB_ALWAYS_INLINE int nlib_popcnt64(uint64_t x) {
#ifdef NLIB_64BIT
    return (int)_mm_popcnt_u64(x);  // NOLINT
#else
    uint32_t lo = (uint32_t)(x & 0xFFFFFFFFU);  // NOLINT
    uint32_t hi = (uint32_t)((x >> 32) & 0xFFFFFFFFU);  // NOLINT
    return _mm_popcnt_u32(lo) + _mm_popcnt_u32(hi);  // NOLINT
#endif
}
#elif defined(__ARM_NEON__)
static NLIB_ALWAYS_INLINE int nlib_popcnt64(uint64_t x) {
    uint8x8_t x0 = vcnt_u8(vreinterpret_u8_u64(vcreate_u64(x)));
#ifdef __aarch64__
    return vaddv_u8(x0);
#else
    uint8x8_t x1 = vpadd_u8(x0, x0);
    uint8x8_t x2 = vpadd_u8(x1, x1);
    uint8x8_t x3 = vpadd_u8(x2, x2);
    return vget_lane_u8(x3, 0);
#endif
}
#else

extern NLIB_VIS_PUBLIC const unsigned char _nlib_popcnt_array[];

static NLIB_ALWAYS_INLINE int nlib_popcnt(uint32_t x) {
    return _nlib_popcnt_array[(x >> 24) & 0xFF] + _nlib_popcnt_array[(x >> 16) & 0xFF] +
           _nlib_popcnt_array[(x >> 8) & 0xFF] + _nlib_popcnt_array[(x)& 0xFF];
}
static NLIB_ALWAYS_INLINE int nlib_popcnt64(uint64_t x) {
    return _nlib_popcnt_array[(x >> 56) & 0xFF] + _nlib_popcnt_array[(x >> 48) & 0xFF] +
           _nlib_popcnt_array[(x >> 40) & 0xFF] + _nlib_popcnt_array[(x >> 32) & 0xFF] +
           _nlib_popcnt_array[(x >> 24) & 0xFF] + _nlib_popcnt_array[(x >> 16) & 0xFF] +
           _nlib_popcnt_array[(x >> 8) & 0xFF] + _nlib_popcnt_array[(x)& 0xFF];
}
static NLIB_ALWAYS_INLINE int nlib_popcnt16(uint16_t x) {
    return _nlib_popcnt_array[(x >> 8) & 0xFF] + _nlib_popcnt_array[(x)& 0xFF];
}
#endif

// memchr
const void* nlib_memchr(const void* s, int c, size_t n);
const void* nlib_memrchr(const void* s, int c, size_t n);

// 時間
#define nlib_time int64_t
#define NLIB_DIFF_USEC (10957LL * 24LL * 60LL * 60LL * 1000LL * 1000LL)  // NOLINT(readability/define)

inline errno_t nlib_epochtime(nlib_time* t)
{
    if (NLIB_UNLIKELY(!t))
    {
        return EINVAL;
    }
    auto x = nn::os::GetSystemTick();
    *t = nn::os::ConvertToTimeSpan(x).GetMicroSeconds() + NLIB_DIFF_USEC * 10LL;
    return 0;
}

// 環境毎の対応が必要
#if defined(NN_BUILD_CONFIG_OS_WIN32)

//
// Win32
//

// nlib_clz64(INT64_MIN) -> 0, nlib_clz64(1) -> 63
static NLIB_ALWAYS_INLINE int nlib_clz64(uint64_t x)
{
#ifdef NLIB_64BIT
    DWORD cnt;
    return _BitScanReverse64(&cnt, x) ? (int)(63 - cnt) : 64;  // NOLINT
    // return (int)(__lzcnt64(x));  // needs haswell+
#else
    DWORD cnt;
    DWORD dw = (DWORD)(x >> 32);
    if (_BitScanReverse(&cnt, dw))
    {
        return (int)(31 - cnt);  // NOLINT
    }
    else
    {
        dw = (DWORD)(x);
        return _BitScanReverse(&cnt, dw) ?
            (int)(63 - cnt) : 64;  // NOLINT
    }
#endif
}

#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
//
// Horizon
//

// nlib_clz64(INT64_MIN) -> 0, nlib_clz64(1) -> 63
static NLIB_ALWAYS_INLINE int nlib_clz64(uint64_t x)
{
    return x != 0 ? __builtin_clzll(x) : 64;
}
#else
    #error   "未サポートの OS 種別が指定されています。"
#endif

//
// Virtual Memory, Physical Memory
//
inline NLIB_VIS_PUBLIC errno_t nlib_mempagesize(size_t* size)
{
    NN_UNUSED(size);
    return 0;
}
inline NLIB_VIS_PUBLIC errno_t nlib_virtual_alloc(void** ptr, size_t size)
{
    NN_UNUSED(ptr);
    NN_UNUSED(size);
    return 0;
}
inline NLIB_VIS_PUBLIC errno_t nlib_virtual_free(void* ptr, size_t size)
{
    NN_UNUSED(ptr);
    NN_UNUSED(size);
    return 0;
}
inline NLIB_VIS_PUBLIC errno_t nlib_physical_alloc(void* ptr, size_t size, int prot)
{
    NN_UNUSED(ptr);
    NN_UNUSED(size);
    NN_UNUSED(prot);
    return 0;
}
inline NLIB_VIS_PUBLIC errno_t nlib_physical_free(void* ptr, size_t size)
{
    NN_UNUSED(ptr);
    NN_UNUSED(size);
    return 0;
}
inline NLIB_VIS_PUBLIC errno_t nlib_mlock(void* addr, size_t len)
{
    NN_UNUSED(addr);
    NN_UNUSED(len);
    return 0;
}
inline NLIB_VIS_PUBLIC errno_t nlib_munlock(void* addr, size_t len)
{
    NN_UNUSED(addr);
    NN_UNUSED(len);
    return 0;
}

} // namespace nn

#define NLIB_PHYSICAL_ALLOC_PROT_NONE  0  // NOLINT(readability/define)
#define NLIB_PHYSICAL_ALLOC_PROT_READ  1  // NOLINT(readability/define)
#define NLIB_PHYSICAL_ALLOC_PROT_WRITE 2  // NOLINT(readability/define)
#define NLIB_PHYSICAL_ALLOC_PROT_EXEC  4  // NOLINT(readability/define)

// Clang 専用
#if defined(NN_BUILD_CONFIG_COMPILER_CLANG)
namespace std {
    template <typename T>
    struct has_trivial_default_constructor
    {
        static bool const value = __is_trivially_constructible(T);
    };
}
#endif

// nlib 本家の NMalloc.h の内容を無効化
// 本ヘッダと定義が被らないようにするため
#define INCLUDE_NN_NLIB_HEAP_NMALLOC_H_
