﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/os/os_SharedMemoryApi.h>
#include <nn/os/os_SystemEventApi.h>
#include <nn/mbuf/detail/mbuf_MbufPool.h>
#include <nn/mbuf/mbuf_MbufInit.h>
#include "mbuf_MbufImpl.h"
#include "mbuf_MbufAllocation.h"
#include <nn/nn_SdkText.h>

namespace nn { namespace mbuf { namespace
{
    /**
     * @brief       プロセス間通信のためにインデックス化された mbuf です。
     */
    struct MbufIndexed
    {
        uint32_t _nextIndex;
        uint32_t _prevIndex;
        Bit32    _reserved;
    };

    /**
     * @brief       バッファの末尾に置く管理用のブロックです。
     */
    struct TailBlock
    {
        //! マジックナンバーです。
        Bit32    magic;

        //! プールの総数です。
        int32_t  poolCount;

        //! 各プールの設定情報です。
        Config   configs[PoolCountMax];
    };

    //! バッファの末尾に置くマジックナンバーです。Ascii コードで mbuf です。
    const Bit32 MagicNumber  = 0x6675626d;

    //! 無効なインデックスを示す定数です。
    const int InvalidIndex = 0xffffffff;

    bool g_IsServer;                            // mbuf プールを管理するか否かのフラグです。
    Bit8* g_Buffer;                             // mbuf プールや TailBlock を置くバッファです。
    size_t g_BufferSize;                        // 上記のバッファのサイズです。
    TailBlock* g_pTail;                         // TailBlock です。
    detail::MbufPool g_Pools[PoolCountMax];     // mbuf プールです。
    Type g_Types[TypeCountMax];                 // mbuf の用途です。
    int g_TypeCount;                            // g_Types の要素数です。

    /**
     * @brief       TailBlock の位置を計算して取得します。
     */
    TailBlock* GetTailBlock(void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        auto address = static_cast<Bit8*>(buffer) + bufferSize - sizeof(TailBlock);
        return reinterpret_cast<TailBlock*>(address);
    }

    Mbuf* GetPtrFromIndex(int index) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(g_pTail);
        if (index == InvalidIndex)
        {
            return nullptr;
        }
        int poolId      = (index >> 16) & 0xffff;
        int indexInPool = index & 0xffff;
        NN_SDK_ASSERT(poolId < static_cast<int>(g_pTail->poolCount));
        NN_SDK_ASSERT(indexInPool < g_Pools[poolId].GetClusterCount());
        return g_Pools[poolId].GetPtrFromIndex(indexInPool);
    }

    int GetIndexFromPtr(const Mbuf* pMbuf) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(g_pTail);
        if (pMbuf == nullptr)
        {
            return InvalidIndex;
        }
        int poolId      = GetPoolIdMbuf(pMbuf);
        int indexInPool = g_Pools[poolId].GetIndexFromPtr(pMbuf);
        NN_SDK_ASSERT(poolId < static_cast<int>(g_pTail->poolCount));
        NN_SDK_ASSERT(indexInPool < g_Pools[poolId].GetClusterCount());
        return (poolId << 16) | indexInPool;
    }

}}} // end of namespace nn::mbuf::<unnamed>

namespace nn { namespace mbuf
{
    size_t GetRequiredBufferSize(const Config configs[], int poolCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(configs);
        NN_SDK_REQUIRES_MINMAX(poolCount, 1, PoolCountMax);
        size_t requiredSize = sizeof(TailBlock);
        for (int i = 0; i < poolCount; ++i)
        {
            const Config& conf = configs[i];
            NN_SDK_REQUIRES_MINMAX(
                conf.unitSize, nn::mbuf::MbufUnitSizeMin, nn::mbuf::MbufUnitSizeMax);
            NN_SDK_REQUIRES_MINMAX(
                conf.count, nn::mbuf::PoolCapacityMin, nn::mbuf::PoolCapacityMax);
            requiredSize += detail::MbufPool::GetRequiredMemorySize(conf.unitSize, conf.count);
        }
        return requiredSize;
    }

    void Initialize(const Type types[], int typeCount,
                    const Config configs[], int poolCount,
                    void *buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_Buffer == nullptr, NN_TEXT("mbuf ライブラリは既に初期化されています。"));
        NN_SDK_REQUIRES_NOT_NULL(types);
        NN_SDK_REQUIRES_MINMAX(typeCount, 1, TypeCountMax);
        NN_SDK_REQUIRES_MINMAX(typeCount, 1, poolCount);
        NN_SDK_REQUIRES_NOT_NULL(configs);
        NN_SDK_REQUIRES_MINMAX(poolCount, 1, PoolCountMax);
        NN_SDK_REQUIRES_NOT_NULL(buffer);
        NN_SDK_REQUIRES(GetRequiredBufferSize(configs, poolCount) <= bufferSize);

        // TailBlock を生成します。
        auto& tail = *GetTailBlock(buffer, bufferSize);
        tail.magic     = MagicNumber;
        tail.poolCount = poolCount;
        for (int i = 0; i < poolCount; ++i)
        {
            NN_SDK_REQUIRES_RANGE(configs[i].type, 0, typeCount);
            tail.configs[i] = configs[i];
        }

        // 共通の初期化処理です。
        g_IsServer = true;
        Initialize(types, typeCount, buffer, bufferSize);
    }

    void Initialize(const Type types[], int typeCount,
                    void *buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_Buffer == nullptr, NN_TEXT("mbuf ライブラリは既に初期化されています。"));
        NN_SDK_REQUIRES_NOT_NULL(types);
        NN_SDK_REQUIRES_MINMAX(typeCount, 1, TypeCountMax);
        NN_SDK_REQUIRES_NOT_NULL(buffer);

        // バッファを保存しておきます。
        std::memcpy(g_Types, types, sizeof(Type) * typeCount);
        g_TypeCount    = typeCount;
        g_Buffer       = static_cast<Bit8*>(buffer);
        g_BufferSize   = bufferSize;
        g_pTail        = GetTailBlock(buffer, bufferSize);

        // バッファの末尾には TailBlock が格納されているはずです。
        auto& tail = *g_pTail;
        NN_SDK_REQUIRES_EQUAL(tail.magic, MagicNumber);
        NN_SDK_REQUIRES_MINMAX(tail.poolCount, 1, PoolCountMax);
        NN_SDK_REQUIRES(GetRequiredBufferSize(tail.configs, tail.poolCount) <= bufferSize);

        // mbuf pool を構築します。
        uintptr_t currentAddress = reinterpret_cast<uintptr_t>(g_Buffer);
        for (int i = 0; i < tail.poolCount; ++i)
        {
            const Config& conf = tail.configs[i];
            NN_SDK_REQUIRES_RANGE(conf.type, 0, typeCount);
            g_Pools[i].Initialize(
                i, conf.unitSize, conf.count, conf.type, currentAddress, g_IsServer);
            currentAddress += g_Pools[i].GetPoolSize();
        }
        NN_SDK_ASSERT(currentAddress <= reinterpret_cast<uintptr_t>(&tail));
    }

    void Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_Buffer != nullptr, NN_TEXT("mbuf ライブラリは初期化されていません。"));

        // 全ての mbuf プールを解放します。
        for (int i = 0; i < g_pTail->poolCount; ++i)
        {
            g_Pools[i].Finalize();
        }

        // 自身が mbuf プールを管理している場合には TailBlock を消去しておきます。
        if (g_IsServer)
        {
            std::memset(g_pTail, 0, sizeof(TailBlock));
            g_IsServer = false;
        }

        // バッファを解放します。
        g_Buffer = nullptr;
        g_BufferSize = 0;
        g_pTail = nullptr;
        std::memset(g_Types, 0, sizeof(g_Types));
        g_TypeCount = 0;
    }

    Mbuf* TryAllocate(size_t size, AllocationSizePolicty policy, int type) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_Buffer != nullptr, NN_TEXT("mbuf ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_RANGE(type, 0, g_TypeCount);
        Mbuf* pMbuf = nullptr;

        // 指定されたサイズ以上の Mbuf プールからアロケーションを行います。
        for (int i = 0; i < g_pTail->poolCount && pMbuf == nullptr; ++i)
        {
            auto& pool = g_Pools[i];
            if (size <= pool.GetDataSize())
            {
                pMbuf = pool.TryAllocate(type);
            }
        }

        // 指定されたサイズ未満の Mbuf プールも含めてアロケーションを試行します。
        if (policy == AllocationSizePolicty_NotRequired)
        {
            for (int i = 0; i < g_pTail->poolCount && pMbuf == nullptr; ++i)
            {
                auto& pool = g_Pools[i];
                pMbuf = pool.TryAllocate(type);
            }
        }

        return pMbuf;
    }

    Mbuf* Allocate(size_t size, AllocationSizePolicty policy, int type) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_Buffer != nullptr,  NN_TEXT("mbuf ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_RANGE(type, 0, g_TypeCount);
        Mbuf* pMbuf = nullptr;
        for (;;)
        {
            pMbuf = TryAllocate(size, policy, type);
            if (pMbuf != nullptr)
            {
                break;
            }
            nn::os::WaitSystemEvent(g_Types[type].pEvent);
        }
        return pMbuf;
    }

    void Free(Mbuf* pMbuf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_Buffer != nullptr, NN_TEXT("mbuf ライブラリは初期化されていません。"));
        int poolId = GetPoolIdMbuf(pMbuf);
        NN_SDK_REQUIRES_RANGE(poolId, 0, g_pTail->poolCount);
        int type = pMbuf->_type;
        NN_SDK_ASSERT_RANGE(type, 0, g_TypeCount);
        g_Pools[poolId].Free(pMbuf);
        nn::os::SignalSystemEvent(g_Types[type].pEvent);
    }

    int ConvertToIndex(Mbuf* pMbuf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_Buffer != nullptr, NN_TEXT("mbuf ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES(GetPreviousMbuf(pMbuf) == nullptr);
        int headIndex = GetIndexFromPtr(pMbuf);
        do
        {
            Mbuf* pPrev = GetPreviousMbuf(pMbuf);
            Mbuf* pNext = GetNextMbuf(pMbuf);
            auto* pIndexed = reinterpret_cast<MbufIndexed*>(pMbuf);
            pIndexed->_nextIndex = GetIndexFromPtr(pNext);
            pIndexed->_prevIndex = GetIndexFromPtr(pPrev);
            pIndexed->_reserved  = 0;
            pMbuf = pNext;
        } while (pMbuf != nullptr);
        return headIndex;
    }

    Mbuf* ConvertFromIndex(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_Buffer != nullptr, NN_TEXT("mbuf ライブラリは初期化されていません。"));
        NN_SDK_REQUIRES_NOT_EQUAL(index, InvalidIndex);
        Mbuf* pMbufHead = GetPtrFromIndex(index);
        do
        {
            Mbuf* pMbuf = GetPtrFromIndex(index);
            auto* pMbufIndexed = reinterpret_cast<MbufIndexed*>(pMbuf);
            int next = pMbufIndexed->_nextIndex;
            int prev = pMbufIndexed->_prevIndex;
            SetNextMbuf(pMbuf, GetPtrFromIndex(next));
            SetPreviousMbuf(pMbuf, GetPtrFromIndex(prev));
            MbufSetUserPointer(pMbuf, nullptr);
            index = next;
        } while (index != InvalidIndex);
        return pMbufHead;
    }

    int GetFreeCountFromPool( int type ) NN_NOEXCEPT
    {
        int out;
        for (int i = 0; i < g_pTail->poolCount; ++i)
        {
            auto& pool = g_Pools[i];
            out = pool.GetFreeCountByType(type);
            if( out != -1 )
            {
                return out;
            }
        }
        return -1;
    }

}} // end of namespace nn::mbuf
