﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/mbuf/mbuf_Definitions.h>
#include <nn/mbuf/mbuf_Mbuf.h>
#include "mbuf_MbufImpl.h"

namespace nn { namespace mbuf
{
    void InitializeMbuf(Mbuf* mbuf, size_t unitSize, int poolId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(mbuf != NULL);
        size_t dataSize;
        if( unitSize > 0 )
        {
            dataSize = unitSize - MbufHeaderSize;
        }
        else
        {
            dataSize = 0;
        }

        // メンバの初期化
        mbuf->_pNext    = NULL;
        mbuf->_pPrev    = NULL;
        mbuf->_pGeneral = NULL;
        mbuf->_len      = 0;
        mbuf->_capacity = static_cast<uint16_t>(dataSize);
        mbuf->_top      = 0;
        mbuf->_flags    = 0;
        mbuf->_protocol = 0;
        mbuf->_owner    = 0;
        mbuf->_type     = 0;
        mbuf->_pool     = static_cast<Bit16>(poolId);
        std::memset(mbuf->_dst,  0x00, sizeof(detail::MacAddress));
        std::memset(mbuf->_src,  0x00, sizeof(detail::MacAddress));
        std::memset(&mbuf->_data[0], 0x00, dataSize);
    }

    int32_t* GetNextFreeIndexPtr(Mbuf* mbuf) NN_NOEXCEPT
    {
        return &(reinterpret_cast<detail::MbufFree*>(mbuf)->_nextFree);
    }

    void SetNextFreeIndex(Mbuf* mbuf, int freeIndex) NN_NOEXCEPT
    {
        reinterpret_cast<detail::MbufFree*>(mbuf)->_nextFree = freeIndex;
    }

    int GetNextFreeIndex(Mbuf* mbuf) NN_NOEXCEPT
    {
        return reinterpret_cast<const detail::MbufFree*>(mbuf)->_nextFree;
    }

    Mbuf* GetNextMbuf(Mbuf* mbuf) NN_NOEXCEPT
    {
        return mbuf->_pNext;
    }

    const Mbuf* GetNextMbuf(const Mbuf* mbuf) NN_NOEXCEPT
    {
        return mbuf->_pNext;
    }

    Mbuf* GetPreviousMbuf(Mbuf* mbuf) NN_NOEXCEPT
    {
        return mbuf->_pPrev;
    }

    const Mbuf* GetPreviousMbuf(const Mbuf* mbuf) NN_NOEXCEPT
    {
        return mbuf->_pPrev;
    }

    void SetNextMbuf(Mbuf* mbuf, Mbuf* pMbuf) NN_NOEXCEPT
    {
        mbuf->_pNext = pMbuf;
    }

    void SetPreviousMbuf(Mbuf* mbuf, Mbuf* pMbuf) NN_NOEXCEPT
    {
        mbuf->_pPrev = pMbuf;
    }

    bool IsLinkingMbuf(const Mbuf* mbuf) NN_NOEXCEPT
    {
        return mbuf->_pPrev != NULL || mbuf->_pNext != NULL;
    }

    Mbuf* UnlinkMbuf(Mbuf* mbuf) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(mbuf != NULL);
        Mbuf* pNext = GetNextMbuf(mbuf);
        if( pNext != NULL )
        {
            SetNextMbuf(mbuf, NULL);
            SetPreviousMbuf(pNext, NULL);
            return pNext;
        }
        else
        {
            return NULL;
        }
    }

    void CopyFromMbuf(Mbuf* mbuf, const Mbuf* pSource) NN_NOEXCEPT
    {
        SetLengthMbuf(mbuf, pSource->_len);
        mbuf->_type       = pSource->_type;
        mbuf->_flags      = pSource->_flags;
        mbuf->_protocol   = pSource->_protocol;
        mbuf->_top        = pSource->_top;
        mbuf->_owner      = pSource->_owner;

        std::memcpy(mbuf->_src, pSource->_src, sizeof(detail::MacAddress));
        std::memcpy(mbuf->_dst, pSource->_dst, sizeof(detail::MacAddress));

        std::memcpy(GetTopPtrMbuf(mbuf), GetTopPtrMbuf(const_cast<Mbuf*>(pSource)), GetLengthMbuf(mbuf));
    }

    Mbuf* GetTailMbuf(Mbuf* mbuf) NN_NOEXCEPT
    {
        Mbuf* pMbuf = mbuf;
        for( ; ; )
        {
            if( GetNextMbuf(pMbuf) )
            {
                pMbuf = GetNextMbuf(pMbuf);
            }
            else
            {
                break;
            }
        }
        return pMbuf;
    }

    Mbuf* GetHeadMbuf(Mbuf* mbuf) NN_NOEXCEPT
    {
        Mbuf* pMbuf = mbuf;
        for( ; ; )
        {
            if( GetPreviousMbuf(pMbuf) )
            {
                pMbuf = GetPreviousMbuf(pMbuf);
            }
            else
            {
                break;
            }
        }
        return pMbuf;
    }

    size_t SetLengthTrimedMbuf(Mbuf* mbuf, size_t len) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(mbuf != NULL);
        mbuf->_len = static_cast<uint16_t>((len > mbuf->_capacity) ? mbuf->_capacity : len);
        return mbuf->_len;
    }

    size_t GetLengthMbuf(const Mbuf* mbuf) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(mbuf->_len <= mbuf->_capacity);
        return mbuf->_len;
    }

    void SetLengthMbuf(Mbuf* mbuf, size_t len) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(len <= mbuf->_capacity);
        mbuf->_len = static_cast<uint16_t>(len);
    }

    MbufPacketHeaderFlag GetFlagsMbuf(const Mbuf* mbuf) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(mbuf != NULL);
        return static_cast<MbufPacketHeaderFlag>(mbuf->_flags);
    }

    void SetFlagsMbuf(Mbuf* mbuf, MbufPacketHeaderFlag flags) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(mbuf != NULL);
        mbuf->_flags = static_cast<Bit16>(flags);
    }

    void SetTypeMbuf(Mbuf* mbuf, int type) NN_NOEXCEPT
    {
        mbuf->_type = static_cast<Bit16>(type);
    }

    int GetTypeMbuf(const Mbuf* mbuf) NN_NOEXCEPT
    {
        return static_cast<int>(mbuf->_type);
    }

    void SetProtocolMbuf(Mbuf* mbuf, int protocol) NN_NOEXCEPT
    {
        mbuf->_protocol = static_cast<Bit16>(protocol);
    }

    int GetProtocolMbuf(const Mbuf* mbuf) NN_NOEXCEPT
    {
        return mbuf->_protocol;
    }

    void SetTopMbuf(Mbuf* mbuf, size_t top) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(top <= mbuf->_capacity);
        mbuf->_top = static_cast<uint16_t>(top);
    }

    size_t GetWritableLengthMbuf(const Mbuf* mbuf) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(mbuf->_top < mbuf->_capacity);
        return mbuf->_capacity - mbuf->_top;
    }

    size_t GetTopMbuf(const Mbuf* mbuf) NN_NOEXCEPT
    {
        return mbuf->_top;
    }

    void* GetTopPtrMbuf(Mbuf* mbuf, size_t offset) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(mbuf->_top + offset < mbuf->_capacity);
        return &(mbuf->_data[mbuf->_top + offset]);
    }

    const void* GetTopPtrMbuf(const Mbuf* mbuf, size_t offset) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(mbuf->_top + offset < mbuf->_capacity);
        return &(mbuf->_data[mbuf->_top + offset]);
    }

    bool IsAlignedMbuf(uintptr_t address, size_t align, size_t offset) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(align != 0 && (align & (align - 1)) == 0);
        return ((address + offset) & (align - 1)) == 0;
    }

    bool IsDataAlignedMbuf(Mbuf* mbuf, size_t align, size_t offset) NN_NOEXCEPT
    {
        return IsAlignedMbuf(reinterpret_cast<uintptr_t>(GetTopPtrMbuf(mbuf)), align, offset);
    }

    size_t GetCapacityMbuf(const Mbuf* mbuf) NN_NOEXCEPT
    {
        return mbuf->_capacity;
    }

    size_t GetUnitSizeMbuf(const Mbuf* mbuf) NN_NOEXCEPT
    {
        return mbuf->_capacity + MbufHeaderSize;
    }

    int GetPoolIdMbuf(const Mbuf* mbuf) NN_NOEXCEPT
    {
        return static_cast<int>(mbuf->_pool);
    }

    // TORIAEZU:
    size_t GetTotalLengthMbuf(const Mbuf* mbuf) NN_NOEXCEPT
    {
        const Mbuf* pWorkMbuf = mbuf;
        return MbufLength(const_cast<Mbuf*>(pWorkMbuf), NULL);
    }

    bool CopyChainToMbuf(const Mbuf* mbuf, void* pDst, size_t offset, size_t len) NN_NOEXCEPT
    {
        return MbufCopydata(mbuf, offset, len, pDst) == 0;
    }

    void LinkMbuf(Mbuf* mbuf, Mbuf* pNext) NN_NOEXCEPT
    {
        SetNextMbuf(mbuf, pNext);
        SetPreviousMbuf(pNext, mbuf);
    }

}} // end of namespace nn::mbuf
