﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ncm/ncm_Result.h>

#include <nn/lr/lr_Result.h>
#include <nn/lr/lr_RegisteredLocationResolverImpl.h>

namespace nn { namespace lr {

namespace
{
bool ResolvePath(sf::Out<lr::Path> outValue, LocationRedirector* redirector, RegisteredLocations<ncm::ProgramId, 1>* registered, ncm::ProgramId id) NN_NOEXCEPT
{
    Path path;
    if (redirector->FindRedirection(&path, id))
    {
        outValue.Set(path);
        return true;
    }

    if (registered->Find(&path, id))
    {
        outValue.Set(path);
        return true;
    }
    return false;
}
void RegisterPath(RegisteredLocations<ncm::ProgramId, 1>* registered, ncm::ProgramId id, const lr::Path& path) NN_NOEXCEPT
{
    if (registered->Register(id, path) == true)
    {
        return;
    }
    // いっぱいだった場合は、Register されているものを削除して、再度登録
    registered->Clear();
    registered->Register(id, path);
}
void UnregisterPath(RegisteredLocations<ncm::ProgramId, 1>* registered, ncm::ProgramId id) NN_NOEXCEPT
{
    // 削除対象が見つからなくても、成功扱い
    registered->Unregister(id);
}
void RedirectPath(LocationRedirector* redirector, ncm::ProgramId id, const lr::Path& path) NN_NOEXCEPT
{
    redirector->SetRedirection(id, path);
}

}

Result RegisteredLocationResolverImpl::ResolveProgramPath(sf::Out<lr::Path> outValue, ncm::ProgramId id) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(ResolvePath(outValue, &m_ProgramRedirector, &m_RegisteredPrograms, id), ResultProgramNotFound());
    NN_RESULT_SUCCESS;
}

Result RegisteredLocationResolverImpl::RegisterProgramPath(ncm::ProgramId id, const lr::Path& path) NN_NOEXCEPT
{
    RegisterPath(&m_RegisteredPrograms, id, path);
    NN_RESULT_SUCCESS;
}

Result RegisteredLocationResolverImpl::UnregisterProgramPath(ncm::ProgramId id) NN_NOEXCEPT
{
    UnregisterPath(&m_RegisteredPrograms, id);
    NN_RESULT_SUCCESS;
}

Result RegisteredLocationResolverImpl::RedirectProgramPath(ncm::ProgramId id, const lr::Path& path) NN_NOEXCEPT
{
    RedirectPath(&m_ProgramRedirector, id, path);
    NN_RESULT_SUCCESS;
}

Result RegisteredLocationResolverImpl::ResolveHtmlDocumentPath(sf::Out<lr::Path> outValue, ncm::ProgramId id) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(ResolvePath(outValue, &m_HtmlDocumentRedirector, &m_RegisteredHtmlDocuments, id), ResultHtmlDocumentNotFound());
    NN_RESULT_SUCCESS;
}

Result RegisteredLocationResolverImpl::RegisterHtmlDocumentPath(ncm::ProgramId id, const lr::Path& path) NN_NOEXCEPT
{
    RegisterPath(&m_RegisteredHtmlDocuments, id, path);
    NN_RESULT_SUCCESS;
}

Result RegisteredLocationResolverImpl::UnregisterHtmlDocumentPath(ncm::ProgramId id) NN_NOEXCEPT
{
    UnregisterPath(&m_RegisteredHtmlDocuments, id);
    NN_RESULT_SUCCESS;
}

Result RegisteredLocationResolverImpl::RedirectHtmlDocumentPath(ncm::ProgramId id, const lr::Path& path) NN_NOEXCEPT
{
    RedirectPath(&m_HtmlDocumentRedirector, id, path);
    NN_RESULT_SUCCESS;
}

}}
