﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief フレームヒープ API です。内部実装は mem_DetailFrameHeap.cpp 内にあります。
 */

#include "./detail/lmem_DetailFrameHeap.h"
#include "./detail/lmem_DetailCommonHeap.h"


namespace nn { namespace lmem {

HeapHandle CreateFrameHeap(void* startAddress, size_t size, int option) NN_NOEXCEPT
{
    HeapHandle heapHandle = detail::CreateFrameHeap(startAddress, size, option);
    if(option & CreationOption_ThreadSafe)
    {
        nn::os::InitializeMutex(&(heapHandle->mutex), false, 0);
    }
    return heapHandle;
}

HeapHandle CreateFrameHeap(void* startAddress, size_t size, int option, HeapCommonHead* pHeapHead) NN_NOEXCEPT
{
    HeapHandle heapHandle = detail::CreateFrameHeap(startAddress, size, option, pHeapHead);
    if(option & CreationOption_ThreadSafe)
    {
        nn::os::InitializeMutex(&(heapHandle->mutex), false, 0);
    }
    return heapHandle;
}

void DestroyFrameHeap(HeapHandle heapHandle) NN_NOEXCEPT
{
    if(heapHandle->attribute & CreationOption_ThreadSafe)
    {
        nn::os::FinalizeMutex(&(heapHandle->mutex));
    }
    detail::DestroyFrameHeap( heapHandle );
}

MemoryRange AdjustFrameHeap( HeapHandle heapHandle, AdjustMode adjustMode ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::AdjustFrameHeap(heapHandle, adjustMode);
}

void* AllocateFromFrameHeap(HeapHandle heapHandle, size_t size) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::AllocFromFrameHeap(heapHandle, size, DefaultAlignment);
}

void* AllocateFromFrameHeap(HeapHandle heapHandle, size_t size, int alignment) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::AllocFromFrameHeap(heapHandle, size, alignment);
}

void FreeToFrameHeap( HeapHandle heapHandle, FreeMode mode ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    detail::FreeToFrameHeap( heapHandle, mode );
}

size_t ResizeFrameHeapBlock( HeapHandle heapHandle, void* pBlock, size_t newSize ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::ResizeForMemoryBlockFrameHeap( heapHandle, pBlock, newSize );
}

size_t GetFrameHeapAllocatableSize( HeapHandle heapHandle, int alignment ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::GetAllocatableSizeForFrameHeap( heapHandle, alignment );
}

FrameHeapState GetFrameHeapState( HeapHandle heapHandle ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::GetFrameHeapState( heapHandle );
}

void RestoreFrameHeapState( HeapHandle heapHandle, const FrameHeapState& state ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    detail::RestoreFrameHeapState(heapHandle, state);
}

void DumpFrameHeap( HeapHandle heapHandle ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    detail::DumpFrameHeap(heapHandle);
}

}}  // nn::mem
