﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief 拡張ヒープ API です。内部実装は mem_DetailExpHeap.cpp 内にあります。
 */

#include "./detail/lmem_DetailExpHeap.h"

namespace nn { namespace lmem {

HeapHandle CreateExpHeap( void* startAddress, size_t size, int option ) NN_NOEXCEPT
{
    HeapHandle heapHandle = detail::CreateExpHeap( startAddress, size, option );
    if(option & CreationOption_ThreadSafe)
    {
        nn::os::InitializeMutex(&(heapHandle->mutex), false, 0);
    }
    return heapHandle;
}

void DestroyExpHeap( HeapHandle heapHandle ) NN_NOEXCEPT
{
    if(heapHandle->attribute & CreationOption_ThreadSafe)
    {
        nn::os::FinalizeMutex(&(heapHandle->mutex));
    }
    detail::DestroyExpHeap( heapHandle );
}

void* AllocateFromExpHeap( HeapHandle heapHandle, size_t size ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::AllocFromExpHeap( heapHandle, size, DefaultAlignment );
}

void* AllocateFromExpHeap( HeapHandle heapHandle, size_t size, int alignment ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::AllocFromExpHeap( heapHandle, size, alignment );
}

void FreeToExpHeap( HeapHandle heapHandle, void* pBlock ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    detail::FreeToExpHeap( heapHandle, pBlock );
}

size_t ResizeExpHeapBlock( HeapHandle heapHandle, void* pBlock, size_t newSize ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::ResizeExpHeapMemoryBlock( heapHandle, pBlock, newSize );
}

MemoryRange AdjustExpHeap( HeapHandle heapHandle ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::AdjustExpHeap( heapHandle );
}

size_t GetExpHeapTotalFreeSize( const HeapHandle heapHandle ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::GetExpHeapTotalFreeSize( heapHandle );
}

size_t GetExpHeapAllocatableSize( const HeapHandle heapHandle, int alignment ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::GetExpHeapAllocatableSize( heapHandle, alignment );
}

void SetExpHeapAllocationMode( HeapHandle heapHandle, AllocationMode allocationMode ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    detail::SetExpHeapAllocMode( heapHandle, static_cast<uint16_t>(allocationMode) );
}

AllocationMode GetExpHeapAllocationMode( const HeapHandle heapHandle ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return static_cast<AllocationMode>(detail::GetExpHeapAllocMode( heapHandle ));
}

void SetExpHeapUseMarginOfAlignment( HeapHandle heapHandle, bool reuse ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    detail::SetUseExpHeapMarginOfAlignment( heapHandle, reuse );
}

bool GetExpHeapUseMarginOfAlignment( const HeapHandle heapHandle ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::GetUseExpHeapMarginOfAlignment( heapHandle );
}

void SetExpHeapGroupId( HeapHandle heapHandle, uint16_t groupId ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    detail::SetExpHeapGroupId( heapHandle, groupId );
}

Bit16 GetExpHeapGroupId( const HeapHandle heapHandle ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::GetExpHeapGroupId( heapHandle );
}

size_t GetExpHeapBlockSize( const void* pBlock ) NN_NOEXCEPT
{
    return static_cast<size_t>(detail::GetExpHeapMemoryBlockSize( pBlock ));
}

Bit16 GetExpHeapGroupIdOfBlock( const void* pBlock ) NN_NOEXCEPT
{
    return detail::GetMemoryBlockHeapGroupId( pBlock );
}

AllocationDirection GetExpHeapAllocationDirectionOfBlock( const void* pBlock ) NN_NOEXCEPT
{
    return static_cast<AllocationDirection>(detail::GetMemoryBlockHeapAllocDir( pBlock ));
}

bool CheckExpHeap( const HeapHandle heapHandle, int optFlag ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::CheckExpHeap( heapHandle, optFlag );
}

bool CheckExpHeapBlock( const HeapHandle heapHandle, const void* pBlock, int optFlag ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    return detail::CheckExpHeapMemoryBlock( pBlock, heapHandle, optFlag );
}

void VisitExpHeapAllBlocks( HeapHandle heapHandle, HeapVisitor visitor, uintptr_t userParam ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    detail::VisitExpHeapAllocated( heapHandle, visitor, userParam );
}

void DumpExpHeap( const HeapHandle heapHandle ) NN_NOEXCEPT
{
    detail::HeapLocker locker(heapHandle);
    detail::DumpExpHeap( heapHandle );
}

}}

