﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_UnitHeap.h>

namespace nn { namespace lmem { namespace detail {

/* ========================================================================
    関数プロトタイプ
   ======================================================================== */

/**
 * @brief       ユニットヒープを作成します。
 * @param[in]   startAddress    ヒープ領域の先頭アドレス
 * @param[in]   heapSize        ヒープ領域のサイズ
 * @param[in]   unitSize        メモリブロックのサイズ
 * @param[in]   alignment       メモリブロックのアライメント
 * @param[in]   option          オプションフラグ
 * @param[in]   placement       ヒープ管理領域の配置位置
 * @param[in]   pHeapHead       ヒープ共通ヘッダ
 * @return      関数が成功した場合、作成されたユニットヒープのハンドルが返ります。
 *              関数が失敗すると、 nullptr が返ります。
 * @details     pHeapHead が nullptr なら、ヒープ管理領域はヒープ内部に作成されます。@n
 *              そうでなければ、 pHeapHead をヒープの管理領域として使用します。
 */
HeapHandle CreateUnitHeap( void* startAddress,
                           size_t heapSize,
                           size_t unitSize,
                           int alignment,
                           uint16_t option,
                           InfoPlacement placement,
                           HeapCommonHead* pHeapHead) NN_NOEXCEPT;

/**
 * @brief       ユニットヒープを作成します。
 * @param[in]   startAddress    ヒープ領域の先頭アドレス
 * @param[in]   heapSize        ヒープ領域のサイズ
 * @param[in]   unitSize        メモリブロックのサイズ
 * @param[in]   alignment       メモリブロックのアライメント
 * @param[in]   option          オプションフラグ
 * @param[in]   pHeapHead       ヒープ共通ヘッダ
 * @return      関数が成功した場合、作成されたユニットヒープのハンドルが返ります。
 *              関数が失敗すると、 nullptr が返ります。
 * @details     ヒープ管理領域として、引数 pHeapHead で渡された領域を使用します。
 */
HeapHandle CreateUnitHeap( void* startAddress,
                           size_t heapSize,
                           size_t unitSize,
                           int alignment,
                           uint16_t option,
                           HeapCommonHead* pHeapHead ) NN_NOEXCEPT;

/**
 * @brief       ユニットヒープを破棄します。
 * @param[in]   heap    ユニットヒープのハンドル
 * @return      破棄したヒープが占めていた領域へのポインタを返します。
 */
void DestroyUnitHeap  ( HeapHandle heap ) NN_NOEXCEPT;

/**
 * @brief       ヒープオブジェクトを無効化します。@n
 *              この関数を呼んだ後は Finalize 以外の操作をヒープに対して行うことができなくなります。
 * @param[in]   heap    ユニットヒープのハンドル
 */
void InvalidateUnitHeap( HeapHandle heap ) NN_NOEXCEPT;

/**
* @brief      ヒープとして与えられた領域を拡張します。
*/
void ExtendUnitHeapArea(HeapHandle heap, size_t size) NN_NOEXCEPT;

/**
 * @brief       ユニットヒープからメモリブロックを確保します。
 * @param[in]   heap    ユニットヒープのハンドル
 * @return      メモリブロックの確保が成功した場合、確保したメモリブロックへのポインタが返ります。@n
                失敗した場合、NULLが返ります。
 */
void* AllocFromUnitHeap( HeapHandle heap ) NN_NOEXCEPT;

/**
 * @brief       ユニットヒープへメモリブロックを返却します。
 * @param[in]   heap        ユニットヒープのハンドル
 * @param[in]   memBlock    返却するメモリブロックへのポインタ
 */
void FreeToUnitHeap( HeapHandle heap, void* pBlock ) NN_NOEXCEPT;

/**
 *  @brief      このユニットヒープのユニットサイズを取得します。
 *  @param[in]   heapHandle  ヒープハンドル
 */
size_t GetUnitHeapUnitSize( HeapHandle heapHandle ) NN_NOEXCEPT;

/**
 * @brief       ユニットヒープの空きメモリブロック数を取得します。
 * @param[in]   heap    ユニットヒープのハンドル
 * @return      ユニットヒープの空きメモリブロック数を返します。
 */
int GetUnitHeapAllocatableCount( HeapHandle heap ) NN_NOEXCEPT;

/**
 *  @brief      現在このヒープから確保されているユニットの数を取得します。
 * @param[in]   heap    ユニットヒープのハンドル
 */
int GetUnitHeapAllocatedCount( HeapHandle heap ) NN_NOEXCEPT;

/**
 *  @brief      ユニットのアライメントを取得します。
 *  @param[in]   heapHandle  ヒープハンドル
 */
int GetUnitHeapAlignment( HeapHandle heap ) NN_NOEXCEPT;

/**
 * @brief       メモリブロックのサイズと個数から必要なヒープのサイズを取得します。
 * @param[in]   unitSize        メモリブロックのサイズ(バイト値)
 * @param[in]   unitNum         確保するメモリブロックの総数
 * @param[in]   alignment       メモリブロックのアライメント
 * @param[in]   hasHeadInternally   ヒープの管理領域をヒープ内部に持つか
 * @return      必要なヒープのサイズを返します。
 */
size_t GetRequiredUnitHeapSize( size_t unitSize, int unitNum, int alignment, bool hasHeadInternally ) NN_NOEXCEPT;

/**
 * @brief       ユニットヒープ内部の情報を表示します。
 * @param[in]   heap    ユニットヒープのハンドル
 */
void DumpUnitHeap( HeapHandle heap ) NN_NOEXCEPT;



}}} // nn::mem::detail
