﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "lmem_DetailCommonHeap.h"
#include "lmem_DetailUnitHeap.h"
#include "lmem_DetailCommonImpl.h"

namespace nn { namespace lmem { namespace detail {

namespace {

const int MinimumAlignment = 4; // アライメントの最小値

/* ========================================================================
    static関数
   ======================================================================== */

/* ------------------------------------------------------------------------
    メモリブロックリスト操作
   ------------------------------------------------------------------------ */

/**
 * @brief       リストの先頭からメモリブロックを取得します。
 * @param[in]   list リストへのポインタ
 */
UnitHead* PopUnit( UnitList* list )
{
    UnitHead*  block = list->pHead;
    if ( block )
    {
        list->pHead = block->pNext;
    }

    return block;
}

/**
 * @brief   メモリブロックをリストの先頭に追加します。
 * @param[in]   link    追加するリスト
 * @param[in]   block   追加するメモリブロック
 */
inline void PushUnit(UnitList* list, UnitHead* block)
{
    block->pNext = list->pHead;
    list->pHead = block;
}

/**
 * @brief   ヒープヘッダへのポインタから、ユニットヒープヘッダへのポインタを取得します。
 * @param   pHead   ヒープヘッダへのポインタ
 * @return  ユニットヒープヘッダへのポインタを返します。
 */
inline UnitHeapHead* GetUnitHeapHeadPtrFromHeapHead( HeapHead* pHead )
{
    return &pHead->specificHeapHead.unitHeapHead;
}

/**
 * @brief       有効なヒープハンドルかどうかシグニチャをチェックします。
 * @param[in]   handle  ヒープハンドル
 * @retval      true    有効なヒープハンドルである場合
 * @retbal      false   有効なヒープハンドルではない場合
 */
inline bool IsValidUnitHeapHandle( HeapHandle handle )
{
    if ( handle == nullptr )
    {
        return false;
    }

    {
        HeapHead* pHeapHead = handle;
        return  pHeapHead->signature == MemUnitHeapSignature;
    }
}

/**
 * @brief       指定されたアドレスが確保されていないユニットの先頭アドレスかどうか判定します。
 * @param[in]   handle  ヒープハンドル
 * @param[in]   addr    判定するアドレス
 * @return      確保されていないユニットなら、 true を返します。
 */
bool IsFreeUnit(HeapHandle handle, uintptr_t addr)
{
    if(handle->specificHeapHead.unitHeapHead.freeList.pHead == nullptr)
    {
        return false;
    }
    else
    {
        UnitHead* pHead = handle->specificHeapHead.unitHeapHead.freeList.pHead;
        while(pHead != nullptr)
        {
            if(reinterpret_cast<uintptr_t>(pHead) == addr)
            {
                return true;
            }
            pHead = pHead->pNext;
        }
    }

    return false;
}

/**
 * @brief       UnitHeapHead を初期化します。
 * @param[out]  pUnitHeapHead   ユニットヒープのヘッダへのポインタ
 * @param[in]   pHeapStart      ヒープの開始アドレス
 * @param[in]   unitSize        ユニットサイズ
 * @param[in]   unitNum         ユニットの数
 * @param[in]   alignment       アライメント
 */
void InitUnitHeapHead(UnitHeapHead* pUnitHeapHead, void* pHeapStart, size_t unitSize, int unitNum, int alignment)
{
    pUnitHeapHead->freeList.pHead = reinterpret_cast<UnitHead*>(pHeapStart);
    pUnitHeapHead->unitSize = unitSize;

    {
        UnitHead* pUnitHead = pUnitHeapHead->freeList.pHead;
        int i;

        for ( i = 0; i < unitNum - 1; ++i, pUnitHead = pUnitHead->pNext )
        {
            pUnitHead->pNext = (UnitHead*)AddSizeToPtr( pUnitHead, unitSize );
        }

        pUnitHead->pNext = NULL;
    }

    pUnitHeapHead->count = 0;
    pUnitHeapHead->alignment = alignment;
}

}   // unnamed namespace

/* ========================================================================
    外部関数の実装
   ======================================================================== */

/**
 * @brief   ユニットヒープを作成します。
 * @details     pHeapHead が nullptr なら、ヒープ管理領域はヒープ内部に作成されます。@n
 *              そうでなければ、 pHeapHead をヒープの管理領域として使用します。
 */
HeapHandle CreateUnitHeap(void* startAddress,
                          size_t heapSize,
                          size_t unitSize,
                          int alignment,
                          uint16_t option,
                          InfoPlacement placement,
                          HeapCommonHead* pHeapHead) NN_NOEXCEPT
{
    UnitHeapHead* pUnitHeapHead = nullptr;
    void* pHeapStart = nullptr;
    void* pHeapEnd = nullptr;

    NN_SDK_ASSERT(startAddress != NULL);
    NN_SDK_ASSERT(placement == InfoPlacement_Head || placement == InfoPlacement_Tail);

    // アライメントが 4 未満の 2 のべき乗の場合は、4 にする
    if(alignment == 1 || alignment == 2)
    {
        alignment = 4;
    }

    NN_SDK_ASSERT( alignment % MinimumAlignment == 0 );
    NN_SDK_ASSERT((abs(alignment) & (abs(alignment) - 1)) == 0);
    NN_SDK_ASSERT( MinimumAlignment <= alignment);

    // ユニットはユニット同士をリンクリストでつなげるよう、
    // 最低でもポインタが格納できるサイズでなければならない
    NN_SDK_ASSERT(unitSize >= sizeof(uintptr_t));

    if(pHeapHead == nullptr)
    {
        // ヒープの管理領域をヒープ内部に作成する場合
        if(placement == InfoPlacement_Head)
        {
            pHeapHead = reinterpret_cast<HeapHead*>(RoundUpPtr( startAddress, MinimumAlignment ));
            pHeapEnd = RoundDownPtr( AddSizeToPtr( startAddress, heapSize ), MinimumAlignment );
            pUnitHeapHead = GetUnitHeapHeadPtrFromHeapHead( pHeapHead );
            pHeapStart = RoundUpPtr( AddSizeToPtr( pHeapHead, sizeof(HeapHead) ), alignment );
        }
        else if(placement == InfoPlacement_Tail)
        {
            pHeapEnd = RoundDownPtr( SubSizeToPtr(AddSizeToPtr( startAddress, heapSize ), sizeof(HeapHead)), MinimumAlignment );
            pHeapHead = reinterpret_cast<HeapHead*>(pHeapEnd);
            pUnitHeapHead = GetUnitHeapHeadPtrFromHeapHead( pHeapHead );
            pHeapStart = RoundUpPtr( startAddress, alignment );
        }
    }
    else
    {
        // ヒープの管理領域が外部から渡された
        pUnitHeapHead = GetUnitHeapHeadPtrFromHeapHead( pHeapHead );
        pHeapEnd = RoundDownPtr( AddSizeToPtr( startAddress, heapSize ), MinimumAlignment );
        pHeapStart = RoundUpPtr( startAddress, alignment );
    }

    unitSize = RoundUp( unitSize, alignment );    // 実質のブロックサイズ

    // ユニットヒープ用のサイズがとれない
    if ( CompareHigherPtr( pHeapStart, pHeapEnd ) > 0 )
    {
        return nullptr;
    }

    // ユニットが一つも作成できないサイズである
    size_t unitNum = GetOffsetFromPtr( pHeapStart, pHeapEnd ) / unitSize;
    if ( unitNum == 0 )
    {
        return nullptr;
    }

    pHeapEnd = AddSizeToPtr( pHeapStart, unitNum * unitSize );

    // ヒープ共通初期化
    InitHeapHead(pHeapHead,
                 MemUnitHeapSignature,
                 pHeapStart,
                 pHeapEnd,
                 option);

    // ユニットヒープのヘッダの初期化
    InitUnitHeapHead(pUnitHeapHead,
                     pHeapStart,
                     unitSize,
                     static_cast<int>(unitNum),
                     alignment);

    return pHeapHead;
}

/**
 * @brief ユニットヒープを破棄します。
 */
void DestroyUnitHeap( HeapHandle heap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValidUnitHeapHandle(heap) );

    if(heap->specificHeapHead.unitHeapHead.freeList.pHead != nullptr)
    {
        NN_SDK_ASSERT(heap->specificHeapHead.unitHeapHead.count == 0);
        heap->specificHeapHead.unitHeapHead.freeList.pHead = nullptr;
    }

    FinalizeHeap(heap);
}

/**
 * @brief       ヒープオブジェクトを無効化します。@n
 *              この関数を呼んだ後は Finalize 以外の操作をヒープに対して行うことができなくなります。
 */
void InvalidateUnitHeap( HeapHandle heap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValidUnitHeapHandle( heap ) );
    heap->specificHeapHead.unitHeapHead.freeList.pHead = nullptr;
}

/**
 * @brief      ヒープとして与えられた領域を拡張します。
 */
void ExtendUnitHeapArea(HeapHandle heap, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidUnitHeapHandle(heap));
    //NN_SDK_ASSERT(size % heap->specificHeapHead.unitHeapHead.unitSize == 0);

    UnitHead* pCurrent;
    // フリーリストの末尾ポインタを取得
    if (heap->specificHeapHead.unitHeapHead.freeList.pHead)
    {
        pCurrent = heap->specificHeapHead.unitHeapHead.freeList.pHead;
        // フリーリストの末尾へ移動
        while (pCurrent->pNext != NULL)
        {
            pCurrent = pCurrent->pNext;
        }

        pCurrent->pNext = reinterpret_cast<UnitHead*>(heap->pHeapEnd);
        pCurrent = pCurrent->pNext;
        pCurrent->pNext = NULL;
    }
    else
    {
        heap->specificHeapHead.unitHeapHead.freeList.pHead = reinterpret_cast<UnitHead*>(heap->pHeapEnd);
        pCurrent = heap->specificHeapHead.unitHeapHead.freeList.pHead;
        pCurrent->pNext = NULL;
    }

    // フリーリストの末尾にこれから拡張する領域をつなぐ
    void* pHeapAddStart = heap->pHeapEnd;
    void* pHeapNewEnd = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(heap->pHeapEnd) + size);
    size_t unitNum = GetOffsetFromPtr(pHeapAddStart, pHeapNewEnd) / heap->specificHeapHead.unitHeapHead.unitSize;

    for (size_t i = 0; i < unitNum - 1; ++i, pCurrent = pCurrent->pNext)
    {
        pCurrent->pNext = (UnitHead*)AddSizeToPtr(pCurrent, heap->specificHeapHead.unitHeapHead.unitSize);
    }

    pCurrent->pNext = NULL;

    heap->pHeapEnd = pHeapNewEnd;
}

/**
 * @brief ユニットヒープからメモリブロックを確保します。
 */
void* AllocFromUnitHeap( HeapHandle heap ) NN_NOEXCEPT
{
    UnitHead* pUnitHead;

    NN_SDK_ASSERT( IsValidUnitHeapHandle( heap ) );

    {

        UnitHeapHead* pUnitHeapHead = GetUnitHeapHeadPtrFromHeapHead( heap );

        pUnitHead = PopUnit( &pUnitHeapHead->freeList );

        if ( pUnitHead )
        {
#if defined(NN_DETAIL_HEAP_DEBUG)
            FillAllocMemory( heap, pUnitHead, pUnitHeapHead->unitSize );
#endif
            ++(pUnitHeapHead->count);
        }
    }
    return pUnitHead;
}

/**
 * @brief ユニットヒープへメモリブロックを返却します。
 */
void FreeToUnitHeap(HeapHandle heap, void* pBlock) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValidUnitHeapHandle( heap ) );
    if ( pBlock == NULL )
    {
        return;
    }

    {
        UnitHeapHead* pUnitHeapHead = GetUnitHeapHeadPtrFromHeapHead( heap );

#if defined(NN_DETAIL_HEAP_DEBUG)
        FillFreeMemory( heap, pBlock, pUnitHeapHead->unitSize );
#endif

        PushUnit( &pUnitHeapHead->freeList, reinterpret_cast<UnitHead*>(pBlock) );
        --(pUnitHeapHead->count);
    }
}

/**
 *  @brief      このユニットヒープのユニットサイズを取得します。
 */
size_t GetUnitHeapUnitSize( HeapHandle heap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValidUnitHeapHandle( heap ) );
    return heap->specificHeapHead.unitHeapHead.unitSize;
}

/**
 * @brief ユニットヒープの空きメモリブロック数を取得します。
 */
int GetUnitHeapAllocatableCount( HeapHandle heap ) NN_NOEXCEPT
{
    int cnt = 0;

    NN_SDK_ASSERT(IsValidUnitHeapHandle(heap));

    {
        UnitHeapHead* pUnitHeapHead = GetUnitHeapHeadPtrFromHeapHead( heap );
        UnitHead* pUnitHead = pUnitHeapHead->freeList.pHead;

        for ( ; pUnitHead; pUnitHead = pUnitHead->pNext )
        {
            ++cnt;
        }
    }
    return cnt;
}

/**
 *  @brief      現在このヒープから確保されているユニットの数を取得します。
 */
int GetUnitHeapAllocatedCount( HeapHandle heap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValidUnitHeapHandle( heap ) );
    return heap->specificHeapHead.unitHeapHead.count;
}

/**
 *  @brief      ユニットのアライメントを取得します。
 *  @param[in]   heapHandle  ヒープハンドル
 */
int GetUnitHeapAlignment( HeapHandle heap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValidUnitHeapHandle( heap ) );
    return heap->specificHeapHead.unitHeapHead.alignment;
}

/**
 * @brief メモリブロックのサイズと個数から必要なヒープのサイズを取得します。
 */
size_t GetRequiredUnitHeapSize(size_t unitSize, int unitNum, int alignment, bool hasHeadInternally) NN_NOEXCEPT
{
    if(hasHeadInternally)
    {
        return  sizeof(HeapHead)    // ヒープが内部で使用するサイズ
                + (alignment - 1)   // アライメントの調整に必要なサイズの最大
                + unitNum * RoundUp( unitSize, alignment ); // 全ユニットが必要とするサイズ
    }
    return (alignment - 1)   // アライメントの調整に必要なサイズの最大
           + unitNum * RoundUp( unitSize, alignment ); // 全ユニットが必要とするサイズ
}

/**
 * @brief ユニットヒープ内部の情報を表示します。
 */
void DumpUnitHeap( HeapHandle heap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValidUnitHeapHandle( heap ) );

    {
        HeapHead    *const pHeapHead     = heap;
        UnitHeapHead *const pUnitHeapHead = GetUnitHeapHeadPtrFromHeapHead( pHeapHead );
        const size_t HeapSize = GetOffsetFromPtr( pHeapHead->pHeapStart, pHeapHead->pHeapEnd );
        const size_t UnitSize = pUnitHeapHead->unitSize;

        DumpHeapHead(pHeapHead);

        NN_SDK_LOG("                     address(from - to):             size\n");   // ヘッダー行

        // ---------------- UsedBlock のダンプ ----------------
        NN_SDK_LOG("    (Used Nodes)\n" );
        if(pUnitHeapHead->count == 0)
        {
            NN_SDK_LOG("    NONE\n");
        }
        else
        {
            for( int i = 0; i < static_cast<int>(HeapSize / UnitSize); i++ )
            {
                uintptr_t pStart = reinterpret_cast<uintptr_t>(pHeapHead->pHeapStart) + pUnitHeapHead->unitSize * i;
                if(! IsFreeUnit(heap, pStart))
                {
                    NN_SDK_LOG("    %16p - %16p: %16zu\n",
                        pStart, pStart + UnitSize, UnitSize);
                }
            }
        }

        // ---------------- FreeBlock のダンプ ----------------
        NN_SDK_LOG("    (Free Nodes)\n" );
        if(pUnitHeapHead->freeList.pHead == nullptr)
        {
            NN_SDK_LOG("    NONE\n");
        }
        else
        {
            UnitHead* pUnitHead = pUnitHeapHead->freeList.pHead;
            while(pUnitHead != NULL)
            {
                uintptr_t pStart = reinterpret_cast<uintptr_t>(pUnitHead);
                NN_UNUSED(pStart);
                NN_SDK_LOG("    %16p - %16p: %16zu\n",
                    pStart, pStart + UnitSize, UnitSize);
                pUnitHead = pUnitHead->pNext;
            }
        }

        size_t usedSize = UnitSize * pUnitHeapHead->count;
        NN_UNUSED(usedSize);
        NN_SDK_LOG("\n");
        NN_SDK_LOG("    %zu / %zu bytes (%d%%) used\n",
            usedSize, HeapSize, static_cast<int>(static_cast<uint64_t>(usedSize) * 100 / HeapSize));
        NN_SDK_LOG("\n");

    }
}

}}} // nn::mem::detail
