﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief ヒープを管理するリストに関する構造体や関数について定義します。
 */

#include <nn/lmem/lmem_Common.h>

namespace nn { namespace lmem { namespace detail {

/**
 * @brief       リスト構造体を初期化します。
 * @param[out]  pOutList    リスト構造体へのポインタ
 * @param[in]   offset      リストに繋げたい構造体の中に存在する、LinkedListNode型のメンバ変数の構造体内でのオフセット
 * @details     offset には stddef.hで定義されているoffsetofマクロを使用すると、便利です@n
 */
void InitList(LinkedList* pOutList, uint16_t offset) NN_NOEXCEPT;

/**
 * @brief       オブジェクトをリストの最後に追加します。
 * @param[out]  pOutList    リスト構造体へのポインタ
 * @param[in]   pObject     リストに繋げたいオブジェクトへのポインタ。
 */
void AppendListObject(LinkedList* pOutList, void* pObject) NN_NOEXCEPT;

/**
 * @brief       オブジェクトをリストの先頭に挿入します。
 * @param[out]  pOutList    リスト構造体へのポインタ
 * @param[in]   pObject     リストに繋げたいオブジェクトへのポインタ
 */
void PrependListObject(LinkedList* pOutList, void* pObject) NN_NOEXCEPT;

/**
 * @brief       オブジェクトを指定された位置に挿入します。
 * @param[out]  pOutList    リスト構造体へのポインタ
 * @param[in]   target      挿入したい位置にあるオブジェクトへのポインタ
 * @param[in]   pObject     リストに繋げたいオブジェクトへのポインタ
 * @details     オブジェクトは、targetで指定されたオブジェクトの前に挿入されます。@n
                挿入先が指定されていない場合（targetがNULLの場合）、オブジェクトはリストの最後に追加されます。
 */
void InsertListObject(LinkedList* pOutList, void* target, void* pObject) NN_NOEXCEPT;

/**
 * @brief       オブジェクトをリストから削除します。
 * @param[out]  pOutList    リスト構造体へのポインタ
 * @param[in]   pObject     リストから削除したいオブジェクトへのポインタ
 */
void RemoveListObject(LinkedList* pOutList, void* pObject) NN_NOEXCEPT;

/**
 * @brief       pObjectで指定されたオブジェクトの次に繋がれているオブジェクトを返します。
 * @param[in]   pList    リスト構造体へのポインタ
 * @param[in]   pObject  リスト中のオブジェクトへのポインタ
 * @return      指定されたオブジェクトの次のオブジェクトへのポインタを返します。@n
 *              もし、次のオブジェクトが無ければ、NULLを返します。
 * @details     pObjectにNULLが指定されていた場合には、リストの先頭に繋がれているオブジェクトを返します。
 */
void* GetNextListObject(const LinkedList* pList, const void* pObject) NN_NOEXCEPT;

/**
 * @brief       pObjectで指定されたオブジェクトの前に繋がれているオブジェクトを返します。
 * @param[in]   pList   リスト構造体へのポインタ
 * @param[in]   pObject リスト中のオブジェクトへのポインタ。
 * @return      指定されたオブジェクトの前のオブジェクトへのポインタを返します。@n
 *              もし、前のオブジェクトが無ければ、NULLを返します。
 * @details     pObjectにNULLが指定されていた場合には、リストの後尾に繋がれているオブジェクトを返します。
 */
void* GetPrevListObject(const LinkedList* pList, const void* pObject) NN_NOEXCEPT;

/**
 * @brief       リストのＮ番目に繋がれているオブジェクトへのポインタを返します。
 * @param[in]   pList   リスト構造体へのポインタ
 * @param[in]   index   オブジェクトのインデックス。
 * @return      オブジェクトへのポインタを返します。@m
 *              もし、指定されたインデックスのオブジェクトが無かった場合には、NULLが返ります。
 * @details     先頭から順番にリストをたどる為、リストの後部に繋がれているオブジェクト程、時間がかかります。
 */
void* GetNthListObject(const LinkedList* pList, uint16_t index) NN_NOEXCEPT;

/**
 * @brief       引数で指定されたリストに登録されているオブジェクトの数を取得します。
 * @param[in]   pList    リスト構造体へのポインタ
 * @return      指定されたリストに登録されているオブジェクトの数を返します。
 */
inline uint16_t GetListSize( const LinkedList* pList ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pList );
    return pList->numObjects;
}

}}}

