﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ヒープを管理する双方向リンクリストに関する処理の実装です。
 * @detail  ヒープ同士をリンクリストでつなぎ、更に階層構造で管理することができます。@n
 *          ルートのリンクリストは g_RootList で、その g_RootList によって繋がれる HeapHead の childList を利用することで、子リストを作ることができます。
 */

#include "lmem_DetailList.h"

namespace nn { namespace lmem { namespace detail {

namespace {

/**
 * @brief   リストに繋げたいオブジェクトのアドレスを LinkedList の offset に合せます
 */
template <typename T, typename U>
inline LinkedListNode* AdjustOffsetToHeapHeadLink(T list, U obj) NN_NOEXCEPT
{
    return reinterpret_cast<LinkedListNode*>(reinterpret_cast<uintptr_t>(obj) + list->offset);
}

/**
 * @brief       最初のオブジェクトをリストに加えます。
 * @param[out]  pOutList    リスト構造体へのポインタ
 * @param[in]   pObject     リストに繋げたいオブジェクトへのポインタ
 */
void SetFirstObject(LinkedList* pOutList, void* pObject) NN_NOEXCEPT
{
    LinkedListNode* pLink;

    NN_SDK_ASSERT(pOutList);
    NN_SDK_ASSERT(pObject);

    pLink = AdjustOffsetToHeapHeadLink(pOutList, pObject); // pObject の LinkedListNode にアドレスを合せる

    pLink->pNextObject = NULL;
    pLink->pPrevObject = NULL;
    pOutList->pHeadObject = pObject;
    pOutList->pTailObject = pObject;
    pOutList->numObjects++;
}

} // unnamed namespace

/**
 * @brief       リスト構造体を初期化します。
 */
void InitList(LinkedList* pOutList, uint16_t offset) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutList);

    pOutList->pHeadObject = NULL;
    pOutList->pTailObject = NULL;
    pOutList->numObjects = 0;
    pOutList->offset     = offset;
}

/**
 * @brief       オブジェクトをリストの最後に追加します。
 */
void AppendListObject(LinkedList* pOutList, void* pObject) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutList);
    NN_SDK_ASSERT(pObject);

    if (pOutList->pHeadObject == NULL)
    {
        // リストが空の時。
        SetFirstObject(pOutList, pObject);
    }
    else
    {
        LinkedListNode* pLink = AdjustOffsetToHeapHeadLink(pOutList, pObject);

        pLink->pPrevObject = pOutList->pTailObject;
        pLink->pNextObject = NULL;

        AdjustOffsetToHeapHeadLink(pOutList, pOutList->pTailObject)->pNextObject = pObject;
        pOutList->pTailObject = pObject;
        pOutList->numObjects++;
    }
}

/**
 * @brief       オブジェクトをリストの先頭に挿入します。
 */
void PrependListObject(LinkedList* pOutList, void* pObject) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutList);
    NN_SDK_ASSERT(pObject);

    if (pOutList->pHeadObject == NULL)
    {
        // リストが空の時。
        SetFirstObject(pOutList, pObject);
    }
    else
    {
        LinkedListNode* pLink = AdjustOffsetToHeapHeadLink(pOutList, pObject);

        pLink->pPrevObject = NULL;
        pLink->pNextObject = pOutList->pHeadObject;

        AdjustOffsetToHeapHeadLink(pOutList, pOutList->pHeadObject)->pPrevObject = pObject;
        pOutList->pHeadObject = pObject;
        pOutList->numObjects++;
    }
}

/**
 * @brief       オブジェクトを指定された位置に挿入します。
 */
void InsertListObject(LinkedList* pOutList, void* pTarget, void* pObject) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutList);
    NN_SDK_ASSERT(pObject);

    if (pTarget == NULL)
    {
        // targetが指定されていない場合は、AppendListObject()と同じ。
        AppendListObject(pOutList, pObject);
    }
    else if (pTarget == pOutList->pHeadObject)
    {
        // targetがリストの先頭である場合はPrependListObject()と同じ。
        PrependListObject(pOutList, pObject);
    }
    else
    {
        LinkedListNode* pLink    = AdjustOffsetToHeapHeadLink(pOutList, pObject);
        void*       pPrevObj = AdjustOffsetToHeapHeadLink(pOutList, pTarget)->pPrevObject;
        LinkedListNode* pPrevLink = AdjustOffsetToHeapHeadLink(pOutList, pPrevObj);

        pLink->pPrevObject    = pPrevObj;
        pLink->pNextObject    = pTarget;
        pPrevLink->pNextObject = pObject;
        AdjustOffsetToHeapHeadLink(pOutList, pTarget)->pPrevObject = pObject;
        pOutList->numObjects++;
    }
}

/**
 * @brief       オブジェクトをリストから削除します。
 */
void RemoveListObject(LinkedList* pOutList, void* pObject) NN_NOEXCEPT
{
    LinkedListNode* pLink;

    NN_SDK_ASSERT(pOutList);
    NN_SDK_ASSERT(pObject);

    pLink = AdjustOffsetToHeapHeadLink(pOutList, pObject);

    if (pLink->pPrevObject == NULL)
    {
        pOutList->pHeadObject = pLink->pNextObject;
    }
    else
    {
        AdjustOffsetToHeapHeadLink(pOutList, pLink->pPrevObject)->pNextObject = pLink->pNextObject;
    }
    if (pLink->pNextObject == NULL)
    {
        pOutList->pTailObject = pLink->pPrevObject;
    }
    else
    {
        AdjustOffsetToHeapHeadLink(pOutList, pLink->pNextObject)->pPrevObject = pLink->pPrevObject;
    }
    pLink->pPrevObject = NULL;
    pLink->pNextObject = NULL;
    pOutList->numObjects--;
}

/**
 * @brief       pObjectで指定されたオブジェクトの次に繋がれているオブジェクトを返します。
 */
void* GetNextListObject(const LinkedList* pList, const void* pObject) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pList);

    if (pObject == NULL)
    {
        return pList->pHeadObject;
    }
    return AdjustOffsetToHeapHeadLink(pList, pObject)->pNextObject;
}

/**
 * @brief       pObjectで指定されたオブジェクトの前に繋がれているオブジェクトを返します。
 */
void* GetPrevListObject(const LinkedList* pList, const void* pObject) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pList);

    if (pObject == NULL)
    {
        return pList->pTailObject;
    }
    return AdjustOffsetToHeapHeadLink(pList, pObject)->pPrevObject;
}

/**
 * @brief       リストのＮ番目に繋がれているオブジェクトへのポインタを返します。
 */
void* GetNthListObject(const LinkedList* pList, uint16_t index) NN_NOEXCEPT
{
    int count  = 0;
    LinkedListNode* pObject = NULL;

    NN_SDK_ASSERT(pList);

    while ((pObject = reinterpret_cast<LinkedListNode*>(GetNextListObject(pList, pObject))) != NULL)
    {
        if (index == count)
        {
            return pObject;
        }
        count++;
    }
    return NULL;
}

}}}



