﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include "lmem_DetailCommonHeap.h"
#include "lmem_DetailExpHeap.h"

namespace nn { namespace lmem { namespace detail {

namespace {

// フリーメモリブロックのシグネチャ
const uint16_t FreeBlockSignature = 0x4652;     // "FR"

// 使用メモリブロックのシグネチャ
const uint16_t UsedBlockSignature = 0x5544;     // "UD"

// グループIDの最大値
const int MaxGroupId = 0xff;

// グループIDのデフォルト値
const int DefaultGroupId = 0;

// アライメントの最小値
const int MinimumAlignment = 4;

// アライメントによるアドレスのずれ（パディング値）を保持しておくアライメントの最大値
// これを超えるアライメントが指定された場合、パディングはフリーブロックとして追加する
const int MaxPaddingAlignment = 128;

// デフォルトのメモリ確保モード
const int DefaultAllocationMode = AllocationMode_FirstFit;

// フリーブロックとして登録する最小サイズ (ヘッダは含まないサイズ)
const int MinimumFreeBlockSize = 4;

/**
 * @brief   メモリブロックとするリージョンを表す構造体です。
 * @detail  メモリブロックとして利用したいアドレス空間を一時的に保管するための構造体です。
 */
struct MemoryRegion
{
    void* pStart;
    void* pEnd;
};

/**
 * @brief HeapHandle が有効なものか判別します。
 */
inline bool IsValidHeapHandle(HeapHandle heapHandle) NN_NOEXCEPT
{
    return heapHandle->signature == MemExpHeapSignature;
}

/**
 * @brief ヒープヘッダへのポインタから、拡張ヒープヘッダへのポインタを取得します。
 * @param[in]   pHead   ヒープヘッダへのポインタ
 * @return      拡張ヒープヘッダへのポインタを返します。
 */
inline ExpHeapHead* GetHeapHeadPtrFromHeapHead(HeapHead* pHead) NN_NOEXCEPT
{
    return &pHead->specificHeapHead.expHeapHead;
}

inline ExpHeapHead const* GetHeapHeadPtrFromHeapHead(HeapHead const* pHead) NN_NOEXCEPT
{
    return &pHead->specificHeapHead.expHeapHead;
}

/**
 * @brief       拡張ヒープヘッダへのポインタから、ヒープヘッダへのポインタを取得します。
 * @param[in]   pHead   拡張ヒープヘッダへのポインタ
 * @return      ヒープヘッダへのポインタを返します。
 */
inline HeapHead* GetHeapHeadPtrFromHeapHead(ExpHeapHead* pHead) NN_NOEXCEPT
{
    return reinterpret_cast<HeapHead*>(SubSizeToPtr(pHead, sizeof(HeapHead) - sizeof(SpecificHeapHead)));
}

/**
 * @brief       拡張ヒープハンドルから、拡張ヒープヘッダへのポインタを取得します。
 * @param[in]   heapHandle    拡張ヒープハンドル
 * @return      拡張ヒープヘッダへのポインタを返します。
 */
inline ExpHeapHead* GetHeapHeadPtrFromHandle(HeapHandle heapHandle) NN_NOEXCEPT
{
    return GetHeapHeadPtrFromHeapHead(heapHandle);
}

/**
 * @brief       ExpHeapMemoryBlockHead構造体へのポインタから、メモリブロックへのポインタを取得します。
 * @param[in]   pBlockHead     ExpHeapMemoryBlockHead構造体へのポインタ
 * @return      メモリブロックへのポインタを返します。
 */
inline void* GetMemoryBlockMemPtr(ExpHeapMemoryBlockHead* pBlockHead) NN_NOEXCEPT
{
    return AddSizeToPtr(pBlockHead, sizeof(ExpHeapMemoryBlockHead));
}

inline const void*
GetMemoryBlockMemConstPtr(const ExpHeapMemoryBlockHead* pBlockHead) NN_NOEXCEPT
{
    return reinterpret_cast<const void*>( GetUIntPtr(pBlockHead) + sizeof(ExpHeapMemoryBlockHead));
}

/**
 * @brief       メモリブロックへのポインタから、 ExpHeapMemoryBlockHead 構造体へのポインタを取得します。
 * @param[in]   pMemBlock    メモリブロックへのポインタ
 * @return      ExpHeapMemoryBlockHead 構造体へのポインタを返します。
 */
inline ExpHeapMemoryBlockHead* GetMemoryBlockHeadPtr(void* pMemBlock) NN_NOEXCEPT
{
    return reinterpret_cast<ExpHeapMemoryBlockHead*>(SubSizeToPtr(pMemBlock, sizeof(ExpHeapMemoryBlockHead)));
}

inline const ExpHeapMemoryBlockHead* GetMemoryBlockHeadConstPtr(const void* pMemBlock) NN_NOEXCEPT
{
    return reinterpret_cast<const ExpHeapMemoryBlockHead*>(GetUIntPtr(pMemBlock) - sizeof(ExpHeapMemoryBlockHead));
}

/**
 * @brief       メモリブロックの末尾アドレスを取得します。
 */
inline void* GetMemoryBlockEndAddr(ExpHeapMemoryBlockHead* pBlockHead) NN_NOEXCEPT
{
    return AddSizeToPtr(GetMemoryBlockMemPtr(pBlockHead), pBlockHead->blockSize);
}

/**
 * @brief       ExpHeapMemoryBlockHead 構造体のアライメントによるアドレスのずれ（パディング値）を取得します。
 * @param[in]   pBlockHead  ExpHeapMemoryBlockHead構造体へのポインタ
 * @return      ExpHeapMemoryBlockHead 構造体のアライメント値を返します。
 * @details     パディング値は 0 ～ 127 バイトまで保存されます。それ以上の値はパディング値ではなく、新たなフリーメモリブロックになります。
 */
inline uint16_t GetMemoryBlockPaddingAlignment(const ExpHeapMemoryBlockHead* pBlockHead) NN_NOEXCEPT
{
    return static_cast<uint16_t>(GetBitValue(pBlockHead->attribute, 8, 7));
}

/**
 * @brief       ExpHeapMemoryBlockHead 構造体のアライメントによるアドレスのずれ（パディング値）をセットします。
 * @param[in]   pBlockHead  ExpHeapMemoryBlockHead構造体へのポインタ
 * @param[in]   alignment   セットするアライメント値
 * @details     パディング値は 0 ～ 127 バイトまで保存されます。それ以上の値はパディング値ではなく、新たなフリーメモリブロックになります。
 */
inline void SetMemoryBlockPaddingAlignment(ExpHeapMemoryBlockHead* pBlockHead, uint16_t padding) NN_NOEXCEPT
{
    SetBitValue(&pBlockHead->attribute, 8, 7, padding);
}

/**
 * @brief       ExpHeapMemoryBlockHead 構造体のグループIDを取得します。
 * @param[in]   pBlockHead  ExpHeapMemoryBlockHead構造体へのポインタ
 */
inline uint16_t GetMemoryBlockGroupId(const ExpHeapMemoryBlockHead* pBlockHead) NN_NOEXCEPT
{
    return static_cast<uint16_t>(GetBitValue(pBlockHead->attribute, 0, 8));
}

/**
 * @brief       ExpHeapMemoryBlockHead 構造体のグループIDをセットします。
 * @param[in]   pBlockHead  ExpHeapMemoryBlockHead構造体へのポインタ
 * @param[in]   id          グループID
 */
inline void SetMemoryBlockGroupId(ExpHeapMemoryBlockHead* pBlockHead, uint16_t id) NN_NOEXCEPT
{
    SetBitValue(&pBlockHead->attribute, 0, 8, id);
}

/**
 * @brief       ExpHeapMemoryBlockHead 構造体の確保方向を取得します。
 * @param[in]   pBlockHead  ExpHeapMemoryBlockHead構造体へのポインタ
 */
inline uint16_t GetMemoryBlockAllocDir(const ExpHeapMemoryBlockHead* pBlockHead) NN_NOEXCEPT
{
    return static_cast<uint16_t>(GetBitValue(pBlockHead->attribute, 15, 1));
}

/**
 * @brief       ExpHeapMemoryBlockHead 構造体の確保方向をセットします。
 * @param[in]   pBlockHead     ExpHeapMemoryBlockHead構造体へのポインタ
 * @param[in]   mode    セットする値
 */
inline void SetMemoryBlockAllocDir(ExpHeapMemoryBlockHead* pBlockHead, uint16_t mode) NN_NOEXCEPT
{
    SetBitValue(&pBlockHead->attribute, 15, 1, mode);
}

/**
 * @brief       拡張ヒープのメモリ確保モードを取得します。
 * @param[in]   pHead:  拡張ヒープヘッダへのポインタ
 * @return      拡張ヒープのメモリ確保モードを返します
 */
inline uint16_t GetAllocMode(const ExpHeapHead* pHead) NN_NOEXCEPT
{
    return static_cast<uint16_t>(GetBitValue(pHead->feature, 0, 1));
}

/**
 * @brief       拡張ヒープのメモリ確保モードをセットします。
 * @param[in]   pHead 拡張ヒープヘッダへのポインタ
 * @param[in]   mode    メモリ確保モード
 */
inline void SetAllocMode(ExpHeapHead* pHead, uint16_t mode) NN_NOEXCEPT
{
    SetBitValue(&pHead->feature, 0, 1, mode);
}

/**
 * @brief MemoryRegion 型のポインタ region に pBlockHead のアドレスを代入します。
 */
void GetRegionOfMemoryBlock(MemoryRegion* pRegion, ExpHeapMemoryBlockHead* pBlockHead) NN_NOEXCEPT
{
    pRegion->pStart = SubSizeToPtr(pBlockHead, GetMemoryBlockPaddingAlignment(pBlockHead));
    pRegion->pEnd   = GetMemoryBlockEndAddr(pBlockHead);
}

/**
 * @brief   ExpHeapMemoryBlockHead 型のポインタ pBlockHead を、 pList から削除します。
 * @return  削除したブロックのひとつ前のブロックのポインタを返します。
 */
ExpHeapMemoryBlockHead* RemoveMemoryBlock(ExpHeapMemoryBlockList* pList,  ExpHeapMemoryBlockHead* pBlockHead) NN_NOEXCEPT
{
    ExpHeapMemoryBlockHead *const pPrev = pBlockHead->pPrevBlock;
    ExpHeapMemoryBlockHead *const pNext = pBlockHead->pNextBlock;

    // 前参照リンク
    if (pPrev)
    {
        pPrev->pNextBlock = pNext;
    }
    else
    {
        pList->pHead = pNext;
    }

    // 次参照リンク
    if (pNext)
    {
        pNext->pPrevBlock = pPrev;
    }
    else
    {
        pList->pTail = pPrev;
    }

    return pPrev;
}

/**
 * @brief       ExpHeapMemoryBlockHead 型の変数 pTarget を pList 内の pPrev の次に挿入します。
 * @param[out]  pList   挿入先のメモリブロックリスト
 * @param[in]   pTarget 挿入するメモリブロック
 * @param[in]   pPrev   挿入する位置の一つ前のメモリブロック
 * @return      挿入した pTarget のポインタを返します。
 */
ExpHeapMemoryBlockHead* InsertMemoryBlock(ExpHeapMemoryBlockList* pList, ExpHeapMemoryBlockHead* pTarget, ExpHeapMemoryBlockHead* pPrev) NN_NOEXCEPT
{
    ExpHeapMemoryBlockHead* next;

    // 前参照リンク
    pTarget->pPrevBlock = pPrev;
    if (pPrev)
    {
        next = pPrev->pNextBlock;
        pPrev->pNextBlock = pTarget;
    }
    else
    {
        next = pList->pHead;
        pList->pHead = pTarget;
    }

    // 次参照リンク
    pTarget->pNextBlock = next;
    if (next)
    {
        next->pPrevBlock = pTarget;
    }
    else
    {
        pList->pTail = pTarget;
    }

    return pTarget;
}

/**
 * @brief       メモリブロックをリストの最後に追加します。
 * @param[in]   link    追加するリスト
 * @param[in]   pBlockHead   追加するメモリブロック
 */
inline void AppendMemoryBlock(ExpHeapMemoryBlockList* pList, ExpHeapMemoryBlockHead* pBlockHead) NN_NOEXCEPT
{
    InsertMemoryBlock(pList, pBlockHead, pList->pTail);
}

/**
 * @brief       メモリブロックの初期化を行います。
 * @param[in]   pRegion     メモリブロックとするリージョンを表す構造体へのポインタ
 * @param[in]   signature   メモリブロックのシグネチャ
 * @return      初期化したメモリブロックへのポインタを返します。
 */
ExpHeapMemoryBlockHead* InitMemoryBlock(const MemoryRegion* pRegion, uint16_t signature) NN_NOEXCEPT
{
    ExpHeapMemoryBlockHead* pBlockHead = reinterpret_cast<ExpHeapMemoryBlockHead*>(pRegion->pStart);

    pBlockHead->signature = signature;
    pBlockHead->attribute = 0;
    pBlockHead->blockSize = GetOffsetFromPtr(GetMemoryBlockMemPtr(pBlockHead), pRegion->pEnd);
    pBlockHead->pPrevBlock = NULL;
    pBlockHead->pNextBlock = NULL;

    return pBlockHead;
}

/**
 * @brief       メモリブロックをフリーブロック用に初期化を行います。
 * @param[in]   pRegion メモリブロックとするリージョンを表す構造体へのポインタ
 * @return      初期化したメモリブロックへのポインタを返します。
 */
inline ExpHeapMemoryBlockHead* InitFreeMemoryBlock(const MemoryRegion* pRegion) NN_NOEXCEPT
{
    return InitMemoryBlock(pRegion, FreeBlockSignature);
}

/**
 * @brief       拡張ヒープの初期化を行います。
 * @param[in]   pStartAddress    拡張ヒープとするメモリの開始アドレス。
 * @param[in]   pEndAddress      拡張ヒープとするメモリの終了アドレス +1。
 * @param[in]   optFlag         オプションフラグ。
 * @return      ヒープヘッダへのポインタを返します。
 */
HeapHead* InitHeap(void* pStartAddress, void* pEndAddress, int optFlag) NN_NOEXCEPT
{
    HeapHead* pHeapHead = reinterpret_cast<HeapHead*>(pStartAddress);
    ExpHeapHead* pExpHeapHead = GetHeapHeadPtrFromHeapHead(pHeapHead);

    InitHeapHead(           // ヒープ共通初期化
        pHeapHead,
        MemExpHeapSignature,
        AddSizeToPtr( pExpHeapHead, sizeof(SpecificHeapHead) ),   // heapStart
        pEndAddress,                                        // heapEnd
        static_cast<uint16_t>(optFlag));

    pExpHeapHead->groupId = DefaultGroupId;      // グループID
    pExpHeapHead->feature = 0;
    pExpHeapHead->isReuse = false;
    SetAllocMode(pExpHeapHead, DefaultAllocationMode);

    // フリーのブロックを作る
    {
        ExpHeapMemoryBlockHead* pBlockHead;
        MemoryRegion region;
        region.pStart = pHeapHead->pHeapStart;
        region.pEnd   = pHeapHead->pHeapEnd;
        pBlockHead = InitFreeMemoryBlock(&region);

        // ブロックリスト
        pExpHeapHead->freeList.pHead = pBlockHead;
        pExpHeapHead->freeList.pTail = pBlockHead;
        pExpHeapHead->usedList.pHead = NULL;
        pExpHeapHead->usedList.pTail = NULL;

        return pHeapHead;
    }
}

/**
 * @brief       フリーブロックの中から新しいメモリブロックを確保します。
 * @param[in]   pHead           拡張ヒープヘッダへのポインタ
 * @param[in]   pFreeBlockHead  フリーブロックヘッダへのポインタ
 * @param[in]   pBlock          確保するメモリブロックのアドレス
 * @param[in]   size            確保するメモリブロックのサイズ
 * @param[in]   direction       確保方向
 * @return      確保したメモリブロックの先頭のポインタを返します。
 */
void* AllocUsedBlockFromFreeBlock(ExpHeapHead* pHead,
                                  ExpHeapMemoryBlockHead* pFreeBlockHead,
                                  void* pBlock,
                                  size_t size,
                                  AllocationDirection direction) NN_NOEXCEPT
{
    MemoryRegion freeRegionTop;
    MemoryRegion freeRegionBottom;
    ExpHeapMemoryBlockHead* pFreeBlockHeadPrev;

    // pFreeBlockHead を freeRegionTop と Bottom に分ける
    GetRegionOfMemoryBlock(&freeRegionTop, pFreeBlockHead);
    freeRegionBottom.pEnd   = freeRegionTop.pEnd;
    freeRegionBottom.pStart = AddSizeToPtr(pBlock, size);
    freeRegionTop.pEnd   = SubSizeToPtr(pBlock, sizeof(ExpHeapMemoryBlockHead));

    pFreeBlockHeadPrev = RemoveMemoryBlock(&pHead->freeList, pFreeBlockHead);  // 一旦フリーブロックを削除

    // アライメントによって生まれた隙間領域、もしくは、確保領域を引いたフリーブロックの余りが
    // フリーブロックとするには、サイズが小さい場合
    // もしくは、断片化領域を再利用しない場合
    if ((GetOffsetFromPtr(freeRegionTop.pStart, freeRegionTop.pEnd) < sizeof(ExpHeapMemoryBlockHead) + MinimumFreeBlockSize) ||
        (direction == AllocationDirection_Front && !pHead->isReuse && GetOffsetFromPtr(freeRegionTop.pStart, freeRegionTop.pEnd) < MaxPaddingAlignment))
    {
        freeRegionTop.pEnd = freeRegionTop.pStart;  // 使用ブロックのアライメント値に含める
    }
    else
    {
        pFreeBlockHeadPrev = InsertMemoryBlock(&pHead->freeList, InitFreeMemoryBlock(&freeRegionTop), pFreeBlockHeadPrev);
    }

    // フリーブロック作る余裕が無い場合
    // もしくは、断片化領域を再利用しない場合
    if ((GetOffsetFromPtr(freeRegionBottom.pStart, freeRegionBottom.pEnd) < sizeof(ExpHeapMemoryBlockHead) + MinimumFreeBlockSize) ||
        (direction == AllocationDirection_Rear && !pHead->isReuse && GetOffsetFromPtr(freeRegionBottom.pStart, freeRegionBottom.pEnd) < MaxPaddingAlignment))
    {
        freeRegionBottom.pStart= freeRegionBottom.pEnd;   // 使用ブロックに含める
    }
    else
    {
        InsertMemoryBlock(&pHead->freeList, InitFreeMemoryBlock(&freeRegionBottom), pFreeBlockHeadPrev);
    }

    // デバグ用メモリフィル
#if defined(NN_DETAIL_HEAP_DEBUG)
    FillAllocMemory(GetHeapHeadPtrFromHeapHead(pHead), freeRegionTop.pEnd, GetOffsetFromPtr(freeRegionTop.pEnd, freeRegionBottom.pStart));
#endif

    // 新規ブロック作成
    {
        ExpHeapMemoryBlockHead* pBlockHeadNewUsed;
        MemoryRegion region;

        region.pStart = SubSizeToPtr(pBlock, sizeof(ExpHeapMemoryBlockHead));
        region.pEnd   = freeRegionBottom.pStart;

        pBlockHeadNewUsed = InitMemoryBlock(&region, UsedBlockSignature);
        SetMemoryBlockAllocDir(pBlockHeadNewUsed, static_cast<uint16_t>(direction));
        SetMemoryBlockPaddingAlignment(pBlockHeadNewUsed, static_cast<uint16_t>(GetOffsetFromPtr(freeRegionTop.pEnd, pBlockHeadNewUsed)));
        SetMemoryBlockGroupId(pBlockHeadNewUsed, pHead->groupId);
        AppendMemoryBlock(&pHead->usedList, pBlockHeadNewUsed);
    }

    return pBlock;
}

/**
 * @brief       ヒープの先頭からメモリブロックを確保します。
 * @param[in]   pHeapHead     ヒープヘッダへのポインタ。
 * @param[in]   size        確保するメモリブロックのサイズ。
 * @param[in]   alignment   アライメント値。
 * @return      メモリブロックの確保が成功した場合、確保したメモリブロックへのポインタが返ります。@n
 *              失敗した場合、NULLが返ります。
 * @details     アライメントによるアドレスの切り上げで、ヘッダとメモリブロックの間に空き領域が発生する可能性があります。
 */
void* AllocFromHead(HeapHead* pHeapHead, size_t size, int alignment) NN_NOEXCEPT
{
    ExpHeapHead* pExpHeapHead = GetHeapHeadPtrFromHeapHead(pHeapHead);

    // 最初にみつかったものを割り当てるか
    const bool isAllocFirst = GetAllocMode(pExpHeapHead) == AllocationMode_FirstFit;

    ExpHeapMemoryBlockHead* pBlockHead      = NULL;
    ExpHeapMemoryBlockHead* pBlockHeadFound = NULL;
    size_t foundSize = SIZE_MAX;
    void* pFoundBlock = NULL;

    // フリーブロック検索
    for (pBlockHead = pExpHeapHead->freeList.pHead; pBlockHead; pBlockHead = pBlockHead->pNextBlock)
    {
        void *const pBlock = GetMemoryBlockMemPtr(pBlockHead);
        void *const pReqestBlock = RoundUpPtr(pBlock, alignment);
        const size_t offset = GetOffsetFromPtr(pBlock, pReqestBlock);    // 発生したずれ

        if ( pBlockHead->blockSize >= size + offset &&
             foundSize > pBlockHead->blockSize )
        {
            pBlockHeadFound = pBlockHead;
            foundSize = pBlockHead->blockSize;
            pFoundBlock = pReqestBlock;

            if (isAllocFirst || foundSize == size)
            {
                break;
            }
        }
    }

    if (!pBlockHeadFound) // 条件に合うブロックが見つからない
    {
        return NULL;
    }

    // 発見したフリーブロックから領域確保
    return AllocUsedBlockFromFreeBlock(pExpHeapHead,
                                       pBlockHeadFound,
                                       pFoundBlock,
                                       size,
                                       AllocationDirection_Front);
}

/**
 * @brief       ヒープの末尾からメモリブロックを確保します。
 * @param[in]   pHeapHead     ヒープヘッダへのポインタ
 * @param[in]   size        確保するメモリブロックのサイズ
 * @param[in]   alignment   アライメント値
 * @return      メモリブロックの確保が成功した場合、確保したメモリブロックへのポインタが返ります。@n
 *              失敗した場合、NULLが返ります。
  * @details     アライメントによるアドレスの切り下げで、ヘッダとメモリブロックの間に空き領域が発生する可能性があります。
 */
void* AllocFromTail(HeapHead* pHeapHead, size_t size, int alignment) NN_NOEXCEPT
{
    ExpHeapHead* pExpHeapHead = GetHeapHeadPtrFromHeapHead(pHeapHead);

    // 最初にみつかったものを割り当てるか
    const bool isAllocFirst = GetAllocMode(pExpHeapHead) == AllocationMode_FirstFit;

    ExpHeapMemoryBlockHead* pBlockHead      = NULL;
    ExpHeapMemoryBlockHead* pBlockHeadFound = NULL;
    size_t foundSize = SIZE_MAX;
    void* pFoundBlock = NULL;

    // フリーブロック検索
    for (pBlockHead = pExpHeapHead->freeList.pTail; pBlockHead; pBlockHead = pBlockHead->pPrevBlock)
    {
        void *const pBlock    = GetMemoryBlockMemPtr(pBlockHead);
        void *const pBlockEnd = AddSizeToPtr(pBlock, pBlockHead->blockSize);
        void *const pReqestBlock = RoundDownPtr(SubSizeToPtr(pBlockEnd, size), alignment);  // アライン済みのアドレス

        if ( pReqestBlock >= pBlock &&
             foundSize > pBlockHead->blockSize )
        {
            pBlockHeadFound = pBlockHead;
            foundSize    = pBlockHead->blockSize;
            pFoundBlock  = pReqestBlock;

            if (isAllocFirst || foundSize == size)
            {
                break;
            }
        }
    }

    if (!pBlockHeadFound) // 条件に合うブロックが見つからない
    {
        return NULL;
    }

    // 発見したフリーブロックから領域確保
    return AllocUsedBlockFromFreeBlock(pExpHeapHead,
                                       pBlockHeadFound,
                                       pFoundBlock,
                                       size,
                                       AllocationDirection_Rear);
}

/**
 * @brief       空きリージョンをフリーメモリブロックへ組み入れます。
 * @param[in]   pEHHead:  拡張ヒープヘッダへのポインタ
 * @param[in]   pRegion:  空きリージョンへのポインタ
 * @return      関数が成功すれば true を返します。失敗すれば false を返します。
 * @details     フリーブロックと隣接している場合は、フリーブロックを拡張します。@n
 *              フリーブロックと隣接しておらず、かつフリーブロックとするほどのサイズが無い場合は、後方に隣接する使用済みブロックのアライメント値とします。@n
 *              後方に隣接する使用済みブロックが無い場合は、関数は失敗します。
 */
bool RecycleRegion(ExpHeapHead* pHead, const MemoryRegion* pRegion) NN_NOEXCEPT
{
    ExpHeapMemoryBlockHead* pBlockHeadPrevFree  = NULL;   // 直前フリーブロック
    MemoryRegion freeRgn = *pRegion;

    // 指定エリアに隣接したフリーエリアを検索
    {
        ExpHeapMemoryBlockHead* pBlockHead;

        for (pBlockHead = pHead->freeList.pHead; pBlockHead; pBlockHead = pBlockHead->pNextBlock)
        {
            if (pBlockHead < pRegion->pStart)
            {
                pBlockHeadPrevFree = pBlockHead;
                continue;
            }

            if (pBlockHead == pRegion->pEnd)   // 後方に隣接するブロックか?
            {
                // 空きリージョンを結合
                freeRgn.pEnd = GetMemoryBlockEndAddr(pBlockHead);
                RemoveMemoryBlock(&pHead->freeList, pBlockHead);

                // ヘッダ部をUnallocatedで埋める
#if defined(NN_DETAIL_HEAP_DEBUG)
                FillUnallocatedMemory(GetHeapHeadPtrFromHeapHead(pHead), pBlockHead, sizeof(ExpHeapMemoryBlockHead));
#endif
            }
            break;
        }
    }

    // 直前のフリーブロックが前方に隣接するブロックか?
    if (pBlockHeadPrevFree && GetMemoryBlockEndAddr(pBlockHeadPrevFree) == pRegion->pStart)
    {
        // 空きリージョンを結合
        freeRgn.pStart = pBlockHeadPrevFree;
        pBlockHeadPrevFree = RemoveMemoryBlock(&pHead->freeList, pBlockHeadPrevFree);
    }

    if (GetOffsetFromPtr(freeRgn.pStart, freeRgn.pEnd) < sizeof(ExpHeapMemoryBlockHead)) // ブロックになれない大きさ
    {
        // ResizeForMemoryBlockHeap()で小さいサイズを縮小しようとしていて、
        // かつ後ろにフリーブロックが無い場合にここに到達
        return false;
    }

    // デバグ用メモリフィル
#if defined(NN_DETAIL_HEAP_DEBUG)
    FillFreeMemory(GetHeapHeadPtrFromHeapHead(pHead), pRegion->pStart, GetOffsetFromPtr(pRegion->pStart, pRegion->pEnd));
#endif

    InsertMemoryBlock(&pHead->freeList, InitFreeMemoryBlock(&freeRgn), pBlockHeadPrevFree);

    return true;
}

/**
 * @brief       メモリブロックのヘッダの内容が妥当であるかチェックします。
 * @param[in]   pBlockHead  チェックするメモリブロックのヘッダへのポインタ
 * @param[in]   pHeapHead   拡張ヒープのヘッダへのポインタ
 * @param[in]   signature   メモリブロックのシグネチャ
 * @param[in]   heapType    メモリブロックのタイプ(使用 or フリー)
 * @param[in]   flag        フラグ
 * @return      メモリブロックのヘッダの内容が妥当なら true、そうでないなら false を返します。
 */
bool CheckMemoryBlock(ExpHeapMemoryBlockHead const* pBlockHead,
                      HeapHead const* pHeapHead,
                      uint16_t signature,
                      const char* heapType,
                      uint32_t flag) NN_NOEXCEPT
{
    const bool isPrint = 0 != (flag & ErrorOption_Print);
    const void *const pMemBlock = GetMemoryBlockMemConstPtr(pBlockHead);

    NN_UNUSED(isPrint);
    NN_UNUSED(heapType);

    if (pHeapHead)
    {
        if ( GetUIntPtr(pBlockHead) < GetUIntPtr(pHeapHead->pHeapStart)
            || GetUIntPtr(pMemBlock) > GetUIntPtr(pHeapHead->pHeapEnd))
        {
            NN_SDK_ASSERT(! isPrint, "[Exp Heap] Bad %s memory block address. - address %p, heap area [%p - %p)\n",
                heapType, pMemBlock, pHeapHead->pHeapStart, pHeapHead->pHeapEnd);
            return false;
        }
    }

    if (pBlockHead->signature != signature)    // シグネチャが異なる?
    {
        NN_SDK_ASSERT(! isPrint, "[Exp Heap] Bad %s memory block signature. - address %p, signature %04X\n",
            heapType, pMemBlock, pBlockHead->signature);
        return false;
    }

    if (pHeapHead)
    {
        if (GetUIntPtr(pMemBlock) + pBlockHead->blockSize > GetUIntPtr(pHeapHead->pHeapEnd))
        {
            NN_SDK_ASSERT(! isPrint, "[Exp Heap] wrong size %s memory block. - address %p, block size %zu\n",
                heapType, pMemBlock, pBlockHead->blockSize);
            return false;
        }
    }

    return true;
}

/**
 * @brief       使用メモリブロックのヘッダの内容が妥当であるかチェックします。
 * @param[in]   pBlockHead  チェックするメモリブロックのヘッダへのポインタ
 * @param[in]   pHeapHead   拡張ヒープのヘッダへのポインタ
 * @param[in]   flag        フラグ
 * @return      メモリブロックのヘッダの内容が妥当なら true、そうでないなら false を返します。
 */
inline bool CheckUsedMemoryBlock(ExpHeapMemoryBlockHead const* pBlockHead, HeapHead const* pHeapHead, uint32_t flag) NN_NOEXCEPT
{
    return CheckMemoryBlock(pBlockHead, pHeapHead, UsedBlockSignature, "used", flag);
}

/**
 * @brief       フリーメモリブロックのヘッダの内容が妥当であるかチェックします。
 * @param[in]   pBlockHead チェックするメモリブロックのヘッダへのポインタ
 * @param[in]   pHeapHead 拡張ヒープのヘッダへのポインタ
 * @param[in]   flag    フラグ
 * @return      メモリブロックのヘッダの内容が妥当なら true、そうでないなら false を返します。
 */
inline bool CheckFreeMemoryBlock(ExpHeapMemoryBlockHead const* pBlockHead, HeapHead const* pHeapHead, uint32_t flag) NN_NOEXCEPT
{
    return CheckMemoryBlock(pBlockHead, pHeapHead, FreeBlockSignature, "free", flag);
}

/**
 * @brief       メモリブロックの前へのリンクが正しいかチェックします。
 * @param[in]   pBlockHead  チェックするメモリブロックのヘッダへのポインタ
 * @param[in]   pPrevBlock  チェックするメモリブロックの前のメモリブロックのヘッダへのポインタ
 * @param[in]   flag        フラグ
 * @return      メモリブロックの前へのリンクが正しいなら true、そうでないなら false を返します。
 */
bool CheckMemoryBlockPrevPtr(const ExpHeapMemoryBlockHead* pBlockHead, const ExpHeapMemoryBlockHead* pPrevBlock, uint32_t flag) NN_NOEXCEPT
{
    const bool isPrint = 0 != (flag & ErrorOption_Print);

    NN_UNUSED(isPrint);

    if (pBlockHead->pPrevBlock != pPrevBlock)
    {
        NN_SDK_ASSERT(! isPrint, "[Exp Heap] Wrong link memory block. - address %p, previous address %p != %p\n",
            GetMemoryBlockMemConstPtr(pBlockHead), pBlockHead->pPrevBlock, pPrevBlock);
        return false;
    }

    return true;
}

/**
 * @brief       メモリブロックの次へのリンクが正しいかチェックします。
 * @param[in]   pBlockHead  チェックするメモリブロックのヘッダへのポインタ
 * @param[in]   pNextBlock  チェックするメモリブロックの次のメモリブロックのヘッダへのポインタ
 * @param[in]   flag        フラグ
 * @return      メモリブロックの次へのリンクが正しいなら true、そうでないなら false を返します。
 */
bool CheckMemoryBlockNextPtr(const ExpHeapMemoryBlockHead* pBlockHead, const ExpHeapMemoryBlockHead* pNextBlock, uint32_t flag) NN_NOEXCEPT
{
    const bool isPrint = 0 != (flag & ErrorOption_Print);

    NN_UNUSED(isPrint);

    if (pBlockHead->pNextBlock != pNextBlock)
    {
        NN_SDK_ASSERT(! isPrint, "[Exp Heap] Wrong link memory block. - address %p, next address %p != %p\n",
            GetMemoryBlockMemConstPtr(pBlockHead), pBlockHead->pNextBlock, pNextBlock);
        return false;
    }

    return true;
}

/**
 * @brief       メモリブロックのポインタがメモリブロックリストの先頭/末尾であるかチェックします。
 * @param[in]   pBlockHead      チェックするメモリブロックのヘッダへのポインタ
 * @param[in]   pBlockHeadTail  メモリブロックリストの先頭/末尾メモリブロックのヘッダへのポインタ
 * @param[in]   pHeadType       先頭か末尾を表す文字列
 * @param[in]   flag            フラグ
 * @return      メモリブロックのポインタがメモリブロックリストの先頭/末尾であるなら true、そうでないなら false を返します。
 */
bool CheckMemoryBlockLinkTail(const ExpHeapMemoryBlockHead* pBlockHead,
                              const ExpHeapMemoryBlockHead* pBlockHeadTail,
                              const char* heapType,
                              uint32_t flag) NN_NOEXCEPT
{
    const bool isPrint = 0 != (flag & ErrorOption_Print);

    NN_UNUSED(isPrint);
    NN_UNUSED(heapType);

    if (pBlockHead != pBlockHeadTail)
    {
        NN_SDK_ASSERT(! isPrint, "[Exp Heap] Wrong memory brock list %s pointer. - address %p, %s address %p != %p\n",
            heapType, GetMemoryBlockMemConstPtr(pBlockHead), heapType, pBlockHead, pBlockHeadTail);
        return false;
    }

    return true;
}

/**
 * @brief       使用メモリブロックの妥当性をチェックします。
 * @param[in]   pMemoryBlock    チェックするメモリブロック
 * @param[in]   heapHandle      メモリブロックを含有する拡張ヒープのハンドル
 * @return      指定したメモリブロックが問題なければ true を返します。@n
                問題があれば false を返します。
 * @details     heapHandle に NULLを指定すると、メモリブロックがヒープに含まれているかのチェックを行いません。
 */
bool IsValidUsedMemoryBlock(const void* pMemoryBlock, HeapHandle heapHandle) NN_NOEXCEPT
{
    HeapHead* pHeapHead = heapHandle;

    if (! pMemoryBlock)
    {
        return false;
    }

    return CheckUsedMemoryBlock(GetMemoryBlockHeadConstPtr(pMemoryBlock), pHeapHead, 0);
}

} // unnamed namespace

/* ========================================================================
    外部関数(公開)
   ======================================================================== */

/**
 * @brief 拡張ヒープを作成します。
 */
HeapHandle CreateExpHeap(void* pStartAddress, size_t size, int option) NN_NOEXCEPT
{
    void* pEndAddress;

    NN_SDK_ASSERT_NOT_NULL(pStartAddress);

    pEndAddress   = RoundDownPtr(AddSizeToPtr(pStartAddress, size), MinimumAlignment);
    pStartAddress = RoundUpPtr(pStartAddress, MinimumAlignment);

    if ( GetUIntPtr(pStartAddress) > GetUIntPtr(pEndAddress) ||
         GetOffsetFromPtr(pStartAddress, pEndAddress) < sizeof(ExpHeapMemoryBlockHead) + MinimumAlignment)
    {
        return NULL;
    }

    // Expヒープ向け初期化
    HeapHead* pHeapHead = InitHeap( pStartAddress, pEndAddress, option);
    return pHeapHead;  // ヒープヘッダへのポインタをそのままハンドル値とする
}

/**
 * @brief       拡張ヒープを破棄します。
 * @param[in]   heapHandle    拡張ヒープのハンドル
 */
void DestroyExpHeap(HeapHandle heapHandle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidHeapHandle(heapHandle));
    FinalizeHeap(heapHandle);
}

/**
 * @brief       拡張ヒープからメモリブロックを確保します。
 */
void* AllocFromExpHeap(HeapHandle heapHandle, size_t size, int alignment) NN_NOEXCEPT
{
    void* pMemory = NULL;

    NN_SDK_ASSERT(IsValidHeapHandle(heapHandle));

    // アライメントが 4 未満の 2 のべき乗の場合は、4 にする
    if(alignment == 1 || alignment == 2)
    {
        alignment = 4;
    }
    else if(alignment == -1 || alignment == -2)
    {
        alignment = -4;
    }

    // alignment のチェック
    NN_SDK_ASSERT(!(abs(alignment) & (abs(alignment) - 1)));
    NN_SDK_ASSERT(MinimumAlignment <= abs(alignment));

    if (size == 0)
    {
        size = 1;
    }

    size = RoundUp(size, MinimumAlignment);    // 実際に確保するサイズ

    if (alignment >= 0)     // 前から確保
    {
        pMemory = AllocFromHead(heapHandle, size, alignment);
    }
    else                    // 後ろから確保
    {
        pMemory = AllocFromTail(heapHandle, size, -alignment);
    }

    return pMemory;
}

/**
 * @brief       拡張ヒープから確保されたメモリブロックのサイズを変更します。
 */
size_t ResizeExpHeapMemoryBlock(HeapHandle heapHandle, void* pMemoryBlock, size_t size) NN_NOEXCEPT
{
    ExpHeapHead* pHead;
    ExpHeapMemoryBlockHead* pTargetBlock;

    NN_SDK_ASSERT(IsValidHeapHandle(heapHandle));
    NN_SDK_ASSERT(IsValidUsedMemoryBlock(pMemoryBlock, heapHandle));

    pHead = GetHeapHeadPtrFromHandle(heapHandle);
    pTargetBlock = GetMemoryBlockHeadPtr(pMemoryBlock);

    size = RoundUp(size, MinimumAlignment);
    if (size == pTargetBlock->blockSize)  // ブロックサイズ変更なしの場合
    {
        return size;
    }

    // 新しいエリアが拡大するとき
    if (size > pTargetBlock->blockSize)
    {
        void* currentUsedEnd = GetMemoryBlockEndAddr(pTargetBlock);   // 現使用ブロックのend address
        ExpHeapMemoryBlockHead* pBlockHead;

        // 次のフリーブロックを探す
        for (pBlockHead = pHead->freeList.pHead; pBlockHead; pBlockHead = pBlockHead->pNextBlock)
        {
            if (pBlockHead == currentUsedEnd)
            {
                break;
            }
        }

        // 次のフリーブロックがないか、サイズが足りない場合
        if (! pBlockHead || size > pTargetBlock->blockSize + sizeof(ExpHeapMemoryBlockHead) + pBlockHead->blockSize)
        {
            return 0;
        }

        {
            MemoryRegion regionNewFree;
            void* oldFreeStart;
            ExpHeapMemoryBlockHead* nextBlockPrev;

            // フリーブロックのリージョンを取得し、フリーブロックを一旦外す
            GetRegionOfMemoryBlock(&regionNewFree, pBlockHead);
            nextBlockPrev = RemoveMemoryBlock(&pHead->freeList, pBlockHead);

            oldFreeStart = regionNewFree.pStart;
            regionNewFree.pStart = AddSizeToPtr(pMemoryBlock, size); // 新しくフリーとなるリージョン位置

            // 余りがメモリブロックサイズ未満なら
            if (GetOffsetFromPtr(regionNewFree.pStart, regionNewFree.pEnd) < sizeof(ExpHeapMemoryBlockHead))
            {
                regionNewFree.pStart = regionNewFree.pEnd;  // 使用ブロックに吸収
            }

            pTargetBlock->blockSize = GetOffsetFromPtr(pMemoryBlock, regionNewFree.pStart);  // 対象ブロックサイズ変更

            // 余りがメモリブロックサイズ以上なら
            if (GetOffsetFromPtr(regionNewFree.pStart, regionNewFree.pEnd) >= sizeof(ExpHeapMemoryBlockHead))
            {
                InsertMemoryBlock(&pHead->freeList, InitFreeMemoryBlock(&regionNewFree), nextBlockPrev);   // 新しくフリーブロックを作る
            }

#if defined(NN_DETAIL_HEAP_DEBUG)
            // 拡張した部分フィル
            FillAllocMemory(heapHandle, oldFreeStart, GetOffsetFromPtr(oldFreeStart, regionNewFree.pStart));
#endif

        }
    }
    // 新しいエリアが縮小するとき
    else
    {
        MemoryRegion regionNewFree;
        const size_t oldBlockSize = pTargetBlock->blockSize;

        regionNewFree.pStart = AddSizeToPtr(pMemoryBlock, size); // 新しくフリーとなるリージョン位置
        regionNewFree.pEnd   = GetMemoryBlockEndAddr(pTargetBlock);   // 現使用ブロックのend address

        pTargetBlock->blockSize = size;  // 対象ブロックサイズ変更

        if (! RecycleRegion(pHead, &regionNewFree))    // フリーリストに返してみる
        {
            pTargetBlock->blockSize = oldBlockSize;  // 失敗したら、元に戻す
        }
    }

    return pTargetBlock->blockSize;
}

/**
 * @brief       拡張ヒープへメモリブロックを返却します。
 */
void FreeToExpHeap(HeapHandle heapHandle, void* pMemoryBlock) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidHeapHandle(heapHandle));

    {
        HeapHead* pHeapHead = heapHandle;
        ExpHeapHead* pExpHeapHead = GetHeapHeadPtrFromHandle(pHeapHead);
        ExpHeapMemoryBlockHead* pBlock = GetMemoryBlockHeadPtr(pMemoryBlock);
        MemoryRegion region;

        // このヒープの中に入っているか
        NN_SDK_ASSERT(pHeapHead->pHeapStart <= pMemoryBlock && pMemoryBlock < pHeapHead->pHeapEnd);

        GetRegionOfMemoryBlock(&region, pBlock);
        RemoveMemoryBlock(&pExpHeapHead->usedList, pBlock);   // 使用リストからはずす
        RecycleRegion(pExpHeapHead, &region);   // 指定アドレスから指定サイズをフリーに加える
    }
}

/**
 * @brief       拡張ヒープの空き領域のサイズの合計を取得します。
 */
size_t GetExpHeapTotalFreeSize( HeapHandle heapHandle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidHeapHandle(heapHandle));

    {
        size_t sumSize = 0;
        ExpHeapHead const* pExpHeapHead = GetHeapHeadPtrFromHandle(heapHandle);
        ExpHeapMemoryBlockHead const* pBlock;

        for(pBlock = pExpHeapHead->freeList.pHead; pBlock; pBlock = pBlock->pNextBlock)
        {
            sumSize += pBlock->blockSize;
        }

        return sumSize;
    }
}

/**
 * @brief       拡張ヒープ内の割り当て可能な最大サイズを取得します。
 */
size_t GetExpHeapAllocatableSize(HeapHandle heapHandle, int alignment) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidHeapHandle(heapHandle));

    // alignment のチェック
    NN_SDK_ASSERT(!(abs(alignment) & (abs(alignment) - 1)));
    NN_SDK_ASSERT(MinimumAlignment <= abs(alignment));

    alignment = abs(alignment); // 念のため正数化

    {
        ExpHeapHead const* pExpHeapHead = GetHeapHeadPtrFromHandle(heapHandle);
        size_t maxSize = 0;
        size_t offsetMin = SIZE_MAX;
        ExpHeapMemoryBlockHead* pBockHead;

        for (pBockHead = pExpHeapHead->freeList.pHead; pBockHead; pBockHead = pBockHead->pNextBlock)
        {
            // アライメントを考慮したメモリブロック位置
            void* baseAddress = RoundUpPtr(GetMemoryBlockMemPtr(pBockHead), alignment);

            if (GetUIntPtr(baseAddress) < GetUIntPtr(GetMemoryBlockEndAddr(pBockHead)))
            {
                const size_t blockSize = GetOffsetFromPtr(baseAddress, GetMemoryBlockEndAddr(pBockHead));
                // アライメントによる空きエリア
                const size_t offset = GetOffsetFromPtr(GetMemoryBlockMemPtr(pBockHead), baseAddress);

                // サイズが大きい場合、あるいはサイズが同じでも無駄な空間がより小さい場合は、
                // メモリブロックを更新
                if ( maxSize < blockSize ||  (maxSize == blockSize && offsetMin > offset) )
                {
                    maxSize = blockSize;
                    offsetMin= offset;
                }
            }
        }

        return maxSize;
    }
}

/**
 * @brief       拡張ヒープのメモリ確保モードをセットします。
 */
uint16_t SetExpHeapAllocMode(HeapHandle heapHandle, uint16_t mode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidHeapHandle(heapHandle));

    {
        ExpHeapHead *const pExpHeapHead = GetHeapHeadPtrFromHandle(heapHandle);
        uint16_t oldMode = GetAllocMode(pExpHeapHead);
        SetAllocMode(pExpHeapHead, mode);

        return oldMode;
    }
}

/**
 * @brief       拡張ヒープのメモリ確保モードを取得します。
 */
uint16_t GetExpHeapAllocMode(HeapHandle heapHandle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidHeapHandle(heapHandle));
    return GetAllocMode(GetHeapHeadPtrFromHandle(heapHandle));
}

/**
 * @brief       アライメントの際に発生する隙間の領域を再利用するかどうかを設定します。
 */
bool SetUseExpHeapMarginOfAlignment(HeapHandle heapHandle, bool isReuse) NN_NOEXCEPT
{
    ExpHeapHead *const pExpHeapHead = GetHeapHeadPtrFromHandle(heapHandle);
    bool isReuseBefore;

    NN_SDK_ASSERT(IsValidHeapHandle(heapHandle));

    isReuseBefore = pExpHeapHead->isReuse;
    pExpHeapHead->isReuse = isReuse;

    return isReuseBefore;
}

/**
 * @brief       アライメントの際に発生する隙間の領域を再利用するかどうかの値を返します。
 */
bool GetUseExpHeapMarginOfAlignment(HeapHandle heapHandle) NN_NOEXCEPT
{
    ExpHeapHead const* pExpHeapHead = GetHeapHeadPtrFromHandle(heapHandle);

    NN_SDK_ASSERT(IsValidHeapHandle(heapHandle));

    return pExpHeapHead->isReuse;
}


/**
 * @brief       拡張ヒープのグループIDをセットします。
 */
uint16_t SetExpHeapGroupId(HeapHandle heapHandle, uint16_t groupId) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidHeapHandle(heapHandle));
    NN_SDK_ASSERT(groupId <= MaxGroupId);

    {
        ExpHeapHead* pExpHeapHead = GetHeapHeadPtrFromHandle(heapHandle);
        uint16_t oldGroupId = pExpHeapHead->groupId;
        pExpHeapHead->groupId = groupId;

        return oldGroupId;
    }
}

/**
 * @brief       拡張ヒープのグループIDを取得します。
 */
uint16_t GetExpHeapGroupId(HeapHandle heapHandle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidHeapHandle(heapHandle));
    return GetHeapHeadPtrFromHandle(heapHandle)->groupId;
}

/**
 * @brief       拡張ヒープから確保されたメモリブロックのサイズを取得します。
 */
size_t GetExpHeapMemoryBlockSize(const void* pMemBlock) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidUsedMemoryBlock(pMemBlock, NULL));
    return GetMemoryBlockHeadConstPtr(pMemBlock)->blockSize;
}

/**
 * @brief       拡張ヒープから確保されたメモリブロックのグループIDを取得します。
 */
uint16_t GetMemoryBlockHeapGroupId(const void* pMemBlock) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidUsedMemoryBlock(pMemBlock, NULL));
    return GetMemoryBlockGroupId(GetMemoryBlockHeadConstPtr(pMemBlock));
}

/**
 * @brief       拡張ヒープから確保されたメモリブロックの確保方向を取得します。
 */
uint16_t GetMemoryBlockHeapAllocDir(const void* pMemBlock) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidUsedMemoryBlock(pMemBlock, NULL));
    return GetMemoryBlockAllocDir(GetMemoryBlockHeadConstPtr(pMemBlock));
}

/**
 * @brief       拡張ヒープの空き領域を解放し、拡張ヒープが使用するメモリ領域を縮小します。
 */
MemoryRange AdjustExpHeap(HeapHandle heapHandle) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValidHeapHandle(heapHandle) );

    HeapHead* pHeapHead = heapHandle;
    ExpHeapHead* pExpHeapHead = GetHeapHeadPtrFromHeapHead(pHeapHead);
    ExpHeapMemoryBlockHead* pBlockHead  = pExpHeapHead->freeList.pTail;

    // 空き領域が無い場合は失敗
    if ( pBlockHead == NULL )
    {
        return MakeMemoryRange(heapHandle->pHeapEnd, 0);
    }

    void* const pBlock      = GetMemoryBlockMemPtr( pBlockHead );
    void* const pBlockEnd   = AddSizeToPtr( pBlock, pBlockHead->blockSize );

    // 末尾から確保されたメモリブロックが存在する場合は失敗
    if ( pBlockEnd != heapHandle->pHeapEnd )
    {
        return MakeMemoryRange(heapHandle->pHeapEnd, 0);
    }

    // 解放されたフリーブロックをフリーリストから削除
    RemoveMemoryBlock( &pExpHeapHead->freeList, pBlockHead );

    size_t blockSize = pBlockHead->blockSize + sizeof( ExpHeapMemoryBlockHead );
    pHeapHead->pHeapEnd = SubSizeToPtr( pHeapHead->pHeapEnd, blockSize );

    return MakeMemoryRange(pHeapHead->pHeapEnd, blockSize);

}

/**
 * @brief 拡張ヒープが破壊されていないかどうかをチェックします。(デバッグ用関数)
 */
bool CheckExpHeap(HeapHandle heapHandle, uint32_t option) NN_NOEXCEPT
{
    const bool isPrint = 0 != (option & ErrorOption_Print);
    size_t totalBytes  = 0;

    NN_UNUSED(isPrint);

    if (! IsValidHeapHandle(heapHandle))
    {
        NN_SDK_ASSERT(! isPrint, "[Exp Heap] Invalid heap handle. - %p\n", heapHandle);
        return false;
    }

    {
        HeapHead const* pHeapHead = heapHandle;
        ExpHeapHead const* pExpHeapHead = GetHeapHeadPtrFromHeapHead(pHeapHead);
        ExpHeapMemoryBlockHead* pBlockHead = NULL;
        ExpHeapMemoryBlockHead* pPrevBlock = NULL;

        // 使用ブロックのチェック
        for (pBlockHead = pExpHeapHead->usedList.pHead; pBlockHead; pPrevBlock = pBlockHead, pBlockHead = pBlockHead->pNextBlock)
        {
            if ( ! CheckUsedMemoryBlock(pBlockHead, pHeapHead, option)
                || ! CheckMemoryBlockPrevPtr(pBlockHead, pPrevBlock, option)   // 前ブロックへのポインタが1ループ前のメモリブロックのポインタと同じでない?
            )
            {
                return false;
            }

            // 占有サイズを積算
            totalBytes += sizeof(ExpHeapMemoryBlockHead) + pBlockHead->blockSize + GetMemoryBlockPaddingAlignment(pBlockHead);
        }

        if (! CheckMemoryBlockLinkTail(pPrevBlock, pExpHeapHead->usedList.pTail, "tail", option))  // 末尾ブロックが最後のブロックへのポインタを指しているか?
        {
            return false;
        }

        // フリーブロックのチェック
        pBlockHead = NULL;
        pPrevBlock = NULL;
        for (pBlockHead = pExpHeapHead->freeList.pHead; pBlockHead; pPrevBlock = pBlockHead, pBlockHead = pBlockHead->pNextBlock)
        {
            if ( ! CheckFreeMemoryBlock(pBlockHead, pHeapHead, option)
                || ! CheckMemoryBlockPrevPtr(pBlockHead, pPrevBlock, option)   // 前ブロックへのポインタが1ループ前のメモリブロックのポインタと同じでない?
            )
            {
                return false;
            }

            // 占有サイズを積算
            totalBytes += sizeof(ExpHeapMemoryBlockHead) + pBlockHead->blockSize;
        }

        if (! CheckMemoryBlockLinkTail(pPrevBlock, pExpHeapHead->freeList.pTail, "tail", option))  // 末尾ブロックが最後のブロックへのポインタを指しているか?
        {
            return false;
        }

        // 全体の結果表示
        if (totalBytes != GetOffsetFromPtr(pHeapHead->pHeapStart, pHeapHead->pHeapEnd))
        {
            NN_SDK_ASSERT(! isPrint, "[Exp Heap] Incorrect total memory block size. - heap size %p, sum size %zu\n",
                GetOffsetFromPtr(pHeapHead->pHeapStart, pHeapHead->pHeapEnd), totalBytes);
            return false;
        }

        return true;
    }
}

/**
 * @brief       拡張ヒープのメモリブロックが破壊されていないかどうかをチェックします。(デバッグ用関数)
 */
bool CheckExpHeapMemoryBlock(const void* pMemoryBlock, HeapHandle heapHandle, uint32_t option) NN_NOEXCEPT
{
    const bool isPrint = 0 != (option & ErrorOption_Print);
    const ExpHeapMemoryBlockHead* pBlockHead = NULL;
    HeapHead const* pHeapHead = heapHandle;

    NN_UNUSED(isPrint);

    if (! pMemoryBlock)
    {
        NN_SDK_ASSERT(! isPrint, "[Exp Heap] Invalid block. - %p\n", pMemoryBlock);
        return false;
    }

    pBlockHead = GetMemoryBlockHeadConstPtr(pMemoryBlock);

    if (! CheckUsedMemoryBlock(pBlockHead, pHeapHead, option))
    {
        return false;
    }

    if (pBlockHead->pPrevBlock)
    {
        if ( ! CheckUsedMemoryBlock(pBlockHead->pPrevBlock, pHeapHead, option)     // 前ブロックのシグネチャとサイズのチェック
            || ! CheckMemoryBlockNextPtr(pBlockHead->pPrevBlock, pBlockHead, option)  // 前ブロックの次ブロックへのポインタが自分を指しているか?
        )
        {
            return false;
        }
    }
    else
    {
        if (pHeapHead)
        {
            // prevが NULL のときは、リストのpHeadポインタは自分を指しているはず
            if (! CheckMemoryBlockLinkTail(pBlockHead, GetHeapHeadPtrFromHeapHead(pHeapHead)->usedList.pHead, "head", option))
            {
                return false;
            }
        }
    }

    if (pBlockHead->pNextBlock)
    {
        if ( ! CheckUsedMemoryBlock(pBlockHead->pNextBlock, pHeapHead, option)     // 次ブロックのシグネチャとサイズのチェック
            || ! CheckMemoryBlockPrevPtr(pBlockHead->pNextBlock, pBlockHead, option)  // 次ブロックの前ブロックへのポインタが自分を指しているか?
        )
        {
            return false;
        }
    }
    else
    {
        if (pHeapHead)
        {
            // nextが NULL のときは、リストのtailポインタは自分を指しているはず
            if (! CheckMemoryBlockLinkTail(pBlockHead, GetHeapHeadPtrFromHeapHead(pHeapHead)->usedList.pTail, "tail", option))
            {
                return false;
            }
        }
    }

    return true;
}

/**
 * @brief       拡張ヒープから割り当てられたメモリブロック全てに対して、ユーザが指定した関数を呼ばせます。(デバッグ用関数)
 * @param[in]   heapHandle        拡張ヒープのハンドル
 * @param[in]   visitor     各メモリブロックに対して呼ばせる関数。
 * @param[in]   userParam   visitor関数に渡すユーザ指定のパラメータ
 * @details     visitor関数で呼ばれるメモリブロックの順番は、確保した順番になります。@n
                visitor の型 HeapVisitor は次のように定義されています。@n
                typedef void (*HeapVisitor)(void* pMemBlock, HeapHandle heapHandle, uintptr_t userParam);@n
                @li pMemBlock   メモリブロックへのポインタ
                @li heapHandle  メモリブロックを含有するヒープ
                @li userParam   ユーザー用パラメータ
 */
void VisitExpHeapAllocated(HeapHandle heapHandle, HeapVisitor visitor, uintptr_t userParam) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsValidHeapHandle(heapHandle));
    NN_SDK_ASSERT_NOT_NULL(visitor);

    {
        ExpHeapMemoryBlockHead* pBlock = GetHeapHeadPtrFromHandle(heapHandle)->usedList.pHead;

        while (pBlock)
        {
            ExpHeapMemoryBlockHead* pNextBlock = pBlock->pNextBlock;
            (*visitor)(GetMemoryBlockMemPtr(pBlock), heapHandle, userParam);
            pBlock = pNextBlock;
        }
    }
}

/**
 * @brief       拡張ヒープ内部の情報を表示します。
 */
void DumpExpHeap(HeapHandle heapHandle) NN_NOEXCEPT
{

    NN_SDK_ASSERT(IsValidHeapHandle(heapHandle));

    size_t  usedSize = 0;
    size_t  usedCount = 0;
    size_t  freeCount = 0;

    HeapHead* pHeapHead = heapHandle;
    ExpHeapHead const* pExpHeapHead = GetHeapHeadPtrFromHandle(pHeapHead);

    DumpHeapHead(pHeapHead);

    NN_SDK_LOG("     attr          address:           size    gid aln           prev_ptr         next_ptr\n");   // ヘッダー行

    // ---------------- UsedBlock のダンプ ----------------
    NN_SDK_LOG("    (Used Blocks)\n" );

    if (pExpHeapHead->usedList.pHead == NULL)
    {
        NN_SDK_LOG("     NONE\n");
    }
    else
    {
        ExpHeapMemoryBlockHead* pBlock;

        for (pBlock = pExpHeapHead->usedList.pHead; pBlock; pBlock = pBlock->pNextBlock)
        {
            if (pBlock->signature != UsedBlockSignature)
            {
                NN_SDK_LOG("    xxxxx %16p: ----------------  --- ---  (-------- --------)\nabort\n", pBlock);
                break;
            }

            NN_SDK_LOG("    %s %16p: %16zu  %3d %3d  (%16p %16p)\n",
                        GetMemoryBlockAllocDir(pBlock) == AllocationDirection_Rear ? " rear" : "front",
                        GetMemoryBlockMemPtr(pBlock),
                        pBlock->blockSize,
                        GetMemoryBlockGroupId( pBlock ),
                        GetMemoryBlockPaddingAlignment( pBlock ),
                        pBlock->pPrevBlock ? GetMemoryBlockMemPtr(pBlock->pPrevBlock): NULL,
                        pBlock->pNextBlock ? GetMemoryBlockMemPtr(pBlock->pNextBlock): NULL
                );

            // ---- 使用量
            usedSize += sizeof(ExpHeapMemoryBlockHead) + pBlock->blockSize + GetMemoryBlockPaddingAlignment(pBlock);

            usedCount ++;
        }
    }

    // ---------------- FreeBlock のダンプ ----------------
    NN_SDK_LOG("    (Free Blocks)\n" );

    if (pExpHeapHead->freeList.pHead == NULL)
    {
        NN_SDK_LOG("     NONE\n" );
    }
    else
    {
        ExpHeapMemoryBlockHead* pBlock;

        for (pBlock = pExpHeapHead->freeList.pHead; pBlock; pBlock = pBlock->pNextBlock)
        {
            if (pBlock->signature != FreeBlockSignature)
            {
                NN_SDK_LOG("    xxxxx %16p: ----------------  --- ---  (-------- --------)\nabort\n", pBlock);
                break;
            }

            NN_SDK_LOG("    %s %16p: %16zu  %3d %3d  (%16p %16p)\n",
                        " free",
                        GetMemoryBlockMemPtr(pBlock),
                        pBlock->blockSize,
                        GetMemoryBlockGroupId( pBlock ),
                        GetMemoryBlockPaddingAlignment( pBlock ),
                        pBlock->pPrevBlock ? GetMemoryBlockMemPtr(pBlock->pPrevBlock): NULL,
                        pBlock->pNextBlock ? GetMemoryBlockMemPtr(pBlock->pNextBlock): NULL
                );

            freeCount ++;
        }
    }

    NN_SDK_LOG("\n");

    {
        size_t heapSize  = GetOffsetFromPtr(pHeapHead->pHeapStart, pHeapHead->pHeapEnd); // ヒープサイズ(データ領域のサイズ)
        NN_UNUSED(heapSize);
        NN_SDK_LOG("    %zu / %zu bytes (%d%%) used (U:%zu F:%zu)\n",
                    usedSize, heapSize, static_cast<int>(static_cast<uint64_t>(usedSize) * 100 / heapSize), usedCount, freeCount);
    }

    NN_SDK_LOG("\n");
}

}}}
