﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief   ヒープ共通処理の内部実装です。
 * @details ヒープ共通で利用される inline 関数が定義されています。
 */

#include <nn/lmem/lmem_Common.h>
#include "lmem_DetailCommonHeap.h"

namespace nn { namespace lmem { namespace detail {

/**
 * @brief       指定した境界値に合わせるため、上位方向に切り上げます。
 * @param[in]   value       対象となるデータ
 * @param[in]   alignment   境界値
 * @return      指定した境界に切り上げられた値を返します。
 */
template <typename T, typename U>
inline T RoundUp(T value, U alignment) NN_NOEXCEPT
{
    return (((value) + (alignment - 1)) & ~(alignment - 1));
}

/**
 * @brief void ポインタを uintptr_t にキャストします。
 */
inline uintptr_t GetUIntPtr(const void* ptr) NN_NOEXCEPT
{
    return reinterpret_cast<uintptr_t>(const_cast<void*>(ptr));
}

/**
 * @brief 指定した境界値に合わせるため、上位方向に切り上げます。
 */
template <typename T>
inline void* RoundUpPtr(void *ptr, T alignment) NN_NOEXCEPT
{
    return reinterpret_cast<void*>(RoundUp(GetUIntPtr(ptr), alignment));
}

/**
 * @brief       指定した境界値に合わせるため、下位方向に切り下げます。
 * @param[in]  value       対象となるデータ
 * @param[in]  alignment   境界値
 * @return      指定した境界に切り下げられた値を返します。
 */
template <typename T, typename U>
inline T RoundDown(T value, U alignment) NN_NOEXCEPT
{
    return ((value) & ~(alignment - 1));
}

/**
 * @brief 指定した境界値に合わせるため、下位方向に切り下げます。
 */
template <typename T>
inline void* RoundDownPtr(void* ptr, T alignment) NN_NOEXCEPT
{
    return reinterpret_cast<void*>(RoundDown(GetUIntPtr(ptr), alignment));
}

/**
 * @brief       特定のビット位置の値を取得します。
 * @param[in]   data    取得するビットデータを含むデータ
 * @param[in]   start   開始ビット(0から始まる)
 * @param[in]   bits    ビット数
 * @return      start から bits 個の値を返します。
 */
template <typename T>
inline T GetBitValue(T data, int start, int bits) NN_NOEXCEPT
{
    return ((data) >>(start)) & ((static_cast<T>(1) <<(bits)) - static_cast<T>(1));
}

/**
 * @brief       特定のビット位置に値をセットします。
 * @param[in]   pData   セットするビットデータを格納する変数
 * @param[in]   start   開始ビット(0から始まる)
 * @param[in]   bits    ビット数
 * @param[in]   value   セットするビットデータ
 * @details     start から bits 個の値を value に置き換えます。
 */
template <typename T, typename U>
inline void SetBitValue(T* pData, int start, int bits, U value) NN_NOEXCEPT
{
    const T maskBits = (static_cast<T>(1) <<(bits)) - static_cast<T>(1);
    const T newVal = static_cast<T>(value) & maskBits;      // 安全のためマスク
    *pData &= ~maskBits << start;                           // セットする領域をクリア
    *pData |= newVal << start;
}

/**
 * @brief   ポインタ同士のアドレスの差を求め、サイズとして返します。
 * @pre     pStart < pEnd
 */
inline size_t GetOffsetFromPtr(const void* pStart, const void* pEnd) NN_NOEXCEPT
{
    NN_SDK_ASSERT(GetUIntPtr(pStart) <= GetUIntPtr(pEnd));
    return GetUIntPtr(pEnd) - GetUIntPtr(pStart);
}

/**
 * @brief ptr から val ぶん加算した先のポインタを返します。
 */
inline void* AddSizeToPtr(void* ptr, size_t val) NN_NOEXCEPT
{
    return reinterpret_cast<void*>( GetUIntPtr(ptr) + val );
}

/**
 * @brief ptr から val ぶん減算した先のポインタを返します。
 */
inline void* SubSizeToPtr(void* ptr, size_t val) NN_NOEXCEPT
{
    return reinterpret_cast<void*>( GetUIntPtr(ptr) - val );
}

/**
 * @brief   ポインタアドレスを比較し、それに応じた値を返します。
 * @return  比較結果から、0、 1、 -1 のどれかを返します。
 * @retval  0   2つのアドレスは同じ
 * @retval  1   a の方が大きい
 * @retval  -1  b の方が大きい
 */
inline int CompareHigherPtr( const void* a, const void* b )
{
    const uintptr_t wa = reinterpret_cast<uintptr_t>(a);
    const uintptr_t wb = reinterpret_cast<uintptr_t>(b);

    if (wa == wb)
    {
        return 0;
    }
    if (wa < wb)
    {
        return -1;
    }
    return 1;
}

/**
 * @brief HeapHead 構造体の attribute のオプションフラグを取得します。
 */
inline Bit8 GetHeapOpt(const HeapHead* pHeapHead) NN_NOEXCEPT
{
    return static_cast<Bit8>(GetBitValue(pHeapHead->attribute, 0, 8));
}

/**
 * @brief HeapHead 構造体の attribute のオプションフラグをセットします。
 */
inline void SetHeapOpt(HeapHead* pHeapHead, int optFlag) NN_NOEXCEPT
{
    SetBitValue(&pHeapHead->attribute, 0, 8, optFlag);
}

/**
 * @brief       メモリブロックを指定の値でフィルします。
 * @param[out]  pOutMemory  セット先のメモリブロック
 * @param[in]   data        セットするデータ
 * @param[in]   size        セットする値のサイズ
 * @detail      メモリフィルは Debug ビルドおよび Develop ビルドでのみ機能します。
 */
inline void FillMemory(void* pOutMemory, uint32_t data, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT((size % sizeof(uint32_t)) == 0);     // size は4バイトアライメントされてるはず
    uint32_t* buffer = reinterpret_cast<uint32_t*>(pOutMemory);
    NN_SDK_ASSERT((reinterpret_cast<uintptr_t>(buffer) % sizeof(uint32_t)) == 0);     // dest は4バイトアライメントされてるはず
    for(int i = 0; static_cast<size_t>(i) < size / sizeof(data); ++i)
    {
        buffer[i] = data;
    }
}

/**
 * @brief   確保時のメモリにフィルします。
 * @detail  メモリフィルは Debug ビルドおよび Develop ビルドでのみ機能します。
 */
inline void FillAllocMemory(HeapHead* pHeapHead, void* address, size_t size) NN_NOEXCEPT
{

    if (GetHeapOpt(pHeapHead) & CreationOption_ZeroClear)
    {
        FillMemory(address, 0, size);
    }
    else
    {
        if (GetHeapOpt(pHeapHead) & CreationOption_DebugFill)
        {
            FillMemory(address, GetHeapFillVal(FillType_Allocate), size);
        }
    }
}

/**
 * @brief   解放時のメモリにフィルします。
 * @detail  メモリフィルは Debug ビルドおよび Develop ビルドでのみ機能します。
 */
inline void FillFreeMemory(HeapHead* pHeapHead, void* address, size_t size) NN_NOEXCEPT
{
    if (GetHeapOpt(pHeapHead) & CreationOption_DebugFill)
    {
        FillMemory(address, GetHeapFillVal(FillType_Free), size);
    }
}

/**
 * @brief   ヒープ作成時のメモリにフィルします。
 * @detail  メモリフィルは Debug ビルドおよび Develop ビルドでのみ機能します。
 */
inline void FillUnallocatedMemory(HeapHead* pHeapHead, void* address, size_t size) NN_NOEXCEPT
{
    if (GetHeapOpt(pHeapHead) & CreationOption_DebugFill)
    {
        FillMemory(address, GetHeapFillVal(FillType_Unallocated), size);
    }
}

}}}
