﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/kvdb/kvdb_Result.h>
#include <nn/kvdb/kvdb_KeyValueArchive.h>

namespace nn { namespace kvdb {

namespace
{
    struct KeyValueArchiveHeader
    {
        char        signature[4];
        uint16_t    formatVersion;
        char        reserved[2];
        uint32_t    entryCount;

        bool IsSignatureValid()
        {
            return std::memcmp(signature, GetSignatureString(), sizeof(signature)) == 0;
        }

        static const char* GetSignatureString()
        {
            return "IMKV";
        }

        static KeyValueArchiveHeader Make()
        {
            KeyValueArchiveHeader header = {};
            std::memcpy(header.signature, GetSignatureString(), sizeof(header.signature));

            return header;
        }
    };

    struct KeyValueArchiveEntryHeader
    {
        char        signature[4];
        uint32_t    keySize;
        uint32_t    valueSize;

        bool IsSignatureValid()
        {
            return std::memcmp(signature, GetSignatureString(), sizeof(signature)) == 0;
        }

        static const char* GetSignatureString()
        {
            return "IMEN";
        }

        static KeyValueArchiveEntryHeader Make()
        {
            KeyValueArchiveEntryHeader header = {};
            std::memcpy(header.signature, GetSignatureString(), sizeof(header.signature));

            return header;
        }
    };
}

    KeyValueArchiveSizeCalculator::KeyValueArchiveSizeCalculator() NN_NOEXCEPT : m_Size(sizeof(KeyValueArchiveHeader)){}

    void KeyValueArchiveSizeCalculator::AddEntry(size_t keySize, size_t valueSize) NN_NOEXCEPT
    {
        m_Size += sizeof(KeyValueArchiveEntryHeader) + keySize + valueSize;
    }

    void KeyValueArchiveBufferWriter::WriteHeader(uint32_t entryCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Position, 0u);

        auto header = KeyValueArchiveHeader::Make();
        header.entryCount = entryCount;

        Write(&header, sizeof(header));
    }

    void KeyValueArchiveBufferWriter::WriteEntry(const void* keyData, size_t keySize, const void* valueData, size_t valueSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_Position, 0u);

        auto header = KeyValueArchiveEntryHeader::Make();
        header.keySize = static_cast<uint32_t>(keySize);
        header.valueSize = static_cast<uint32_t>(valueSize);

        Write(&header, sizeof(header));
        Write(keyData, keySize);
        Write(valueData, valueSize);
    }

    void KeyValueArchiveBufferWriter::Write(const void* data, size_t size) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_Position + size <= m_Size && m_Position + size > m_Position);

        std::memcpy(&m_Buffer[m_Position], data, size);

        m_Position += size;
    }

    Result KeyValueArchiveBufferReader::ReadEntryCount(uint32_t* outValue) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_EQUAL(m_Position, 0u);

        KeyValueArchiveHeader header;
        Read(&header, sizeof(header));
        NN_RESULT_THROW_UNLESS(header.IsSignatureValid(), ResultInvalidKeyValueFormat());

        *outValue = header.entryCount;

        NN_RESULT_SUCCESS;
    }

    Result KeyValueArchiveBufferReader::GetKeyValueSize(KeyValueSize* outValue) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_Position, 0u);

        KeyValueArchiveEntryHeader header;
        NN_RESULT_DO(Peak(&header, sizeof(header)));
        NN_RESULT_THROW_UNLESS(header.IsSignatureValid(), ResultInvalidKeyValueFormat());

        KeyValueSize size = { header.keySize, header.valueSize };
        *outValue = size;

        NN_RESULT_SUCCESS;
    }

    Result KeyValueArchiveBufferReader::ReadKeyValue(void* keyData, size_t keySize, void* valueData, size_t valueSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_Position, 0u);

        KeyValueArchiveEntryHeader header;
        NN_RESULT_DO(Read(&header, sizeof(header)));
        NN_RESULT_THROW_UNLESS(header.IsSignatureValid(), ResultInvalidKeyValueFormat());
        NN_SDK_ASSERT_EQUAL(header.keySize, keySize);
        NN_SDK_ASSERT_EQUAL(header.valueSize, valueSize);

        Read(keyData, keySize);
        Read(valueData, valueSize);

        NN_RESULT_SUCCESS;
    }

    Result KeyValueArchiveBufferReader::Peak(void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_Position + size <= m_Size && m_Position + size > m_Position, ResultInvalidKeyValueFormat());

        std::memcpy(buffer, &m_Buffer[m_Position], size);

        NN_RESULT_SUCCESS;
    }

    Result KeyValueArchiveBufferReader::Read(void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(Peak(buffer, size));

        m_Position += size;

        NN_RESULT_SUCCESS;
    }

}}
