﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <memory>
#include <mutex>

#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/kvdb/kvdb_Result.h>
#include <nn/kvdb/kvdb_FileKeyValueStore.h>
#include <nn/kvdb/detail/kvdb_Log.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace kvdb {

    namespace {
        const char Extension[] = ".val";
        const size_t ExtensionLength = sizeof(Extension) - 1;
        const size_t MaxFileNameLength = 255;  // FAT32
        const size_t MaxBodyNameLength = MaxFileNameLength - ExtensionLength;
        const size_t MaxKeySize = MaxBodyNameLength / 2;

        typedef BoundedString<255> FileName;

        FileKeyValueStore::Path MakeKeyPath(const void* key, size_t keySize, const char* directoryPath) NN_NOEXCEPT
        {
            FileKeyValueStore::Path path(directoryPath);
            path.Append("/");

            auto keyPtr = static_cast<const Bit8*>(key);
            for (size_t i = 0; i < keySize; i++)
            {
                path.AppendFormat("%02x", keyPtr[i]);
            }
            path.Append(Extension);

            return path;
        }

        Result FileNameToKey(size_t* outValue, void* buffer, size_t bufferSize, const FileName& fileName) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(fileName.GetLength() >= ExtensionLength + 2, ResultInvalidKeyValueFormat());
            size_t keyBodyLength = fileName.GetLength() - ExtensionLength;
            NN_RESULT_THROW_UNLESS(keyBodyLength % 2 == 0 && fileName.EqualsPostfix(Extension), ResultInvalidKeyValueFormat());

            size_t keySize = (keyBodyLength) / 2;
            NN_RESULT_THROW_UNLESS(keySize <= bufferSize, ResultBufferNotEnough());

            auto ptr = static_cast<Bit8*>(buffer);
            for (size_t i = 0; i < keySize; i++)
            {
                char byteStr[3];
                fileName.GetSubString(byteStr, sizeof(byteStr), i * 2, 2);
                ptr[i] = static_cast<Bit8>(std::strtol(byteStr, nullptr, 16));
            }

            *outValue = keySize;

            NN_RESULT_SUCCESS;
        }
    }

    Result SimpleKeyValueCache::Initialize(void* buffer, size_t bufferSize, int maxCount) NN_NOEXCEPT
    {
        m_Buffer = buffer;
        m_BufferSize = bufferSize;
        m_MaxCount = maxCount;
        m_Count = 0;

        if (m_Buffer)
        {
            bool isSuccess = false;
            m_Allocator.Initialize(m_Buffer, m_BufferSize);
            NN_UTIL_SCOPE_EXIT
            {
                if (!isSuccess)
                {
                    m_Allocator.Finalize();
                }
            };

            m_List = reinterpret_cast<SimpleKeyValue*>(m_Allocator.Allocate(sizeof(SimpleKeyValue) * maxCount));
            NN_RESULT_THROW_UNLESS(m_List, ResultBufferNotEnough());

            isSuccess = true;
        }
        NN_RESULT_SUCCESS;
    }

    void SimpleKeyValueCache::InvalidateAll() NN_NOEXCEPT
    {
        if (!m_Buffer)
        {
            return;
        }

        m_Allocator.Finalize();
        m_Allocator.Initialize(m_Buffer, m_BufferSize);
        m_Count = 0;
        m_List = reinterpret_cast<SimpleKeyValue*>(m_Allocator.Allocate(sizeof(SimpleKeyValue) * m_MaxCount));
        NN_ABORT_UNLESS(m_List);
    }

    void SimpleKeyValueCache::Invalidate(const void*, size_t) NN_NOEXCEPT
    {
        if (!m_Buffer)
        {
            return;
        }

        // 単純化のためとりあえず全消し
        InvalidateAll();
    }

    void SimpleKeyValueCache::Put(const void* key, size_t keySize, const void* value, size_t valueSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(keySize <= SimpleKeyValue::MaxKeySize);
        if (!m_Buffer)
        {
            // バッファが設定設定されていないと何もしない
            return;
        }

        if (m_Count == m_MaxCount)
        {
            // 最大個数に達している状態で Put するとキャッシュを全て破棄する
            // NN_DETAIL_KVDB_TRACE("[FileKeyValueStore] Cache Invalidate since count max\n");
            InvalidateAll();
        }

        auto allocated = m_Allocator.Allocate(valueSize);
        if (!allocated)
        {
            // メモリが確保できないとキャッシュを全て破棄してリトライ
            // キャッシュを破棄しても確保できない場合は何もしない
            // NN_DETAIL_KVDB_TRACE("[FileKeyValueStore] Cache Invalidate since failed to allocate %zu bytes\n", valueSize);
            InvalidateAll();
            allocated = m_Allocator.Allocate(valueSize);
            if (!allocated)
            {
                // NN_DETAIL_KVDB_TRACE("[FileKeyValueStore] Failed to allocate %zu bytes even after invalidate\n", valueSize);
                return;
            }
        }

        SimpleKeyValue keyValue = {};
        std::memcpy(keyValue.key, key, keySize);
        keyValue.keySize = keySize;
        keyValue.value = allocated;
        std::memcpy(keyValue.value, value, valueSize);
        keyValue.valueSize = valueSize;

        m_List[m_Count] = keyValue;
        m_Count++;
    }

    util::optional<size_t> SimpleKeyValueCache::TryGet(void* buffer, size_t bufferSize, const void* key, size_t keySize) NN_NOEXCEPT
    {
        if (!m_Buffer)
        {
            return util::nullopt;
        }

        for (int i = 0; i < m_Count; i++)
        {
            auto& keyValue = m_List[i];
            if (keyValue.keySize == keySize && std::memcmp(keyValue.key, key, keySize) == 0)
            {
                if (bufferSize < keyValue.valueSize)
                {
                    return util::nullopt;
                }

                std::memcpy(buffer, keyValue.value, keyValue.valueSize);
                return keyValue.valueSize;
            }
        }

        return util::nullopt;
    }

    util::optional<size_t> SimpleKeyValueCache::TryGetSize(const void* key, size_t keySize) NN_NOEXCEPT
    {
        if (!m_Buffer)
        {
            return util::nullopt;
        }

        for (int i = 0; i < m_Count; i++)
        {
            auto& keyValue = m_List[i];
            if (keyValue.keySize == keySize && std::memcmp(keyValue.key, key, keySize) == 0)
            {
                return keyValue.valueSize;
            }
        }

        return util::nullopt;
    }

    bool SimpleKeyValueCache::Has(const void* key, size_t keySize) NN_NOEXCEPT
    {
        return TryGetSize(key, keySize);
    }

    Result FileKeyValueStore::Initialize(const char* directoryPath) NN_NOEXCEPT
    {
        return Initialize(directoryPath, nullptr, 0, 0);
    }

    Result FileKeyValueStore::Initialize(const char* directoryPath, void* cacheBuffer, size_t cacheSize, int maxCacheCount) NN_NOEXCEPT
    {
        fs::DirectoryEntryType entryType;
        NN_RESULT_DO(fs::GetEntryType(&entryType, directoryPath));
        NN_RESULT_THROW_UNLESS(entryType == fs::DirectoryEntryType_Directory, fs::ResultPathNotFound());
        m_DirectoryPath.Append(directoryPath);
        m_Cache.Initialize(cacheBuffer, cacheSize, maxCacheCount);

        NN_RESULT_SUCCESS;
    }

    Result FileKeyValueStore::Put(const void* key, size_t keySize, const void* value, size_t valueSize) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        NN_SDK_ASSERT(key);
        NN_SDK_ASSERT(keySize > 0);
        NN_RESULT_THROW_UNLESS(keySize <= MaxKeySize, ResultOutOfMaxKeySize());

        if (m_Cache.Has(key, keySize))
        {
            m_Cache.Invalidate(key, keySize);
        }

        Path keyPath = MakeKeyPath(key, keySize, m_DirectoryPath);

        fs::DeleteFile(keyPath);
        NN_RESULT_DO(fs::CreateFile(keyPath, valueSize));

        nn::fs::FileHandle handle;
        NN_RESULT_DO(fs::OpenFile(&handle, keyPath, fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

        fs::WriteOption option = { fs::WriteOptionFlag_Flush };
        NN_RESULT_DO(fs::WriteFile(handle, 0, value, valueSize, option));

        NN_RESULT_SUCCESS;
    }

    Result FileKeyValueStore::Get(size_t* outValue, void* buffer, size_t bufferSize, const void* key, size_t keySize) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        NN_SDK_ASSERT(key);
        NN_SDK_ASSERT(keySize > 0);
        NN_RESULT_THROW_UNLESS(keySize <= MaxKeySize, ResultOutOfMaxKeySize());

        {
            auto size = m_Cache.TryGet(buffer, bufferSize, key, keySize);
            if (size)
            {
                // キャッシュヒット
                // NN_DETAIL_KVDB_TRACE("[FileKeyValueStore] Cache hit %s\n", MakeKeyPath(key, keySize, m_DirectoryPath));
                *outValue = *size;
                NN_RESULT_SUCCESS;
            }
        }

        nn::fs::FileHandle handle;
        NN_RESULT_TRY(fs::OpenFile(&handle, MakeKeyPath(key, keySize, m_DirectoryPath), fs::OpenMode_Read))
            NN_RESULT_CATCH(fs::ResultPathNotFound)
            {
                NN_RESULT_THROW(ResultKeyNotFound());
            }
        NN_RESULT_END_TRY
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

        int64_t fileSize;
        NN_RESULT_DO(fs::GetFileSize(&fileSize, handle));
        NN_RESULT_THROW_UNLESS(fileSize <= static_cast<int64_t>(bufferSize), ResultBufferNotEnough());

        size_t valueSize = static_cast<size_t>(fileSize);
        NN_RESULT_DO(fs::ReadFile(handle, 0, buffer, valueSize));
        *outValue = valueSize;

        // キャッシュにヒットしていないのでキャッシュに積む
        // NN_DETAIL_KVDB_TRACE("[FileKeyValueStore] Cache put %s size %zu bytes\n", MakeKeyPath(key, keySize, m_DirectoryPath), valueSize);
        m_Cache.Put(key, keySize, buffer, valueSize);

        NN_RESULT_SUCCESS;
    }

    Result FileKeyValueStore::GetSize(size_t* outValue, const void* key, size_t keySize) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        NN_SDK_ASSERT(key);
        NN_SDK_ASSERT(keySize > 0);
        NN_RESULT_THROW_UNLESS(keySize <= MaxKeySize, ResultOutOfMaxKeySize());

        {
            auto size = m_Cache.TryGetSize(key, keySize);
            if (size)
            {
                // キャッシュヒット
                // NN_DETAIL_KVDB_TRACE("[FileKeyValueStore] Cache hit (size) %s\n", MakeKeyPath(key, keySize, m_DirectoryPath));
                *outValue = *size;
                NN_RESULT_SUCCESS;
            }
        }

        nn::fs::FileHandle handle;
        NN_RESULT_TRY(fs::OpenFile(&handle, MakeKeyPath(key, keySize, m_DirectoryPath), fs::OpenMode_Read))
            NN_RESULT_CATCH(fs::ResultPathNotFound)
            {
                NN_RESULT_THROW(ResultKeyNotFound());
            }
        NN_RESULT_END_TRY
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

        int64_t fileSize;
        NN_RESULT_DO(fs::GetFileSize(&fileSize, handle));

        *outValue = static_cast<size_t>(fileSize);

        NN_RESULT_SUCCESS;
    }

    Result FileKeyValueStore::Delete(const void* key, size_t keySize) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        NN_SDK_ASSERT(key);
        NN_SDK_ASSERT(keySize > 0);
        NN_RESULT_THROW_UNLESS(keySize <= MaxKeySize, ResultOutOfMaxKeySize());

        if (m_Cache.Has(key, keySize))
        {
            m_Cache.Invalidate(key, keySize);
        }

        NN_RESULT_TRY(fs::DeleteFile(MakeKeyPath(key, keySize, m_DirectoryPath)))
            NN_RESULT_CATCH(fs::ResultPathNotFound)
            {
                NN_RESULT_THROW(ResultKeyNotFound());
            }
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }
}}
