﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/TargetConfigs/build_Os.h>
#include <nn/diag/text/diag_SdkTextInit.h>
#include <nn/TargetConfigs/build_Compiler.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/init/init_Malloc.h>
#include <nn/os.h>
#include <nn/os/os_SdkMemoryAllocatorForThreadLocal.h>

#include "detail/init_Macro.h"

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
    // 文字コードエンコーディングに関する警告の抑制
    #pragma warning( disable : 4566 )
#endif

namespace nn { namespace init { namespace detail {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    void StartupDefaultForWin32() NN_NOEXCEPT;
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    void StartupDefaultForHorizon() NN_NOEXCEPT;
    void* DefaultAllocatorForThreadLocal(size_t, size_t) NN_NOEXCEPT;
    void  DefaultDeallocatorForThreadLocal(void*, size_t) NN_NOEXCEPT;
#else
    #error "未サポートの OS 種別が指定されています。"
#endif
}}} // namespace nn::init::detail


//--------------------------------------------------------------------------
// このファイルには以下に示す関数以外は定義しないで下さい。
//  - nninitStartup() および その中から呼ばれているローカル関数
//  - nninitStartUp()
//
// Windows VC++ 環境では、nninitStartup() の関数定義を
// アプリの .obj で上書きできるようにするため以下のようにしておく。
//
// - nninitStartup() の関数定義だけを単一の .obj で定義
// - 上記 obj を libnn_init.lib 内に収録して提供
//
// 参考 URL：
//  http://stackoverflow.com/questions/5097961/windows-static-library-with-default-functions
//
//--------------------------------------------------------------------------


//--------------------------------------------------------------------------
//  C 言語用の nninitStartup() 関数の定義
//
NN_INIT_EXTERN_C void nninitStartup()
{
    // OS 依存の何か初期化を行なうため用に最初に呼ぶ
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    nn::init::detail::StartupDefaultForWin32();
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::init::detail::StartupDefaultForHorizon();
#endif

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // デフォルトの nninitStartup() は HORIZON 環境でのみメモリ確保を行なう。
    // Win 環境では実質空関数として定義しておく。

    // 確保可能なヒープのサイズを取得
    nn::os::MemoryInfo memInfo;
    nn::os::QueryMemoryInfo(&memInfo);
    size_t size = static_cast<size_t>(memInfo.totalAvailableMemorySize) - memInfo.totalUsedMemorySize;

    // nn::os::MemoryBlockUnitSize に切り下げ
    size = nn::util::align_down(size , nn::os::MemoryBlockUnitSize);

    // メモリヒープの確保
    {
        auto result = nn::os::SetMemoryHeapSize(size);

        // 最大サイズでの確保に失敗した場合は 32MB の小さいサイズでリトライ。
        if (!result.IsSuccess())
        {
            size = 32 * 1024 * 1024;
            result = nn::os::SetMemoryHeapSize(size);
        }

        NN_ABORT_UNLESS(result.IsSuccess(), NN_TEXT_INIT("nninitStartup(): %u MiB のメモリヒープ獲得に失敗しました。"), size / 1024 / 1024);
    }

    // 標準 C ライブラリの malloc の準備
    {
        uintptr_t p;
        auto result = nn::os::AllocateMemoryBlock(&p, size);
        NN_ABORT_UNLESS(result.IsSuccess(), NN_TEXT_INIT("nninitStartup(): %u MiB のメモリブロック獲得に失敗しました。"), size / 1024 / 1024);

        nn::init::InitializeAllocator(reinterpret_cast<void*>(p), size, true);
    }

    // コンパイラのスレッドローカル実装用のメモリアロケータの登録
    nn::os::SetMemoryAllocatorForThreadLocal(
                            nn::init::detail::DefaultAllocatorForThreadLocal,
                            nn::init::detail::DefaultDeallocatorForThreadLocal);
#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)
}

//--------------------------------------------------------------------------
//  C 言語用の nninitStartUp() 関数（'U' が大文字）の定義。
//  CTR-SDK の同名関数を定義した場合に多重定義でリンクエラーとするための措置。
//
NN_INIT_EXTERN_C void nninitStartUp()
{
    NN_ABORT_UNLESS(false,
        "\n*** Obsoleted nninitStartUp() function is defined.\n"
        "*** Please define nninitStartup() instead.\n\n");
}

