﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Windows.h>
#include <nn/hws/hws_Window.h>

#include "hws_MsgProc.h"

namespace nn { namespace hws {

CreateWindowArg::CreateWindowArg() NN_NOEXCEPT
    : m_pWindowName( nullptr )
    , m_pTitle( nullptr )
    , m_Left( CW_USEDEFAULT )
    , m_Top( CW_USEDEFAULT )
    , m_Width( 0 )
    , m_Height( 0 )
{
}

WindowHandle CreateWindow(const CreateWindowArg& arg) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( arg.GetWindowName() );
    NN_SDK_ASSERT_NOT_NULL( arg.GetTitle() );

    HINSTANCE hInstance = GetModuleHandleA(nullptr);
    if (hInstance == nullptr)
    {
        return WindowHandle();
    }
    WNDCLASSEXA winc;
    if (GetClassInfoExA(hInstance, arg.GetWindowName(), &winc))
    {
        return WindowHandle();
    }
    WNDCLASSEXA newWinc = {
        sizeof( WNDCLASSEXA ),
        CS_OWNDC,
        MsgProc::MsgProcCallback,
        0,
        0,
        hInstance,
        nullptr,
        LoadCursor( nullptr, IDC_ARROW ),
        static_cast<HBRUSH>( GetStockObject( BLACK_BRUSH ) ),
        nullptr,
        arg.GetWindowName(),
        nullptr
    };
    if( !RegisterClassExA( &newWinc ) )
    {
        return WindowHandle();
    }

    DWORD style = WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX;
    DWORD exStyle = WS_EX_APPWINDOW;
    RECT rect = { 0, 0, arg.GetWidth(), arg.GetHeight() };
    AdjustWindowRectEx( &rect, style, FALSE, exStyle );

    HWND hWnd = CreateWindowExA(
        exStyle,
        arg.GetWindowName(),
        arg.GetTitle(),
        style,
        arg.GetLeft(),
        arg.GetTop(),
        rect.right - rect.left,
        rect.bottom - rect.top,
        nullptr,
        nullptr,
        hInstance,
        nullptr );

    return WindowHandle( hWnd );
}

void DestroyWindow( WindowHandle handle ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( handle.IsValid() );

    HWND hWnd = static_cast<HWND>( handle.Get() );
    ATOM atom = static_cast<ATOM>( GetClassLongA( hWnd, GCW_ATOM ) );
    DestroyWindow( hWnd );

    if( atom )
    {
        HINSTANCE hInstance = GetModuleHandleA( nullptr );
        if( hInstance )
        {
            UnregisterClassA( reinterpret_cast<LPCSTR>( atom ), hInstance );
        }
    }
}

WindowHandle FindWindow( const char* name ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( name );

    return WindowHandle( FindWindowA( name, nullptr ) );
}

void ShowWindow( WindowHandle handle ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( handle.IsValid() );

    ShowWindow( static_cast<HWND>( handle.Get() ), SW_SHOW );
}

}
} // namespace wnd
