﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  ホスト・ターゲット間通信に関する API
*/

#include <mutex>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/htc/htc_Api.h>
#include <nn/htc/htc_ApiPrivate.h>
#include <nn/htc/htc_Result.h>
#include <nn/os.h>
#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/htc/htc_IHtcManager.h>
#else
#include "htc_HtcManagerByHipc.h"
#endif

//--------------------------------------------------------------------------
//  C++ 向けの宣言
//--------------------------------------------------------------------------
#if defined(NN_BUILD_CONFIG_OS_WIN)
namespace tma
{
    void Initialize();
    void Finalize();
}
namespace nn { namespace tma {
    nn::sf::SharedPointer<IHtcManager> CreateHtcManager() NN_NOEXCEPT;
}}
#endif

namespace nn { namespace htc {

namespace
{
    int g_InitializeCount = 0;
#if defined(NN_BUILD_CONFIG_OS_WIN)
    tma::IHtcManager* g_Manager;
#else
    nn::sf::SharedPointer<tma::IHtcManager> g_Manager;
#endif

    struct StaticMutex
    {
        os::MutexType    m_Mutex;
        void lock() NN_NOEXCEPT
        {
            nn::os::LockMutex( &m_Mutex );
        }
        void unlock() NN_NOEXCEPT
        {
            nn::os::UnlockMutex( &m_Mutex );
        }
    };
    StaticMutex g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

    bool IsInitialized()
    {
        return (g_InitializeCount > 0);
    }

    Result ConvertToHtcResult(nn::Result result)
    {
        if( nn::os::ResultInvalidParameter::Includes(result) )
        {
            return nn::htc::ResultNotFound();
        }
        else if( nn::os::ResultOutOfMemory::Includes(result) )
        {
            return nn::htc::ResultNotEnoughBuffer();
        }
        else if( nn::os::ResultOverflow::Includes(result) )
        {
            return nn::htc::ResultConnectionFailure();
        }
        else if( nn::htc::ResultConnectionFailure::Includes(result) )
        {
            return nn::htc::ResultConnectionFailure();
        }
        else
        {
            return nn::htc::ResultUnknown();
        }
    }
}

    void Initialize() NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        ++g_InitializeCount;
        if(g_InitializeCount > 1)
        {
            return;
        }

#if defined(NN_BUILD_CONFIG_OS_WIN)
        ::tma::Initialize();
        g_Manager = tma::CreateHtcManager().Detach();
#else
        g_Manager = CreateHtcManagerByHipc();
#endif
    }

    void Finalize() NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        NN_SDK_REQUIRES(g_InitializeCount > 0);
        --g_InitializeCount;
        if(IsInitialized())
        {
            return;
        }

#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::sf::ReleaseSharedObject(g_Manager);
        ::tma::Finalize();
#else
        g_Manager = nullptr;
#endif
    }

    Result GetEnvironmentVariableLength(size_t* pOutValue, const char* variableName) NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(variableName);

        NN_SDK_ASSERT(g_Manager);

        size_t variableNameSize = sizeof(char) * (strlen(variableName) + 1);
        int32_t outSize;
        Result result = g_Manager->GetEnvironmentVariableLength( nn::sf::Out<int32_t>(&outSize), nn::sf::InBuffer(variableName, variableNameSize) );
        if( result.IsSuccess() )
        {
            *pOutValue = outSize;
        }
        else
        {
            result = ConvertToHtcResult(result);
        }

        return result;
    }

    Result GetEnvironmentVariable(size_t* pOutSize, char* outBuffer, size_t bufferSize, const char* variableName) NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_SDK_REQUIRES_NOT_NULL(outBuffer);
        NN_SDK_REQUIRES_NOT_NULL(variableName);

        NN_SDK_ASSERT(g_Manager);

        size_t variableNameSize = sizeof(char) * (strlen(variableName) + 1);
        int32_t outSize;
        Result result = g_Manager->GetEnvironmentVariable( nn::sf::Out<int32_t>(&outSize), nn::sf::OutBuffer(outBuffer, bufferSize), nn::sf::InBuffer(variableName, variableNameSize));
        if( result.IsSuccess() )
        {
            *pOutSize = outSize;
        }
        else
        {
            result = ConvertToHtcResult(result);
        }

        return result;
    }

    void BindHostConnectionEvent(nn::os::SystemEvent* pOutValue) NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        nn::sf::NativeHandle sfHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS( g_Manager->GetHostConnectionEvent( &sfHandle ) );
        pOutValue->AttachReadableHandle( sfHandle.GetOsHandle(),
                                         sfHandle.IsManaged(),
                                         nn::os::EventClearMode_ManualClear );
        sfHandle.Detach();
    }

    void BindHostDisconnectionEvent(nn::os::SystemEvent* pOutValue) NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        nn::sf::NativeHandle sfHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Manager->GetHostDisconnectionEvent(&sfHandle));
        pOutValue->AttachReadableHandle(sfHandle.GetOsHandle(),
            sfHandle.IsManaged(),
            nn::os::EventClearMode_ManualClear);
        sfHandle.Detach();
    }

    void BindHostConnectionEventForSystem(nn::os::SystemEvent* pOutValue) NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        nn::sf::NativeHandle sfHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Manager->GetHostConnectionEventForSystem(&sfHandle));
        pOutValue->AttachReadableHandle(sfHandle.GetOsHandle(),
            sfHandle.IsManaged(),
            nn::os::EventClearMode_ManualClear);
        sfHandle.Detach();
    }

    void BindHostDisconnectionEventForSystem(nn::os::SystemEvent* pOutValue) NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        nn::sf::NativeHandle sfHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Manager->GetHostDisconnectionEventForSystem(&sfHandle));
        pOutValue->AttachReadableHandle(sfHandle.GetOsHandle(),
            sfHandle.IsManaged(),
            nn::os::EventClearMode_ManualClear);
        sfHandle.Detach();
    }

    Result GetBridgeIpAddress(char* outBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(outBuffer);

        NN_SDK_ASSERT(g_Manager);

        return g_Manager->GetBridgeIpAddress(nn::sf::OutBuffer(outBuffer, bufferSize));
    }

    Result GetBridgeSubnetMask(char* outBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(outBuffer);

        NN_SDK_ASSERT(g_Manager);

        return g_Manager->GetBridgeSubnetMask(nn::sf::OutBuffer(outBuffer, bufferSize));
    }

    Result GetBridgeMacAddress(char* outBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(outBuffer);

        NN_SDK_ASSERT(g_Manager);

        return g_Manager->GetBridgeMacAddress(nn::sf::OutBuffer(outBuffer, bufferSize));
    }

    Result GetBridgePort(char* outBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(outBuffer);

        NN_SDK_ASSERT(g_Manager);

        return g_Manager->GetBridgePort(nn::sf::OutBuffer(outBuffer, bufferSize));
    }

    Result GetWorkingDirectoryPathSize(size_t* pOutValue) NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        NN_SDK_ASSERT(g_Manager);

        int32_t outSize;
        Result result = g_Manager->GetWorkingDirectoryPathSize( nn::sf::Out<int32_t>(&outSize) );
        if( result.IsSuccess() )
        {
            *pOutValue = outSize;
        }

        return result;
    }

    Result GetWorkingDirectoryPath(char* outBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        std::lock_guard<StaticMutex> lock(g_Mutex);

        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(outBuffer);

        NN_SDK_ASSERT(g_Manager);

        Result result = g_Manager->GetWorkingDirectoryPath( nn::sf::OutBuffer(outBuffer, bufferSize), static_cast<int32_t>(bufferSize) );

        return result;
    }

}}
