﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_DataIo.h>
#include <nn/gc/detail/gc_GeneralIo.h>
#include <nn/gc/detail/gc_AsicHandler.h>
#include <nn/gc/detail/gc_AsicHandlerCore.h>
#include <nn/gc/detail/gc_StateMachine.h>
#include <nn/gc/detail/gc_GcCrypto.h>
#include <nn/gc/detail/gc_EmbeddedDataHolder.h>
#include <nn/gc/detail/gc_Types.h>
#include <nn/gc/detail/gc_Log.h>

#define NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED if(g_IsGcLibInitialized == false){ NN_ABORT("Error: Gc lib is not initialized\n"); }

namespace nn { namespace gc {

namespace {
    bool g_IsGcLibInitialized = false;
}

// #define NN_DETAIL_GC_DISABLE_GCLIB
#ifdef NN_DETAIL_GC_DISABLE_GCLIB
#include "gc.stub.inner.cpp"
#else

void PresetInternalKeys(const void* encryptedKeyBuffer, const size_t encryptedKeyBufferSize,
                        const void* socCertBuffer, const size_t socCertBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    nn::Result result = nn::gc::detail::EmbeddedDataHolder::SetLibraryEmbeddedKeys();
    if(result.IsFailure())
    {
        nn::gc::detail::AsicHandlerCore::GetInstance().SetSplFatalErrorOccurredFlag(result);
        return;
    }

    result = nn::gc::detail::EmbeddedDataHolder::SetEmmcEmbeddedKeys(
        reinterpret_cast<const char*>(encryptedKeyBuffer), encryptedKeyBufferSize,
        reinterpret_cast<const char*>(socCertBuffer), socCertBufferSize);
    if(result.IsFailure())
    {
        nn::gc::detail::AsicHandlerCore::GetInstance().SetSplFatalErrorOccurredFlag(result);
        return;
    }
}

// Initialize を内包する Initialize とややこしいので注意
// NOTE: Initialize 内部で GPIO ピンの出力を行うため注意（AsicHandler->Initialize->Reset->DeactivateAsicIo, Initialize->DeviceDetector）
void Initialize(void* workBuffer, const size_t workBufferLength, nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    nn::gc::detail::AsicHandler::GetInstance().SetBuffer(workBuffer, workBufferLength, workBufferDeviceVirtualAddress);
    nn::gc::detail::AsicHandler::GetInstance().Initialize();
    g_IsGcLibInitialized = true;
}

void Finalize() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    nn::gc::detail::AsicHandler::GetInstance().Finalize();
    g_IsGcLibInitialized = false;
}

void PowerOffGameCard() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // ここの API が呼ばれるのはシャットダウン時であり、fs のサービスが止まっていて GC の要求も来ない状態を想定しているが、
    // ノンブロッキングの Awaken の処理（カード電源 ON を含みうる）が GC ASIC スレッド側で残っている可能性があるので、これを待つ
    // NOTE: 簡易コードをやめて、カード電源 OFF 自体を OrderWork の index として用意してもいいかもしれない
    detail::AsicHandler::GetInstance().OrderWork(detail::AsicWork_None);

    // カード電源 OFF
    nn::gc::detail::GeneralIo::GetInstance().SetGcPowerOff();
}

void RegisterDeviceVirtualAddress(const uintptr_t bufferAddress, const size_t bufferSize, const nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    nn::gc::detail::DataIo::GetInstance().RegisterDeviceVirtualAddress(bufferAddress, bufferSize, bufferDeviceVirtualAddress);
}

void UnregisterDeviceVirtualAddress(const uintptr_t bufferAddress, const size_t bufferSize, const nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    nn::gc::detail::DataIo::GetInstance().UnregisterDeviceVirtualAddress(bufferAddress, bufferSize, bufferDeviceVirtualAddress);
}

nn::Result GetInitializationResult() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    nn::Result result = nn::gc::detail::AsicHandler::GetInstance().OrderWork(nn::gc::detail::AsicWork_None);
    NN_DETAIL_GC_RESULT_LOG(result);
    return result;
}

nn::Result Activate() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    nn::Result result = nn::gc::detail::AsicHandler::GetInstance().OrderWork(nn::gc::detail::AsicWork_Activate);
    NN_DETAIL_GC_RESULT_LOG(result);
    return result;
}

void Deactivate() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    nn::Result resultToBeDiscarded = nn::gc::detail::AsicHandler::GetInstance().OrderWork(nn::gc::detail::AsicWork_Deactivate);
    NN_UNUSED(resultToBeDiscarded);
}

nn::Result SetCardToSecureMode() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    // ASIC <=> ゲームカード間のセッション構築。セキュアエリアの Read が可能になるところまで
    nn::Result result = nn::gc::detail::AsicHandler::GetInstance().OrderWork(nn::gc::detail::AsicWork_SetCardToSecureMode);
    NN_DETAIL_GC_RESULT_LOG(result);
    return result;
}

nn::Result Read(void* outDataBuffer, const size_t bufferLength, const uint32_t pageAddress, const uint32_t pageCount) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    nn::Result result = nn::gc::detail::AsicHandler::GetInstance().OrderWorkRead(reinterpret_cast<char*>(outDataBuffer), bufferLength, pageAddress, pageCount);
    NN_DETAIL_GC_RESULT_LOG(result);
    return result;
}

void PutToSleep() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    nn::Result result = nn::gc::detail::AsicHandler::GetInstance().OrderWork(nn::gc::detail::AsicWork_PutToSleep);
    if(result.IsFailure())
    {
        NN_DETAIL_GC_ERR_LOG("PutToSleep Failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }
}

void Awaken() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    // Awaken はブロックさせず、直ちに返す
    nn::Result result = nn::gc::detail::AsicHandler::GetInstance().OrderWorkWithNonBlocking(nn::gc::detail::AsicWork_Awaken);
    if(result.IsFailure())
    {
        NN_DETAIL_GC_ERR_LOG("Awaken Failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }
}

nn::Result GetCardStatus(GameCardStatus* pOutValue) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    nn::Result result = nn::gc::detail::AsicHandler::GetInstance().OrderWorkGetCardStatus(pOutValue);
    NN_DETAIL_GC_RESULT_LOG(result);
    return result;
}

bool IsCardInserted() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG_LINE;
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    // GPIO の結果を直で返す
    return nn::gc::detail::StateMachine::GetInstance().IsCardInserted();
}

bool IsCardActivationValid() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG_LINE;
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    bool isCardActivationValid = false;
    nn::gc::detail::AsicHandler::GetInstance().OrderWorkIsCardActivationValid(&isCardActivationValid);
    return isCardActivationValid;
}

nn::Result GetCardDeviceId(void* outBuffer, const size_t outBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    nn::Result result = nn::gc::detail::AsicHandler::GetInstance().OrderWorkGetCardDeviceId(reinterpret_cast<char*>(outBuffer), outBufferSize);
    NN_DETAIL_GC_RESULT_LOG(result);
    return result;
}

nn::Result GetCardDeviceCertificate(void* outBuffer, const size_t outBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    nn::Result result = nn::gc::detail::AsicHandler::GetInstance().OrderWorkGetCardDeviceCertificate(reinterpret_cast<char*>(outBuffer), outBufferSize);
    NN_DETAIL_GC_RESULT_LOG(result);
    return result;
}

nn::Result GetCardImageHash(void* outBuffer, const size_t outBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    nn::Result result = nn::gc::detail::AsicHandler::GetInstance().OrderWorkGetCardImageHash(reinterpret_cast<char*>(outBuffer), outBufferSize);
    NN_DETAIL_GC_RESULT_LOG(result);
    return result;
}

nn::Result GetGameCardIdSet(GameCardIdSet* pOutValue) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;
    nn::Result result = nn::gc::detail::AsicHandler::GetInstance().OrderWorkGetCardIdSet(pOutValue);
    NN_DETAIL_GC_RESULT_LOG(result);
    return result;
}

void RegisterDetectionEventCallback(nn::gc::detail::GcCallbackFunctionPointer const pDetectionEventCallback, void* const pParameter) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    nn::gc::detail::GcCallbackStruct detectionEventCallback = { pDetectionEventCallback, pParameter };
    nn::gc::detail::AsicHandler::GetInstance().RegisterDetectionEventCallback(&detectionEventCallback);
}

void UnregisterDetectionEventCallback() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    nn::gc::detail::AsicHandler::GetInstance().UnregisterDetectionEventCallback();
}

nn::Result GetCardHeader(void* outBuffer, const size_t outBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    nn::Result result = nn::gc::detail::AsicHandler::GetInstance().OrderWorkGetCardHeader(reinterpret_cast<char*>(outBuffer), outBufferSize);
    NN_DETAIL_GC_RESULT_LOG(result);
    return result;
}

nn::Result GetErrorInfo(nn::fs::GameCardErrorReportInfo* pOutGameCardErrorReportInfo) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_CHECK_IF_GCLIB_INITIALIZED;

    nn::Result result = nn::gc::detail::AsicHandler::GetInstance().OrderWorkGetErrorInfo(pOutGameCardErrorReportInfo);
    NN_DETAIL_GC_RESULT_LOG(result);
    return result;
}

#endif




// *** static assert *** //
namespace detail {
NN_STATIC_ASSERT(sizeof(RmaInformation) == 512);
NN_STATIC_ASSERT(sizeof(CardId1) == 4);
NN_STATIC_ASSERT(sizeof(CardId2) == 4);
NN_STATIC_ASSERT(sizeof(CardId3) == 4);
NN_STATIC_ASSERT(sizeof(CardUid) == 64);
NN_STATIC_ASSERT(sizeof(SelfRefreshStatus) == 4);
NN_STATIC_ASSERT(sizeof(CardHeader) == 256);
NN_STATIC_ASSERT(sizeof(T1CardCertificate) == 1024);
NN_STATIC_ASSERT(sizeof(T2CardCertificate) == 1024);
NN_STATIC_ASSERT(sizeof(CardSecurityInformation) == 512 * 4);
NN_STATIC_ASSERT(sizeof(CardInitReceiveData) == 8);
NN_STATIC_ASSERT(sizeof(DevCardErrorCount) == 64);
NN_STATIC_ASSERT(sizeof(DevCardParameter) == 512);
NN_STATIC_ASSERT(sizeof(RefreshResponse) == 64);
} // namespace dtail {

}} // namespace nn { namespace gc {
