﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>

#include <nn/gc/detail/gc_ThreadInterface.h>

namespace nn { namespace gc {
namespace detail {

ThreadInterface::ThreadInterface() NN_NOEXCEPT
{

}

void ThreadInterface::InitializeThread(char* const threadStack, const size_t threadStackSize, const char* threadNamePointer, const int threadPriority) NN_NOEXCEPT
{
    nn::Result result = nn::os::CreateThread( &m_Thread, RunThreadFunction, this, threadStack, threadStackSize, threadPriority );
    if ( !result.IsSuccess() )
    {
        NN_ABORT("failed to create a thread\n");
    }

    nn::os::SetThreadNamePointer(&m_Thread, threadNamePointer);

    // 待ちイベントの初期設定
    InitializeWaitEvents();

    // スレッドの開始
    nn::os::StartThread( &m_Thread );
}

void ThreadInterface::FinalizeThread() NN_NOEXCEPT
{
    // スレッド終了イベントの処理
    nn::os::SignalEvent(&m_ThreadEndEvent);

    // スレッドの終了
    nn::os::WaitThread( &m_Thread );
    nn::os::DestroyThread( &m_Thread );

    // 待ちイベントの終了処理
    FinalizeWaitEvents();
}

void ThreadInterface::InitializeWaitEvents() NN_NOEXCEPT
{
    // スレッド終了イベントの初期化
    nn::os::InitializeEvent(&m_ThreadEndEvent, false, nn::os::EventClearMode_ManualClear);

    // イベント待ちの登録
    nn::os::InitializeMultiWait(&m_Waiter);
    nn::os::InitializeMultiWaitHolder(&m_ThreadEndEventHolder, &m_ThreadEndEvent);
    nn::os::LinkMultiWaitHolder(&m_Waiter, &m_ThreadEndEventHolder);
}

void ThreadInterface::FinalizeWaitEvents() NN_NOEXCEPT
{
    nn::os::UnlinkMultiWaitHolder(&m_ThreadEndEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_ThreadEndEventHolder);
    nn::os::FinalizeMultiWait(&m_Waiter);

    // スレッド終了イベントの終了処理
    nn::os::FinalizeEvent(&m_ThreadEndEvent);
}


void ThreadInterface::RunThreadFunctionImpl() NN_NOEXCEPT
{
    while(NN_STATIC_CONDITION(true))
    {
        nn::os::MultiWaitHolderType* signaledHolder = nn::os::WaitAny(&m_Waiter);
        if (signaledHolder == &m_ThreadEndEventHolder)
        {
            break;
        }

        // メイン処理
        const int sleepTime = 10;
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(sleepTime) );
    }
}


} } }
