﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>

#include <nn/spl/spl_Api.h>

#include <nn/crypto/crypto_RsaPkcs1Sha256Verifier.h>
#include <nn/crypto/crypto_RsaOaepEncryptor.h>
#include <nn/crypto/crypto_RsaOaepDecryptor.h>

#include <nn/gc/detail/gc_Define.h>
#include <nn/gc/detail/gc_Log.h>
#include <nn/gc/detail/gc_Util.h>

#include <nn/gc/detail/gc_GcCrypto.h>
#include <nn/gc/detail/gc_EmbeddedDataHolder.h>
#include <nn/gc/detail/gc_Asn1Parser.h>

namespace nn { namespace gc {
namespace detail{

char GcCrypto::g_AsicCertBuffer[GcCertificateSize];

GcCrypto::GcCrypto() NN_NOEXCEPT
{
    // AES-CTR カウンタは逆方向
    m_SessionAesCtrCounter.UseInverseCounter();

    this->ClearKeys();
}

GcCrypto& GcCrypto::GetInstance() NN_NOEXCEPT
{
    static GcCrypto s_Instance;
    return s_Instance;
}

bool GcCrypto::IsProd() NN_NOEXCEPT
{
    return ! EmbeddedDataHolder::IsDev();
}

// GcCryptoクラス向け内部処理
void GcCrypto::SetCvCounter(const char* randomValue1Buffer, const size_t randomValue1BufferLength,
                            const char* randomValue2Buffer, const size_t randomValue2BufferLength) NN_NOEXCEPT
{
    // バッファの長さ違いは許容しない
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(randomValue1BufferLength == GcRandomValueSize && randomValue2BufferLength == GcRandomValueSize);
    NN_DETAIL_GC_CRYPTO_LOG("Set CV counter\n");

    // SHA256生成器の初期化
    m_Sha256Generator.Initialize();

    // 乱数RND1, RND2 あるいは RND6, RND5M が渡されるので、それを入れる
    m_Sha256Generator.Update(randomValue1Buffer, randomValue1BufferLength);
    m_Sha256Generator.Update(randomValue2Buffer, randomValue2BufferLength);

    // SHA256 を計算
    char resultBuffer[GcSha256Length];
    m_Sha256Generator.GetHash(resultBuffer, sizeof(resultBuffer));

    // 上位16byteのみ使用
    m_SessionCvCounter.ResetCounterWithInitialValue(resultBuffer, GcCounter128bitByteNum);
}

void GcCrypto::SetRandomValuesForKeyUpdate(const char* randomValue5Buffer, const size_t randomValue5BufferLength,
                                           const char* randomValue6Buffer, const size_t randomValue6BufferLength) NN_NOEXCEPT
{
    // バッファの長さ違いは許容しない
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(randomValue5BufferLength == GcRandomValueForKeyUpdateSocSize && randomValue6BufferLength == GcRandomValueSize);
    NN_DETAIL_GC_CRYPTO_LOG("Set random values for key update\n");

    // とりあえず受け取った乱数（RND6）を保持しておく
    std::memcpy(m_RandomValueForKeyUpdateAsic, randomValue6Buffer, randomValue6BufferLength);

    // 鍵更新で使用するための RND5M を計算する
    // SHA256生成器の初期化
    m_Sha256Generator.Initialize();

    // 前回の RND5M（初めての場合は RND2）を先頭に入れる
    // 鍵更新をしたことがなければ、m_HashStockForKeyUpdate にはRND2が入っている（RND5Mではない）
    m_Sha256Generator.Update(m_HashStockForKeyUpdate, sizeof(m_HashStockForKeyUpdate));

    // 今回作った RND5 を後ろにくっつける
    m_Sha256Generator.Update(randomValue5Buffer, randomValue5BufferLength);

    // SHA256 を計算し、RND5M とする
    m_Sha256Generator.GetHash(m_HashStockForKeyUpdate, sizeof(m_HashStockForKeyUpdate));
}

void GcCrypto::UpdateSessionKey() NN_NOEXCEPT
{
    NN_DETAIL_GC_CRYPTO_LOG("Update session key\n");

    // 保持していた RND6, RND5M を使って CVカウンタ, AES-CTR/CBCのIV, Keyの更新
    UpdateCommonKeyFromRandomValues(m_RandomValueForKeyUpdateAsic, sizeof(m_RandomValueForKeyUpdateAsic),
        m_HashStockForKeyUpdate, sizeof(m_HashStockForKeyUpdate));

    // 鍵更新回数をカウントする（0回と分けたいので、最大値だったら1にリセットする）
    m_KeyUpdateCount = (m_KeyUpdateCount == UINT64_MAX) ? 1: (m_KeyUpdateCount + 1);
}

// 外部向け関数
void GcCrypto::ClearKeys() NN_NOEXCEPT
{
    m_LastAesCtrOperation = LastAesCtrOperation_None;

    detail::SecureMemoryZero(g_AsicCertBuffer, sizeof(g_AsicCertBuffer));

    detail::SecureMemoryZero(m_AsicPublicKeyModulus, sizeof(m_AsicPublicKeyModulus));
    detail::SecureMemoryZero(m_AsicPublicKeyExponent, sizeof(m_AsicPublicKeyExponent));

    detail::SecureMemoryZero(m_CommonKey, sizeof(m_CommonKey));

    detail::SecureMemoryZero(m_RandomValueForKeyUpdateAsic, sizeof(m_RandomValueForKeyUpdateAsic));

    detail::SecureMemoryZero(m_HashStockForKeyUpdate, sizeof(m_HashStockForKeyUpdate));
    m_IsHashStockForKeyUpdateReady = false;
    m_KeyUpdateCount = 0;

    detail::SecureMemoryZero(m_AesInitialVector, sizeof(m_AesInitialVector));

    m_SessionAesCtrCounter.ResetCounter();
    m_SessionCvCounter.ResetCounter();
}

nn::Result GcCrypto::VerifyCertificateAndGetPublicKey(const char *certBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_CRYPTO_LOG("Verify certificate\n");
    GC_DETAIL_TEST_BREAK_RESULT;

    Asn1Parser asnParser;
    if(! asnParser.ParseGcAsicCertificate(certBuffer,bufferLength))
    {
        return nn::fs::ResultGameCardParseCertificateFailure();
    }

    // 証明書の署名検証
    if(!nn::crypto::VerifyRsa2048Pkcs1Sha256(
        certBuffer + asnParser.GetSignatureIndex(), asnParser.GetSignatureLength(),
        EmbeddedDataHolder::g_LibraryEmbeddedCa1Modulus, sizeof(EmbeddedDataHolder::g_LibraryEmbeddedCa1Modulus),
        EmbeddedDataHolder::g_LibraryEmbeddedCa1PublicExponent, sizeof(EmbeddedDataHolder::g_LibraryEmbeddedCa1PublicExponent),
        certBuffer + asnParser.GetTbsCertificateIndex(), asnParser.GetTbsCertificateLength()
        ))
    {
        NN_DETAIL_GC_ERR_LOG("Error: certificate is not valid\n");
        return nn::fs::ResultGameCardInvalidCertificate();
    }

    // (SerialNumber)
    char serialNumber[GcAsicSeriralNumberLength];
    asnParser.GetSerialNumber(serialNumber, sizeof(serialNumber));
    NN_DETAIL_GC_DEBUG_MEMDUMP(serialNumber, sizeof(serialNumber), "SerialNumber(size : %d bytes)\n", sizeof(serialNumber));

    // (Modulus)
    asnParser.GetModulus(m_AsicPublicKeyModulus, sizeof(m_AsicPublicKeyModulus));
    NN_DETAIL_GC_DEBUG_MEMDUMP(m_AsicPublicKeyModulus, sizeof(m_AsicPublicKeyModulus), "Modulus(size : %d bytes (%d bits))\n", sizeof(m_AsicPublicKeyModulus), sizeof(m_AsicPublicKeyModulus) * 8);

    // (Exponent)
    asnParser.GetExponent(m_AsicPublicKeyExponent, sizeof(m_AsicPublicKeyExponent));
    NN_DETAIL_GC_DEBUG_MEMDUMP(m_AsicPublicKeyExponent, sizeof(m_AsicPublicKeyExponent), "Exponent(size : %d bytes)\n", sizeof(m_AsicPublicKeyExponent));

    return nn::ResultSuccess();
}

void GcCrypto::GenerateCommonKeyFromRandomValues(const char* randomValue1Buffer, const size_t randomValue1BufferLength,
                                                 const char* randomValue2Buffer, const size_t randomValue2BufferLength) NN_NOEXCEPT
{
    // バッファの長さ違いは許容しない
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(randomValue1BufferLength == GcRandomValueSize && randomValue2BufferLength == GcRandomValueSize);
    NN_DETAIL_GC_CRYPTO_LOG("Generate common key\n");

    // 初回の乱数RND2はあとの鍵更新のために保存しておく
    if ( m_IsHashStockForKeyUpdateReady == false )
    {
        NN_DETAIL_GC_CRYPTO_LOG("Random value (RND2) stored for update key\n");
        memcpy(m_HashStockForKeyUpdate, randomValue2Buffer, randomValue2BufferLength);
        m_IsHashStockForKeyUpdateReady = true;
    }

    // HMAC による鍵生成(内部的に鍵を消去)
    char resultBuffer[GcSha256Length];
    EmbeddedDataHolder::GetHmacForGenerateRandomValues(resultBuffer, sizeof(resultBuffer), randomValue1Buffer, randomValue1BufferLength, randomValue2Buffer, randomValue2BufferLength);

    // 上位16byteは共通鍵（通常のメモリで扱う）
    std::memcpy(m_CommonKey, resultBuffer, GcCommonKeyLength);

    // 下位16byteはAES-CBC, AES-CTRの初期ベクトル（IV）
    std::memcpy(m_AesInitialVector, resultBuffer + GcCommonKeyLength, sizeof(resultBuffer) - GcCommonKeyLength);

    // 使用済みなので消去
    detail::SecureMemoryZero(resultBuffer, sizeof(resultBuffer));

    // 暗号化・復号化で共通のカウンタを持つことと、中断処理にも対応する為、AES-CTRカウンタを別に用意しておく
    NN_DETAIL_GC_CRYPTO_LOG("Set AES-CTR counter\n");
    m_SessionAesCtrCounter.ResetCounterWithInitialValue(m_AesInitialVector, sizeof(m_AesInitialVector));
    m_LastAesCtrOperation = LastAesCtrOperation_None;

    // CVカウンタは HMAC-SHA256 ではなく SHA256 なので、別途算出してセットする
    SetCvCounter(randomValue1Buffer, randomValue1BufferLength, randomValue2Buffer, randomValue2BufferLength);
}


void GcCrypto::AddCommandVerificationValue(char* outWorkBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    // バッファの長さ違いは許容しない
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(bufferLength == GcMmcCmd60DataSize,
        "Buffer for command verification value should be %zu byte (%d byte passed)\n", GcMmcCmd60DataSize, bufferLength);
    NN_DETAIL_GC_CRYPTO_LOG("Add command verification\n");

    // HMAC-SHA256生成器の初期化
    m_HmacSha256Generator.Initialize(EmbeddedDataHolder::g_LibraryEmbeddedHmacKeyForCv, sizeof(EmbeddedDataHolder::g_LibraryEmbeddedHmacKeyForCv));

    // 先頭 32 byte のオペレーションデータをコピー
    const size_t partOperationLength = GcMmcCmd60DataSize / 2;  // 32
    m_HmacSha256Generator.Update(outWorkBuffer, partOperationLength);

    // 途中 16 byte はCVカウンタの値を入れる
    m_HmacSha256Generator.Update(m_SessionCvCounter.valueCounterArray, m_SessionCvCounter.counterLength);

    // 後半 16 byte は固定値を入れる
    m_HmacSha256Generator.Update(EmbeddedDataHolder::g_LibraryEmbeddedCvConstantValue, sizeof(EmbeddedDataHolder::g_LibraryEmbeddedCvConstantValue));

    // HMAC-SHA256 を計算
    char resultBuffer[GcSha256Length];
    m_HmacSha256Generator.GetMac(resultBuffer, sizeof(resultBuffer));

    // 結果を後半 32 byte 領域にコピー
    std::memcpy(outWorkBuffer + partOperationLength, resultBuffer, GcCvLength);

    // CV カウンタをインクリメント
    m_SessionCvCounter.Increment();
}


// 乱数生成
nn::Result GcCrypto::GenerateRandomValue(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    return GcCrypto::GenerateRandomBytesSpl(outBuffer, bufferLength);
}

// 固定値生成
void GcCrypto::GenerateFixedValue(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    for(size_t i = 0; i < bufferLength; i++)
    {
        outBuffer[i] = static_cast<u8>(i);
    }
}

// RSA-OAEP
nn::Result GcCrypto::DecryptWithRsaOaep(size_t* outLength, char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_CRYPTO_LOG("RSA OAEP decrypt %zu byte(s)\n", inBufferLength);
    GC_DETAIL_TEST_BREAK_RESULT;

    size_t resultSize = 0;
    NN_RESULT_DO( EmbeddedDataHolder::DecryptGcMessageSecure(&resultSize, outBuffer, outBufferLength, inBuffer, inBufferLength) );
    *outLength = resultSize;
    NN_RESULT_SUCCESS;
}

void GcCrypto::EncryptWithRsaOaep(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_CRYPTO_LOG("RSA OAEP encrypt %zu byte(s)\n", inBufferLength);
    GC_DETAIL_TEST_BREAK;
    NN_ABORT_UNLESS_RESULT_SUCCESS(EmbeddedDataHolder::EncryptGcMessageSecure(outBuffer, outBufferLength, inBuffer, inBufferLength,
                                                                            m_AsicPublicKeyModulus, sizeof(m_AsicPublicKeyModulus),
                                                                            m_AsicPublicKeyExponent, sizeof(m_AsicPublicKeyExponent)));
}


// AES-CBC
void GcCrypto::DecryptWithAesCbc(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(inBufferLength <= outBufferLength);
    NN_DETAIL_GC_CRYPTO_LOG("AES-CBC decrypt %zu byte(s)\n", inBufferLength);

    // AES-CBC の場合、IV は毎回リセットを行う
    nn::crypto::DecryptAes128Cbc(outBuffer, outBufferLength, m_CommonKey, sizeof(m_CommonKey), m_AesInitialVector, sizeof(m_AesInitialVector), inBuffer, inBufferLength);
}

void GcCrypto::EncryptWithAesCbc(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(inBufferLength <= outBufferLength);
    NN_DETAIL_GC_CRYPTO_LOG("AES-CBC encrypt %zu byte(s)\n", inBufferLength);

    // AES-CBC の場合、IV は毎回リセットを行う
    nn::crypto::EncryptAes128Cbc(outBuffer, outBufferLength, m_CommonKey, sizeof(m_CommonKey), m_AesInitialVector, sizeof(m_AesInitialVector), inBuffer, inBufferLength);
}

// AES-CTR

// メンバに持っているカウンタを使って復号化します。
// 本 API は内部的な処理で必要な Read ときなどに使用します。
void GcCrypto::DecryptWithAesCtr(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(inBufferLength <= outBufferLength);
    NN_DETAIL_GC_CRYPTO_LOG("AES-CTR decrypt %zu byte(s)\n", inBufferLength);

    // 連続した処理でなければ、暗号化処理などによってカウンタ値が更新されていると考えられるため、AES-CTRカウンタを再セットする
    size_t length = nn::crypto::DecryptAes128Ctr(outBuffer, outBufferLength, m_CommonKey, sizeof(m_CommonKey), m_SessionAesCtrCounter.valueCounterArray, m_SessionAesCtrCounter.counterLength, inBuffer, inBufferLength);

    // カウントアップ
    m_SessionAesCtrCounter.CountUp( static_cast<GcCounter128bitContainerType>(length / static_cast<size_t>(GcAes128BlockSize)) );
}

// 入力に与えたカウンタを使って復号化します。
// 本 API はFS など外部からの命令で行われる Read でのみ使用します。
void GcCrypto::DecryptWithAesCtrWithCounter(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength, const char* counterBuffer, const size_t counterLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(inBufferLength <= outBufferLength);
    NN_DETAIL_GC_CRYPTO_LOG("AES-CTR decrypt %zu byte(s)\n", inBufferLength);

    nn::crypto::DecryptAes128Ctr(outBuffer, outBufferLength, m_CommonKey, sizeof(m_CommonKey), counterBuffer, counterLength, inBuffer, inBufferLength);
}

void GcCrypto::EncryptWithAesCtr(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(inBufferLength <= outBufferLength);
    NN_DETAIL_GC_CRYPTO_LOG("AES-CTR encrypt %zu byte(s)\n", inBufferLength);

    // 暗号化
    size_t length = nn::crypto::EncryptAes128Ctr(outBuffer, outBufferLength, m_CommonKey, sizeof(m_CommonKey), m_SessionAesCtrCounter.valueCounterArray, m_SessionAesCtrCounter.counterLength, inBuffer, inBufferLength);

    // カウントアップ
    m_SessionAesCtrCounter.CountUp( static_cast<GcCounter128bitContainerType>(length / static_cast<size_t>(GcAes128BlockSize)) );
}

GcCrypto::~GcCrypto() NN_NOEXCEPT
{
}


// *** static spl wrapper ***
#ifdef NN_BUILD_CONFIG_OS_WIN
#define NN_DETAIL_GC_DISABLE_RESULT_IF_OS_WIN(statement) ResultSuccess()
#define NN_DETAIL_GC_DISABLE_BOOL_IF_OS_WIN(statement) false
#else
#define NN_DETAIL_GC_DISABLE_RESULT_IF_OS_WIN(statement) statement
#define NN_DETAIL_GC_DISABLE_BOOL_IF_OS_WIN(statement) statement
#endif

bool GcCrypto::CheckDevelopmentSpl() NN_NOEXCEPT
{
    return NN_DETAIL_GC_DISABLE_BOOL_IF_OS_WIN(nn::spl::IsDevelopment());
}

nn::Result GcCrypto::DecryptAesKeySpl(void* pOutBuffer, size_t outBufferSize,
        const void* pSource, size_t sourceSize, int generation, Bit32 option) NN_NOEXCEPT
{
    nn::Result result = NN_DETAIL_GC_DISABLE_RESULT_IF_OS_WIN(nn::spl::DecryptAesKey(pOutBuffer, outBufferSize, pSource, sourceSize, generation, option));
    if(result.IsFailure())
    {
        NN_DETAIL_GC_ERR_LOG("DecryptAesKey Failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return nn::fs::ResultGameCardSplDecryptAesKeyFailure();
    }
    NN_RESULT_SUCCESS;
}

nn::Result GcCrypto::DecryptAndStoreGcKeySpl(
        const void* pData,
        size_t      dataSize  ) NN_NOEXCEPT
{
    nn::Result result = NN_DETAIL_GC_DISABLE_RESULT_IF_OS_WIN(nn::spl::DecryptAndStoreGcKey(pData, dataSize));
    if(result.IsFailure())
    {
        NN_DETAIL_GC_ERR_LOG("DecryptAndStoreGcKeySpl Failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return nn::fs::ResultGameCardSplDecryptAndStoreGcKeyFailure();
    }
    NN_RESULT_SUCCESS;
}

nn::Result GcCrypto::GenerateRandomBytesSpl(void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT
{
    nn::Result result = NN_DETAIL_GC_DISABLE_RESULT_IF_OS_WIN(nn::spl::GenerateRandomBytes(pOutBuffer, bufferSize));
    if(result.IsFailure())
    {
        NN_DETAIL_GC_ERR_LOG("GenerateRandomBytesSpl Failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return nn::fs::ResultGameCardSplGenerateRandomBytesFailure();
    }
    NN_RESULT_SUCCESS;
}

nn::Result GcCrypto::DecryptGcMessageSpl(
        size_t*     pOutResultSize,
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pCipher,
        size_t      cipherSize,
        const void* pModulus,
        size_t      modulusSize,
        const void* pLabelDigest,
        size_t      labelDigestSize  ) NN_NOEXCEPT
{
    nn::Result result = NN_DETAIL_GC_DISABLE_RESULT_IF_OS_WIN(nn::spl::DecryptGcMessage(
        pOutResultSize, pResultBuffer, resultBufferSize, pCipher, cipherSize, pModulus, modulusSize, pLabelDigest, labelDigestSize));
    if(result.IsFailure())
    {
        NN_DETAIL_GC_ERR_LOG("DecryptGcMessageSpl Failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return nn::fs::ResultGameCardSplDecryptGcMessageFailure();
    }
    NN_RESULT_SUCCESS;
}


} } }
