﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>

#include <nn/crypto/crypto_Aes128CtrDecryptor.h>

#include <nn/gc/detail/gc_Define.h>
#include <nn/gc/detail/gc_Log.h>
#include <nn/gc/detail/gc_Util.h>

#include <nn/gc/detail/gc_GcCrypto.h>
#include <nn/gc/detail/gc_EmbeddedDataHolder.h>
#include <nn/crypto/crypto_RsaOaepEncryptor.h>

namespace nn { namespace gc {
namespace detail{

bool EmbeddedDataHolder::g_IsDev = false;

// *** バイナリ直埋めこみのデータ ***
// Emmc 埋めこみ
char EmbeddedDataHolder::g_EmmcEmbeddedSocCertificate[GcCertificateSize] = { 0 };
char EmbeddedDataHolder::g_EmmcEmbeddedSocRsaKeyModulus[GcCrypto::GcRsaKeyLength] = { 0 };

// Library 埋めこみ：暗号化されるもの
char EmbeddedDataHolder::g_LibraryEmbeddedHmacKeyForCv[GcCrypto::GcHmacKeyLength] = { 0 };
char EmbeddedDataHolder::g_LibraryEmbeddedCvConstantValue[GcCrypto::GcCvConstLength] = { 0 };

// Library 埋めこみ：暗号化不要かつ dev/prod で不変
char EmbeddedDataHolder::g_LibraryEmbeddedCa1Modulus[GcCrypto::GcCa1PublicModulusLength] = {
    0xaf,0xb6,0xa0,0x1f,0x7f,0x8c,0xec,0xd6,0x62,0xc9,0xf9,0x83,0x69,0x4f,0x0e,0x8e,0xc3,0x71,0x70,0x60,0x63,0xeb,0x90,0x2f,0x1e,0x3b,0xa2,0xcd,0xd5,0x7a,0xaa,0x45,
    0x27,0x61,0xf9,0x10,0xc0,0x0c,0x72,0xe7,0xbd,0x70,0x2e,0x32,0xa0,0xd0,0x8a,0x29,0x43,0x19,0x12,0x8c,0x8d,0x10,0xe4,0x04,0xe0,0x13,0x7f,0x26,0x02,0x1b,0xfd,0x44,
    0xaf,0x70,0xd7,0xbf,0xdf,0x97,0xd2,0x34,0xfd,0xbb,0x51,0x8d,0x7c,0x04,0x9d,0x30,0xff,0xb1,0xb4,0xd3,0xef,0x2c,0xee,0xad,0x4c,0x4a,0x26,0x94,0x15,0x13,0xa9,0xda,
    0xf4,0xa4,0x22,0xe7,0x11,0x8a,0xe4,0xb0,0xe9,0x66,0x23,0xa7,0xed,0x7d,0x73,0x8b,0x32,0xe5,0xe1,0x19,0x34,0x15,0x06,0x5d,0xa6,0xcd,0x80,0xa5,0xc0,0xd0,0xce,0x7c,
    0x3e,0x3d,0x1f,0x2b,0x65,0x26,0xbe,0xae,0x55,0xc7,0x03,0xcf,0x4a,0xd3,0xda,0x54,0x13,0x1f,0x20,0x05,0xe9,0x9b,0x3f,0xde,0x73,0xd3,0xa0,0xff,0xa0,0x7e,0xa9,0x6a,
    0xbc,0xf8,0x6c,0xf1,0x3c,0x72,0x4e,0x6f,0xa1,0x3c,0x20,0xcd,0x3a,0x1a,0x65,0xe0,0xf0,0xf8,0x84,0xeb,0x6b,0x38,0x49,0xb6,0xf2,0x5b,0x81,0x16,0x8b,0x1a,0xe0,0x4f,
    0x18,0x88,0xf1,0xad,0x66,0xa9,0xc8,0xe0,0x9d,0xd0,0x9b,0xfa,0xad,0xbe,0xc9,0x5c,0xe5,0x54,0x2a,0xf4,0x59,0xe2,0xfa,0xd4,0xc9,0x58,0x1c,0x83,0xd9,0x23,0x77,0xdc,
    0x78,0xbe,0xca,0x7b,0xc1,0x69,0xf3,0x9c,0xfe,0xf7,0xb9,0x9e,0xd6,0x44,0x70,0x1c,0x8b,0x08,0xb0,0x44,0xe5,0x63,0xfb,0xb8,0x45,0x40,0xda,0xa2,0x3c,0xb9,0xfb,0x75
};
char EmbeddedDataHolder::g_LibraryEmbeddedCa1PublicExponent[GcCrypto::GcCa1PublicExponentLength] = {
    0x01,0x00,0x01
};

// ドライバ直持ちの暗号化された鍵類
const EmbeddedDataHolder::ConcatenatedGcLibraryEmbeddedKeys EmbeddedDataHolder::g_LibraryLocalEmbeddedGcLibKeys[2] = {
    {
        {0x98,0x42,0xD1,0x45,0x92,0xEE,0x79,0xAE,0xE3,0xAA,0xC9,0xEA,0x6A,0x67,0xC4,0xB4,0x5E,0x18,0x1E,0x0C,0xC0,0xA2,0x1C,0x0E,0x05,0xA4,0x49,0x30,0x53,0x7F,0xC8,0xE2},
        {0xB9,0xFB,0x97,0x31,0x0A,0x4E,0x28,0xE7,0x1E,0x69,0x8C,0xEE,0xED,0x26,0x20,0x14,0x63,0x76,0xBC,0x1D,0x86,0xED,0x11,0x01,0x4D,0xB0,0xFC,0x88,0xD4,0x64,0x15,0x03},
        {0x68,0x95,0x4D,0x5A,0x87,0x57,0x81,0xB6,0x6C,0xD1,0xEF,0x40,0x9D,0x74,0xF1,0xA5},
        {0xDA,0xCA,0x1F,0x3E,0x78,0x96,0xCA,0x2F,0x1A,0x47,0xA3,0x19,0x47,0x47,0xC4,0x54,0x5C,0x97,0x02,0x74,0xF2,0x69,0xA2,0x14,0x46,0xFC,0x5B,0x21,0x85,0x29,0xCB,0x16},
    },
    {
        {0xE8,0x17,0xE6,0x0B,0xE2,0x6C,0x32,0x30,0x45,0xF7,0xBA,0x8D,0xBD,0x99,0x15,0x62,0xD1,0x1C,0x1C,0x2C,0x42,0xC1,0x2E,0x1B,0x4A,0xF1,0x65,0x3B,0x0D,0x37,0xF3,0xC6},
        {0x91,0xB2,0x5C,0x22,0xB9,0x47,0xF1,0x15,0xB0,0xEE,0x16,0xC5,0x3F,0xCC,0x58,0xD6,0xA6,0xAC,0x06,0x47,0x3A,0xA0,0x9B,0x12,0xE2,0x50,0x80,0x13,0x49,0x2C,0x3C,0xED},
        {0x35,0x5B,0xA3,0x6D,0x26,0x1E,0xF6,0xC9,0xFA,0xD2,0x43,0x81,0x5A,0xD0,0x22,0x75},
        {0x78,0x5A,0x92,0xE4,0x91,0x49,0xD8,0x28,0x57,0x57,0x49,0x68,0x01,0x0E,0xA4,0x10,0x5B,0x05,0x47,0x03,0xD1,0x1B,0xA7,0xCA,0xD9,0x06,0x10,0x02,0x85,0xA3,0x99,0x26},
    }
};

// 暗号化関連の値
namespace {
    // ドライバ直持ちの鍵類を復号化するための鍵（spl で復号化する： dev/prod で結果が異なり、所望の復号化鍵となる）
    static const char g_LibraryLocalEmbeddedSplEncryptedKek[GcCrypto::GcAesKeyLength] = {
        0x42,0xF1,0xEB,0xCB,0xDD,0xED,0x82,0xAF,0x32,0x4E,0x0D,0xF4,0x84,0xF2,0xAB,0x57
    };
    // NOTE: 鍵類を変更する場合は IV も変える事！
    // ドライバ直持ちの鍵類を復号化する際の IV
    static const char g_LibraryLocalEmbeddedIvForKek[2][GcCrypto::GcAesCbcIvLength] = {
        {0xc6,0x09,0x0a,0x32,0x51,0xa5,0x26,0xec,0x8f,0x2b,0xa9,0x3e,0xcc,0x62,0xf0,0x92},
        {0x23,0xa6,0xfe,0x53,0xe7,0x16,0x86,0xfb,0x9b,0xb5,0x72,0x32,0x42,0x01,0xa3,0xc7}
    };
}

// 使用後必ず pOutKeys を detail::SecureMemoryZero で初期化すること！
nn::Result EmbeddedDataHolder::DecryptoEmbeddedKeys(ConcatenatedGcLibraryEmbeddedKeys* pOutKeys, size_t outKeySize)
{
    // 念のため dev かどうかも逐一確認する
    g_IsDev = static_cast<int>(GcCrypto::CheckDevelopmentSpl());
    // 復号化の対象鍵を選択する
    ConcatenatedGcLibraryEmbeddedKeys keys = g_LibraryLocalEmbeddedGcLibKeys[static_cast<int>(IsDev())];
    const char* emmcEmbeddedIvForKek = g_LibraryLocalEmbeddedIvForKek[static_cast<int>(IsDev())];

    char emmcEmbeddedKek[GcCrypto::GcAesKeyLength] = {0};
    nn::crypto::Aes128CtrDecryptor aes128CtrDecryptor;
    NN_UTIL_SCOPE_EXIT
    {
        // メモリから emmcEmbeddedKek を消去
        detail::SecureMemoryZero(emmcEmbeddedKek, sizeof(emmcEmbeddedKek));
        // rasEncryptor 内部に残っている値を 0 で上書き
        aes128CtrDecryptor.Initialize(emmcEmbeddedKek, sizeof(emmcEmbeddedKek), emmcEmbeddedIvForKek, GcCrypto::GcAesCbcIvLength);
    };

    // 失敗するケースはハンドリングせずここで殺す(fatal がエラーレポートに残す)
    static const int KeyGeneration = 4;// 鍵世代更新時はここの値を変更する
    NN_RESULT_DO( GcCrypto::DecryptAesKeySpl(emmcEmbeddedKek, sizeof(emmcEmbeddedKek),
        reinterpret_cast<const void*>(g_LibraryLocalEmbeddedSplEncryptedKek), sizeof(g_LibraryLocalEmbeddedSplEncryptedKek), KeyGeneration, 0) );

    aes128CtrDecryptor.Initialize(emmcEmbeddedKek, sizeof(emmcEmbeddedKek), emmcEmbeddedIvForKek, GcCrypto::GcAesCbcIvLength);
    aes128CtrDecryptor.Update(pOutKeys, outKeySize, &keys, sizeof(keys));
    NN_RESULT_SUCCESS;
}

nn::Result EmbeddedDataHolder::DecryptGcMessageSecure(
                                size_t*     pOutResultSize,
                                void*       pResultBuffer,
                                size_t      resultBufferSize,
                                const void* pCipher,
                                size_t      cipherSize) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    ConcatenatedGcLibraryEmbeddedKeys keys;
    NN_UTIL_SCOPE_EXIT
    {
        // メモリから keys を消去
        detail::SecureMemoryZero(&keys, sizeof(keys));
    };
    NN_RESULT_DO(DecryptoEmbeddedKeys(&keys, sizeof(keys)));

    NN_RESULT_DO( GcCrypto::DecryptGcMessageSpl(pOutResultSize, pResultBuffer, resultBufferSize, pCipher, cipherSize, g_EmmcEmbeddedSocRsaKeyModulus,
    sizeof(g_EmmcEmbeddedSocRsaKeyModulus), keys.encRsaOaepLabelHash, sizeof(keys.encRsaOaepLabelHash)) );
    NN_RESULT_SUCCESS;
}

nn::Result EmbeddedDataHolder::EncryptGcMessageSecure(
                                char* outBuffer, const size_t outBufferLength,
                                const char* inBuffer, const size_t inBufferLength,
                                char* keyModulus, const size_t keyModulusLength,
                                char* keyExponent, const size_t keyExponentLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    ConcatenatedGcLibraryEmbeddedKeys keys;
    nn::crypto::RsaOaepEncryptor<GcCrypto::GcRsaKeyLength, nn::crypto::Sha256Generator> rsaEncryptor;
    NN_UTIL_SCOPE_EXIT
    {
        // メモリから keys を消去
        detail::SecureMemoryZero(&keys, sizeof(keys));
        rsaEncryptor.Initialize(keyModulus, keyModulusLength, keyExponent, keyExponentLength);
        // rasEncryptor 内部に残っている値を 0 で上書き
        rsaEncryptor.SetLabelDigest(keys.encRsaOaepLabelHash, sizeof(keys.encRsaOaepLabelHash));
    };
    NN_RESULT_DO(DecryptoEmbeddedKeys(&keys, sizeof(keys)));

    // セッション構築時の乱数交換のときのみの使用前提
    char seed[GcRsaOaepSeedSize];
    memset(seed, 0, sizeof(seed));
    rsaEncryptor.Initialize(keyModulus, keyModulusLength, keyExponent, keyExponentLength);
    rsaEncryptor.SetLabelDigest(keys.encRsaOaepLabelHash, sizeof(keys.encRsaOaepLabelHash));
    rsaEncryptor.Encrypt(outBuffer, outBufferLength, inBuffer, inBufferLength, seed, sizeof(seed));
    NN_RESULT_SUCCESS;
}

nn::Result EmbeddedDataHolder::GetHmacForGenerateRandomValues(
                                char* pOutResultBuffer, const size_t resultBufferLength,
                                const char* randomValue1Buffer, const size_t randomValue1BufferLength,
                                const char* randomValue2Buffer, const size_t randomValue2BufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(resultBufferLength == GcCrypto::GcSha256Length &&randomValue1BufferLength == GcRandomValueSize && randomValue2BufferLength == GcRandomValueSize);
    ConcatenatedGcLibraryEmbeddedKeys keys;
    nn::crypto::HmacSha256Generator hmacSha256Generator;
    NN_UTIL_SCOPE_EXIT
    {
        // メモリから keys を消去
        detail::SecureMemoryZero(&keys, sizeof(keys));
        // hmacSha256Generator 内部に残っている値を 0 で上書き
        hmacSha256Generator.Initialize(keys.encHmacKeyForKeyAndIv, sizeof(keys.encHmacKeyForKeyAndIv));
    };
    NN_RESULT_DO(DecryptoEmbeddedKeys(&keys, sizeof(keys)));
    // HMAC-SHA256生成器の初期化
    hmacSha256Generator.Initialize(keys.encHmacKeyForKeyAndIv, sizeof(keys.encHmacKeyForKeyAndIv));

    // 乱数RND1, RND2 あるいは RND6, RND5M が渡されるので、それを入れる
    hmacSha256Generator.Update(randomValue1Buffer, randomValue1BufferLength);
    hmacSha256Generator.Update(randomValue2Buffer, randomValue2BufferLength);

    // HMAC-SHA256 を計算
    hmacSha256Generator.GetMac(pOutResultBuffer, resultBufferLength);
    NN_RESULT_SUCCESS;
}

// 埋め込み鍵を復号化して CV 用 HMAC 鍵・ CV 固定値を取得
nn::Result EmbeddedDataHolder::SetLibraryEmbeddedKeys() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    ConcatenatedGcLibraryEmbeddedKeys keys;
    NN_UTIL_SCOPE_EXIT
    {
        // keys は全てクリアする
        detail::SecureMemoryZero(&keys, sizeof(keys));
    };
    NN_RESULT_DO(DecryptoEmbeddedKeys(&keys, sizeof(keys)));

    // 鍵を代入する
    memcpy(g_LibraryEmbeddedHmacKeyForCv, keys.encHmacKeyForCv, sizeof(keys.encHmacKeyForCv));
    memcpy(g_LibraryEmbeddedCvConstantValue, keys.encCvConstantValue, sizeof(keys.encCvConstantValue));

    // FW を選択する
    SelectAsicFirmware(IsDev());

    NN_RESULT_SUCCESS;
}

nn::Result EmbeddedDataHolder::SetEmmcEmbeddedKeys(const char* encryptedKeyBuffer, const size_t encryptedKeyBufferSize,
                                                   const char* socCertBuffer, const size_t socCertBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    NN_DETAIL_GC_SDK_REQUIRES(socCertBufferSize == GcCertificateSize);

    // SoC 秘密鍵を復号化して spl 内部でセット
    NN_RESULT_DO( GcCrypto::DecryptAndStoreGcKeySpl(encryptedKeyBuffer, encryptedKeyBufferSize) );

    // Cert を GcCrypto に設定
    SetSocCertificate(socCertBuffer, socCertBufferSize);

    // Cert をパースして Modulus を入手
    SetSocRsaKeyModulus(socCertBuffer + nn::gc::detail::GcSocModulusOffset, nn::gc::detail::GcCrypto::GcRsaKeyLength);

    NN_RESULT_SUCCESS;
}

void EmbeddedDataHolder::SetSocRsaKeyModulus(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(bufferLength == sizeof(g_EmmcEmbeddedSocRsaKeyModulus));
    memcpy(g_EmmcEmbeddedSocRsaKeyModulus, buffer, bufferLength);
}

void EmbeddedDataHolder::SetSocCertificate(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(bufferLength == sizeof(g_EmmcEmbeddedSocCertificate));
    memcpy(g_EmmcEmbeddedSocCertificate, buffer, bufferLength);
}

#ifdef NN_GC_ENABLE_OVERWRITE_EMBEDDED_DATA
void EmbeddedDataHolder::SetHmacKeyForCvForDebug(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(bufferLength == sizeof(g_LibraryEmbeddedHmacKeyForCv));
    NN_DETAIL_GC_DETAIL_LOG("Loaded HMAC key for CV manually\n");
    memcpy(g_LibraryEmbeddedHmacKeyForCv, buffer, bufferLength);
}

void EmbeddedDataHolder::SetCvConstantValueForDebug(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(bufferLength == sizeof(g_LibraryEmbeddedCvConstantValue));
    NN_DETAIL_GC_DETAIL_LOG("Loaded constant value for CV manually\n");
    memcpy(g_LibraryEmbeddedCvConstantValue, buffer, bufferLength);
}

void EmbeddedDataHolder::SetCa1Modulus(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(bufferLength == sizeof(g_LibraryEmbeddedCa1Modulus));
    NN_DETAIL_GC_DETAIL_LOG("Loaded CA1 modulus manually\n");
    memcpy(g_LibraryEmbeddedCa1Modulus, buffer, bufferLength);
}

void EmbeddedDataHolder::SetCa1PublicExponent(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(bufferLength == sizeof(g_LibraryEmbeddedCa1PublicExponent));
    NN_DETAIL_GC_DETAIL_LOG("Loaded CA1 public exponent manually\n");
    memcpy(g_LibraryEmbeddedCa1PublicExponent, buffer, bufferLength);
}
#endif


} } }
