﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gc/detail/gc_Log.h>

#include <nn/gc/detail/gc_Define.h>
#include <nn/gc/detail/gc_DataIo.h>
#include <nn/gc/detail/gc_AsicOperation.h>
#include <nn/gc/detail/gc_AsicRegister.h>
#include <nn/gc/detail/gc_GcCrypto.h>
#include <nn/gc/detail/gc_EmbeddedDataHolder.h>
#include <nn/gc/detail/gc_StateMachine.h>

namespace nn { namespace gc {
namespace detail {

// 静的メンバ
char* AsicOperation::g_WorkAlignedBuffer;
size_t AsicOperation::g_WorkAlignedBufferLength;

//--------------------------------------------------------------------------
/*
 * AsicOperation クラスの関数
 */

AsicOperation& AsicOperation::GetInstance() NN_NOEXCEPT
{
    static AsicOperation s_Instance;
    return s_Instance;
}

void AsicOperation::Initialize(char* workAlignedBuffer, const size_t workAlignedBufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_LOG("AsicOperation Initialization\n");

    AsicOperation::g_WorkAlignedBuffer = workAlignedBuffer;
    AsicOperation::g_WorkAlignedBufferLength = workAlignedBufferLength;
}

//--------------------------------------------------------------------------
/*
 * Operation コマンド発行用の API
 */

nn::Result AsicOperation::SendFirmware() NN_NOEXCEPT
{
    StateMachine& state = StateMachine::GetInstance();
    char* firmwareBuffer = EmbeddedDataHolder::g_FwReadBuffer;
    if(state.IsWriterFwSet())
    {
        firmwareBuffer = EmbeddedDataHolder::g_FwWriterBuffer;
    }
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessWriteOperation(firmwareBuffer, GcAsicFirmwareSize, AsicOperationIndex_TRANS_FW), nn::fs::ResultGameCardSendFirmwareFailure());
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::ReceiveCertificate(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(outBuffer, bufferLength, AsicOperationIndex_RD_ASICPUBKEY), nn::fs::ResultGameCardReceiveCertificateFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendSocCertificate(const char* certBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessWriteOperation(certBuffer, bufferLength, AsicOperationIndex_WR_SOCPUBKEY), nn::fs::ResultGameCardSendSocCertificateFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::ReceiveRandomValue(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(outBuffer, bufferLength, AsicOperationIndex_RD_RND1), nn::fs::ResultGameCardReceiveRandomValueFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendRandomValue(const char* dataBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessWriteOperation(dataBuffer, bufferLength, AsicOperationIndex_WR_RND2), nn::fs::ResultGameCardSendRandomValueFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::ReceiveDeviceChallenge(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(outBuffer, bufferLength, AsicOperationIndex_RD_RND3), nn::fs::ResultGameCardReceiveDeviceChallengeFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::RespondDeviceChallenge(const char* hashBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessWriteOperation(hashBuffer, bufferLength, AsicOperationIndex_WR_RESPONSE1), nn::fs::ResultGameCardRespondDeviceChallengeFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendHostChallenge(const char* randomValueBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessWriteOperation(randomValueBuffer, bufferLength, AsicOperationIndex_WR_RND4), nn::fs::ResultGameCardSendHostChallengeFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::ReceiveChallengeResponse(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(outBuffer, bufferLength, AsicOperationIndex_RD_RESPONSE2), nn::fs::ResultGameCardReceiveChallengeResponseFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::ChangeModeToSecure() NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessNoDataOperation(AsicOperationIndex_CHG_SECURE), nn::fs::ResultGameCardChangeModeToSecureFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::EnableCardBus() NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessNoDataOperation(AsicOperationIndex_CARD_BUS_ENABLE), nn::fs::ResultGameCardEnableCardBusFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::ExchangeRandomValuesInSecureMode(char* outBuffer, const size_t receiveBufferLength, const char* randomValueBuffer, const size_t sendBufferLength) NN_NOEXCEPT
{
    // 送信バッファは長さ違いを許容しない・受信バッファは大目の指定を許容する
    NN_DETAIL_GC_SDK_REQUIRES(sendBufferLength == GcRandomValueForKeyUpdateSocSize && receiveBufferLength >= GcRandomValueSize);

    // バッファを用意する
    SetAsicOperationIndex(AsicOperationIndex_EXCHANGE_KEY);

    memcpy(m_AsicOperationBuffer + GcAsicOperationDataStartOffset, randomValueBuffer, GcRandomValueForKeyUpdateSocSize);

    // 乱数を送って受け取る
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(outBuffer, receiveBufferLength, m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
        nn::fs::ResultGameCardExchangeRandomValuesFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::GetCardHeader(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(outBuffer, bufferLength, AsicOperationIndex_CARD_INIT), nn::fs::ResultGameCardGetCardHeaderFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::ChangeGcModeToSecure(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(outBuffer, bufferLength, AsicOperationIndex_CARD_CHG_SECURE), nn::fs::ResultGameCardChangeGcModeToSecureFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::ChangeGcModeToDebug(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(outBuffer, bufferLength, AsicOperationIndex_CARD_INIT_DEBUG), nn::fs::ResultGameCardChangeGcModeToDebugFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::ReadRmaInformation(RmaInformation* pOutInformation) NN_NOEXCEPT
{
    char buffer[GcPageSize];
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(buffer, sizeof(buffer), AsicOperationIndex_RD_RMA), nn::fs::ResultGameCardReadRmaInfoFailure() );
    memcpy(pOutInformation, buffer, sizeof(buffer));
    NN_RESULT_SUCCESS;
}


//--------------------------------------------------------------------------
/*
 * レジスタ操作
 */

nn::Result AsicOperation::WriteRegister(u32 address, u32 value) NN_NOEXCEPT
{
    // バッファを用意する
    SetAsicOperationIndex(AsicOperationIndex_WRITE_REG);

    // レジスタアドレスの指定（構造体のアドレスから変換して得た、ブリッジASIC上のアドレス値をバッファにセット）
    MemcpyFromU32(m_AsicOperationBuffer + GcAsicOperationRegisterAddressOffset, address);

    // データの書き込み
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessWriteOperation(reinterpret_cast<char*>(&value), sizeof(value), m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
        nn::fs::ResultGameCardWriteRegisterFailure());
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::ReadRegister() NN_NOEXCEPT
{
    // データの受け取り
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(AsicRegister::GetInstance().registerArray, AsicRegister::GcAsicRegisterSize, AsicOperationIndex_READ_REG),
        nn::fs::ResultGameCardReadRegisterFailure());
    NN_RESULT_SUCCESS;
}

//--------------------------------------------------------------------------
/*
 * ゲームカードに対してコマンドを投げるAPI
 */

nn::Result AsicOperation::SendCardReadId1(CardId1* pOutId1) NN_NOEXCEPT
{
    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    CardCommandId command;
    switch(StateMachine::GetInstance().GetGameCardMode())
    {
    case GameCardMode_Normal:
        {
            command = CardCommandId_NormalReadId1;
        }
        break;
    case GameCardMode_Secure:
        {
            command = CardCommandId_SecureReadId1;
        }
        break;
    case GameCardMode_Debug:
        {
            command = CardCommandId_DebugReadId1;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    commandBuffer.GetCommandBufferWithNoData(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), command, DataPageSize_4Byte);
    // データの読み取り
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(reinterpret_cast<char*>(pOutId1), sizeof(CardId1), m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
                                        nn::fs::ResultGameCardSendCardReadId1Failure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendCardReadId2(CardId2* pOutId2) NN_NOEXCEPT
{
    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    CardCommandId command;
    switch(StateMachine::GetInstance().GetGameCardMode())
    {
    case GameCardMode_Normal:
        {
            command = CardCommandId_NormalReadId2;
        }
        break;
    case GameCardMode_Secure:
        {
            command = CardCommandId_SecureReadId2;
        }
        break;
    case GameCardMode_Debug:
        {
            command = CardCommandId_DebugReadId2;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    commandBuffer.GetCommandBufferWithNoData(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), command, DataPageSize_4Byte);
    // データの読み取り
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(reinterpret_cast<char*>(pOutId2), sizeof(CardId2), m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
                                        nn::fs::ResultGameCardSendCardReadId2Failure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendCardReadId3(CardId3* pOutId3) NN_NOEXCEPT
{
    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    CardCommandId command;
    switch(StateMachine::GetInstance().GetGameCardMode())
    {
    case GameCardMode_Normal:
        {
            command = CardCommandId_NormalReadId3;
        }
        break;
    case GameCardMode_Secure:
        {
            command = CardCommandId_SecureReadId3;
        }
        break;
    case GameCardMode_Debug:
        {
            command = CardCommandId_DebugReadId3;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    commandBuffer.GetCommandBufferWithNoData(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), command, DataPageSize_4Byte);
    // データの読み取り
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(reinterpret_cast<char*>(pOutId3), sizeof(CardId3), m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
                                        nn::fs::ResultGameCardSendCardReadId3Failure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendCardReadUid(CardUid* pOutCardUid) NN_NOEXCEPT
{
    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    switch(StateMachine::GetInstance().GetGameCardMode())
    {
    case GameCardMode_Secure:
        {
            commandBuffer.GetCommandBufferWithNoData(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), CardCommandId_SecureReadUid, DataPageSize_64Byte);
        }
        break;
    case GameCardMode_Debug:
        {
            commandBuffer.GetCommandBufferWithNoData(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), CardCommandId_DebugReadUid, DataPageSize_64Byte);
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    // データの読み取り
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(reinterpret_cast<char*>(pOutCardUid), sizeof(CardUid), m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
                                        nn::fs::ResultGameCardSendCardReadUidFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendCardReadPage(char* outAlignedDataBuffer, const size_t bufferLength, const u32 pageAddress, const u32 pageCount) NN_NOEXCEPT
{
    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    switch(StateMachine::GetInstance().GetGameCardMode())
    {
    case GameCardMode_Secure:
        {
            commandBuffer.GetCommandBufferWithPageRead(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), CardCommandId_SecureReadPage, pageAddress, pageCount);
        }
        break;
    // 状態設定前に CardHeader を読むので、Normal 以外もここに入れることにする
    default:
        {
            commandBuffer.GetCommandBufferWithPageRead(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), CardCommandId_NormalReadPage, pageAddress, pageCount);
        }
        break;
    }
    // データの読み取り
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperationWithAlignedBuffer(outAlignedDataBuffer, bufferLength, m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
                                        nn::fs::ResultGameCardSendCardReadPageFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendCardReadPageWithNotAlignedBuffer(char* outDataBuffer, const size_t bufferLength, const u32 pageAddress, const u32 pageCount) NN_NOEXCEPT
{
    NN_DETAIL_GC_SDK_REQUIRES(bufferLength >= pageCount * GcPageSize);
    size_t readSize = pageCount * GcPageSize;
    NN_DETAIL_GC_SDK_REQUIRES(readSize <= g_WorkAlignedBufferLength);

    // カウンタを合わせるため、 Read 時は Decrypt までワンセットでやってしまう
    nn::Result result = SendCardReadPage(g_WorkAlignedBuffer, readSize, pageAddress, pageCount);
    DecryptReadData(g_WorkAlignedBuffer, readSize);
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( result, nn::fs::ResultGameCardSendCardReadPageBufferFailure() );

    memcpy(outDataBuffer, g_WorkAlignedBuffer, readSize);
    NN_RESULT_SUCCESS;
}


nn::Result AsicOperation::SendCardWritePage(char* workAlignedDataBuffer, const size_t bufferLength, const u32 pageAddress, const u32 pageCount) NN_NOEXCEPT
{
    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    switch(StateMachine::GetInstance().GetGameCardMode())
    {
    case GameCardMode_Secure:
        {
            commandBuffer.GetCommandBufferWithPageWrite(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), CardCommandId_SecureWritePage, pageAddress, pageCount);
        }
        break;
    case GameCardMode_Debug:
        {
            commandBuffer.GetCommandBufferWithPageWrite(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), CardCommandId_DebugWritePage, pageAddress, pageCount);
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    // データの書き込み
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessWriteOperationWithAlignedBuffer(workAlignedDataBuffer, bufferLength, m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
                                        nn::fs::ResultGameCardSendCardWritePageFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendCardRefresh(RefreshResponse* pOutRefreshResponse) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    NN_RESULT_THROW_UNLESS(StateMachine::GetInstance().GetGameCardMode() == GameCardMode_Secure, nn::fs::ResultGameCardStateCardSecureModeRequired());
    commandBuffer.GetCommandBufferWithNoData(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), CardCommandId_SecureReadRefresh, DataPageSize_64Byte);
    // データの読み取り
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(reinterpret_cast<char*>(pOutRefreshResponse), sizeof(RefreshResponse), m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
                                        nn::fs::ResultGameCardSendCardRefreshFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendCardSetKey() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    NN_RESULT_THROW_UNLESS(StateMachine::GetInstance().GetGameCardMode() == GameCardMode_Secure, nn::fs::ResultGameCardStateCardSecureModeRequired());
    commandBuffer.GetCommandBufferWithNoData(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), CardCommandId_SecureSetKey, DataPageSize_NONE);
    // データの読み取り
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessNoDataOperation(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
                                        nn::fs::ResultGameCardSendCardSetKeyFailure() );
    NN_RESULT_SUCCESS;
}


nn::Result AsicOperation::SendCardSelfRefresh(SelfRefreshResponse* pOutRefreshResponse) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    NN_RESULT_THROW_UNLESS(StateMachine::GetInstance().GetGameCardMode() == GameCardMode_Normal, nn::fs::ResultGameCardStateCardNormalModeRequired());
    commandBuffer.GetCommandBufferWithNoData(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), CardCommandId_NormalReadSelfRefresh, DataPageSize_4Byte);
    // データの読み取り
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(reinterpret_cast<char*>(pOutRefreshResponse), sizeof(SelfRefreshResponse), m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
                                        nn::fs::ResultGameCardSendCardSelfRefreshFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendCardReadRefreshStatus(SelfRefreshStatus* pOutSelfRefreshStatus) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    NN_RESULT_THROW_UNLESS(StateMachine::GetInstance().GetGameCardMode() == GameCardMode_Normal, nn::fs::ResultGameCardStateCardNormalModeRequired());
    commandBuffer.GetCommandBufferWithNoData(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), CardCommandId_NormalReadRefreshStatus, DataPageSize_4Byte);
    // データの読み取り
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(reinterpret_cast<char*>(pOutSelfRefreshStatus), sizeof(SelfRefreshStatus), m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
                                        nn::fs::ResultGameCardSendCardReadRefreshStatusFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendCardReadCrc(uint32_t* pCrc, size_t crcLength, const u32 pageAddress, const u32 pageCount) NN_NOEXCEPT
{
    NN_DETAIL_GC_SDK_REQUIRES(crcLength >= sizeof(uint32_t));
    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    NN_RESULT_THROW_UNLESS(StateMachine::GetInstance().GetGameCardMode() == GameCardMode_Debug, nn::fs::ResultGameCardStateCardDebugModeRequired());
    commandBuffer.GetCommandBufferDirectly(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), DataPageSize_4Byte, DataTransDirectionForCard_Read, 0, CardCommandId_DebugReadCrc, pageAddress, pageCount);
    // データの読み取り
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(reinterpret_cast<char*>(pCrc), sizeof(uint32_t), m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
                                        nn::fs::ResultGameCardSendCardReadCrcFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendCardErase() NN_NOEXCEPT
{
    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    NN_RESULT_THROW_UNLESS(StateMachine::GetInstance().GetGameCardMode() == GameCardMode_Debug, nn::fs::ResultGameCardStateCardDebugModeRequired());
    commandBuffer.GetCommandBufferWithNoData(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), CardCommandId_DebugReadErase, DataPageSize_4Byte);
    // データの読み取り
    CardErrorState cardErrorState = CardErrorState_NoError;
    SetCmd60TimeoutForErase();
    NN_UTIL_SCOPE_EXIT
    {
        SetCmd60TimeoutDefault();
    };
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW(ProcessReadOperation(reinterpret_cast<char*>(&cardErrorState), sizeof(CardErrorState), m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
        nn::fs::ResultGameCardSendCardEraseFailure());
    if(cardErrorState != CardErrorState_NoError)
    {
        return nn::fs::ResultGameCardDebugEraseFailure();
    }
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendCardReadParameter(DevCardParameter* pOutDevCardParameter) NN_NOEXCEPT
{
    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    NN_RESULT_THROW_UNLESS(StateMachine::GetInstance().GetGameCardMode() == GameCardMode_Debug, nn::fs::ResultGameCardStateCardDebugModeRequired());
    commandBuffer.GetCommandBufferDirectly(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), DataPageSize_512Byte, DataTransDirectionForCard_Read, sizeof(DevCardParameter) / GcPageSize, CardCommandId_DebugReadPara, 0, 0);
    // データの読み取り
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperation(reinterpret_cast<char*>(pOutDevCardParameter), sizeof(DevCardParameter), m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
                                        nn::fs::ResultGameCardSendCardReadParameterFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendCardWriteParameter(const DevCardParameter& devCardParameter) NN_NOEXCEPT
{
    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    NN_RESULT_THROW_UNLESS(StateMachine::GetInstance().GetGameCardMode() == GameCardMode_Debug, nn::fs::ResultGameCardStateCardDebugModeRequired());
    commandBuffer.GetCommandBufferDirectly(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), DataPageSize_512Byte, DataTransDirectionForCard_Write, sizeof(DevCardParameter) / GcPageSize, CardCommandId_DebugWritePara, 0, 0);
    // データの読み取り
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessWriteOperation(reinterpret_cast<const char*>(&devCardParameter), sizeof(DevCardParameter), m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
                                        nn::fs::ResultGameCardSendCardWriteParameterFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendCardReadErrorCount(DevCardErrorCount* pOutErrCount, const u32 pageAddress) NN_NOEXCEPT
{
    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    NN_RESULT_THROW_UNLESS(StateMachine::GetInstance().GetGameCardMode() == GameCardMode_Debug, nn::fs::ResultGameCardStateCardDebugModeRequired());
    commandBuffer.GetCommandBufferDirectly(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer), DataPageSize_64Byte, DataTransDirectionForCard_Read, sizeof(DevCardErrorCount) / GcPageSize, CardCommandId_DebugReadErcnt, pageAddress, 0);
    // データの読み取り
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( ProcessReadOperationWithAlignedBuffer(reinterpret_cast<char*>(pOutErrCount), sizeof(DevCardErrorCount), m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
                                        nn::fs::ResultGameCardSendCardReadErrorCountFailure() );
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::SendCardChangeDebugDirect() NN_NOEXCEPT
{
    // バッファを用意する
    AsicOperationCardCommandBuffer commandBuffer;
    NN_RESULT_THROW_UNLESS(StateMachine::GetInstance().GetGameCardMode() == GameCardMode_Normal, nn::fs::ResultGameCardStateCardDebugModeRequired());
    // CHG_DEBUG だけ他のコマンドとフォーマットが違うため決め打ちで入れる
    const char CommandForChgDebug[] =
    {
            0x10, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2E, 0xB7, 0xD0, 0x1B, 0xF1, 0x06, 0x94, 0x56,
            0xF9, 0x3B, 0x38, 0xD5, 0x22, 0x16, 0xF2, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };
    memset(m_AsicOperationBuffer, 0x00, sizeof(m_AsicOperationBuffer));
    memcpy(m_AsicOperationBuffer, CommandForChgDebug, sizeof(CommandForChgDebug));
    // データの読み取り
    CardErrorState cardErrorState = CardErrorState_NoError;
    char outBuffer[4];
    NN_DETAIL_GC_RESULT_DO_LOG_RETHROW(ProcessReadOperation(outBuffer, sizeof(outBuffer), m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer)),
        nn::fs::ResultGameCardSendCardEraseFailure());
    cardErrorState = static_cast<CardErrorState>(outBuffer[0]);
    if(cardErrorState != CardErrorState_NoError)
    {
        return nn::fs::ResultGameCardDebugEraseFailure();
    }
    NN_RESULT_SUCCESS;
}

// ここから Private API

//--------------------------------------------------------------------------
/*
 * Index のみ必要な Operation API
 */

nn::Result AsicOperation::ProcessWriteOperation(const char* inDataBuffer, const size_t bufferLength, const AsicOperationIndex index) NN_NOEXCEPT
{
    SetAsicOperationIndex(index);
    return ProcessWriteOperation(inDataBuffer, bufferLength, m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer));
}

nn::Result AsicOperation::ProcessReadOperation(char* outDataBuffer, const size_t bufferLength, const AsicOperationIndex index) NN_NOEXCEPT
{
    SetAsicOperationIndex(index);
    return ProcessReadOperation(outDataBuffer, bufferLength, m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer));
}

nn::Result AsicOperation::ProcessNoDataOperation(const AsicOperationIndex index) NN_NOEXCEPT
{
    SetAsicOperationIndex(index);
    return ProcessNoDataOperation(m_AsicOperationBuffer, sizeof(m_AsicOperationBuffer));
}

//--------------------------------------------------------------------------
/*
 * ASIC へ送るコマンドバッファを直接指定する Operation API
 */

nn::Result AsicOperation::ProcessWriteOperation(const char* inDataBuffer, const size_t bufferLength, const char* operationBuffer, const size_t operationBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_GC_SDK_REQUIRES(g_WorkAlignedBufferLength >= bufferLength);
    // バッファの準備
    size_t workAlignedBufferSectorRoundedUpLength = CalcUnitCeiling<size_t, DataIo::SectorSize>(bufferLength);
    memset(g_WorkAlignedBuffer, GcPaddingU8, workAlignedBufferSectorRoundedUpLength);
    memcpy(g_WorkAlignedBuffer, inDataBuffer, bufferLength);
    NN_RESULT_DO(ProcessWriteOperationWithAlignedBuffer(g_WorkAlignedBuffer, workAlignedBufferSectorRoundedUpLength, operationBuffer, operationBufferSize));
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::ProcessReadOperation(char* outDataBuffer, const size_t bufferLength, const char* operationBuffer, const size_t operationBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_GC_SDK_REQUIRES(g_WorkAlignedBufferLength >= bufferLength);
    // 読み込みデータ受領
    size_t sectorRoundedUpLength = CalcUnitCeiling<size_t, DataIo::SectorSize>(bufferLength);

    // カウンタを合わせるため、 Read 時は Decrypt までワンセットでやってしまう
    nn::Result result = ProcessReadOperationWithAlignedBuffer(g_WorkAlignedBuffer, sectorRoundedUpLength, operationBuffer, operationBufferSize);
    DecryptReadData(g_WorkAlignedBuffer, sectorRoundedUpLength);
    if(result.IsSuccess())
    {
        memcpy(outDataBuffer, g_WorkAlignedBuffer, bufferLength);
    }
    return result;
}

nn::Result AsicOperation::ProcessNoDataOperation(const char* operationBuffer, const size_t operationBufferSize) NN_NOEXCEPT
{
    NN_RESULT_DO(ExecuteOperation(g_WorkAlignedBuffer, GcMmcCmd60DataSize, operationBuffer, operationBufferSize, DataDirection_NoData));
    NN_RESULT_SUCCESS;
}

//--------------------------------------------------------------------------
/*
 * 使用する側でアラインされたバッファに直接 R/W する  Operation API
 */

nn::Result AsicOperation::ProcessWriteOperationWithAlignedBuffer(char* inAlignedDataBuffer, const size_t alignedBufferLength, const char* operationBuffer, const size_t operationBufferSize) NN_NOEXCEPT
{
    NN_RESULT_DO(ExecuteOperation(inAlignedDataBuffer, alignedBufferLength, operationBuffer, operationBufferSize, DataDirection_Write));
    NN_RESULT_SUCCESS;
}

nn::Result AsicOperation::ProcessReadOperationWithAlignedBuffer(char* outAlignedDataBuffer, const size_t alignedBufferLength, const char* operationBuffer, const size_t operationBufferSize) NN_NOEXCEPT
{
    NN_RESULT_DO(ExecuteOperation(outAlignedDataBuffer, alignedBufferLength, operationBuffer, operationBufferSize, DataDirection_Read));
    NN_RESULT_SUCCESS;
}

//--------------------------------------------------------------------------
/*
 * 実コマンドを発行する API
 */

nn::Result AsicOperation::ExecuteOperation(char* workAlignedBuffer, const size_t workAlignedBufferLength, const char* operationBuffer, const size_t operationBufferSize, const DataDirection direction) NN_NOEXCEPT
{
    NN_DETAIL_GC_SDK_REQUIRES(GcMmcCmd60DataSize <= operationBufferSize);
    DataIo& dataIo = DataIo::GetInstance();
    char tmp[GcMmcCmd60DataSize]; // Write 時の先頭 64 Byte 保存場所に使用
    if(direction == DataDirection_Write)
    {
        // g_WorkAlignedBuffer の先頭 64 Byte を tmp に保存
        memcpy(tmp, g_WorkAlignedBuffer, GcMmcCmd60DataSize);
    }

    memcpy(g_WorkAlignedBuffer, operationBuffer, GcMmcCmd60DataSize); // アラインされたワークバッファにコマンドを入れて発行する
    dataIo.SetTransferTimeout(m_SdmmcTimeoutMilliSeconds);
    nn::Result result = dataIo.SendOperationStart(g_WorkAlignedBuffer, GcMmcCmd60DataSize);
    dataIo.SetDefaultTransferTimeout();

    if(direction == DataDirection_Write)
    {
        // g_WorkAlignedBuffer に先頭 64 Byte を tmp から復活
        NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(GcMmcCmd60DataSize <= g_WorkAlignedBufferLength);
        memcpy(g_WorkAlignedBuffer, tmp, GcMmcCmd60DataSize);
    }

    if(result.IsSuccess())
    {
        if(direction == DataDirection_Read)
        {
            result = dataIo.SendCommandDataRead(workAlignedBuffer, workAlignedBufferLength);
        }
        else if(direction == DataDirection_Write)
        {
            result = dataIo.SendCommandDataWrite(workAlignedBuffer, workAlignedBufferLength);
        }
    }
    else
    {
        // リードのみオペレーション 64 byte の failure 時も強制的に読む（result は更新しない）
        // これは「ホスト → デバイス への CMD は必ず届く」という前提のもと行う（CMD60 は届いており、状態遷移は正しくなる）
        // Refresh 要求や CC エラーの時はここを通る
        if(direction == DataDirection_Read)
        {
            nn::Result resultToBeDiscarded = dataIo.SendCommandDataRead(workAlignedBuffer, workAlignedBufferLength);
            NN_UNUSED(resultToBeDiscarded);
        }
    }

    // 通信に失敗していたら明示的に CMD12 を呼ぶ必要がある（result は更新しない）
    if(result.IsFailure())
    {
        nn::Result resultToBeDiscarded = dataIo.AbortGcAsicOperation();
        NN_UNUSED(resultToBeDiscarded);
    }

    // オペレーションの終了処理
    NN_RESULT_DO(dataIo.FinishOperation());
    return result;
}


//--------------------------------------------------------------------------
/*
 * Read したデータを復号化する API
 */

void AsicOperation::DecryptReadData(char* workBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    StateMachine& gcState = StateMachine::GetInstance();
    GcCrypto& gcCrypto = GcCrypto::GetInstance();
    if ( gcState.IsAsicSecure() )
    {
        // セキュアモードなら AES-CTR で復号化
        gcCrypto.DecryptWithAesCtr(workBuffer, bufferLength, workBuffer, bufferLength);
    }
    else if ( gcState.IsCommonKeyReady() )
    {
        // セキュアモードでなくて、共通鍵があるなら AES-CBC で復号化
        gcCrypto.DecryptWithAesCbc(workBuffer, bufferLength, workBuffer, bufferLength);
    }
}

AsicOperation::AsicOperation() NN_NOEXCEPT
{
    SetCmd60TimeoutDefault();
    DataIo::GetInstance().SetDefaultTransferTimeout();
}

} } }
