﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SystemThreadDefinition.h>

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>

#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Log.h>
#include <nn/gc/detail/gc_DataIo.h>
#include <nn/gc/detail/gc_GeneralIo.h>
#include <nn/gc/detail/gc_AsicOperation.h>
#include <nn/gc/detail/gc_AsicRegister.h>
#include <nn/gc/detail/gc_GcCrypto.h>
#include <nn/gc/detail/gc_StateMachine.h>
#include <nn/gc/detail/gc_AsicHandler.h>
#include <nn/gc/detail/gc_AsicHandlerCore.h>
#include <nn/gc/detail/gc_DeviceDetector.h>

namespace nn { namespace gc {
namespace detail {

#define NN_DETAIL_RING_BUFFER_LOCK_GUARD   std::lock_guard<nn::os::Mutex> lock(m_RingBufferMutex);
#define NN_DETAIL_LOCKER_ACCESS_LOCK_GUARD   std::lock_guard<nn::os::Mutex> lock(m_AsicLockerMutex);

namespace {
    static AsicWorkInfo s_AsicWorkList[AsicHandler::GcAsicWorkQueueLength];
}


WorkClientLocker::WorkClientLocker() NN_NOEXCEPT
{
    nn::os::InitializeEvent(&m_WorkFinishEvent, false, nn::os::EventClearMode_ManualClear);
    this->Reset();
}

WorkClientLocker::~WorkClientLocker() NN_NOEXCEPT
{
    nn::os::FinalizeEvent(&m_WorkFinishEvent);
}

void WorkClientLocker::Reset() NN_NOEXCEPT
{
    m_IsUsed = false;
    nn::os::ClearEvent(&m_WorkFinishEvent);
    m_Result = ResultSuccess();
}

void WorkClientLocker::Hold() NN_NOEXCEPT
{
    this->Reset();
    m_IsUsed = true;
}

void WorkClientLocker::WaitEvent() NN_NOEXCEPT
{
    nn::os::WaitEvent(&m_WorkFinishEvent);
}

void WorkClientLocker::SignalEvent() NN_NOEXCEPT
{
    nn::os::SignalEvent(&m_WorkFinishEvent);
}


// *** AsicWorkRingBuffer

AsicWorkRingBuffer::AsicWorkRingBuffer(AsicWorkInfo* dataBuffer, const size_t dataBufferLength) NN_NOEXCEPT : RingBuffer<AsicWorkInfo>(dataBuffer, dataBufferLength), m_RingBufferMutex(true)
{

}

void AsicWorkRingBuffer::Enqueue(AsicWorkInfo& data) NN_NOEXCEPT
{
    NN_DETAIL_RING_BUFFER_LOCK_GUARD;
    RingBuffer<AsicWorkInfo>::Enqueue(data);
}

void AsicWorkRingBuffer::Dequeue(AsicWorkInfo* pOutValue) NN_NOEXCEPT
{
    NN_DETAIL_RING_BUFFER_LOCK_GUARD;
    RingBuffer<AsicWorkInfo>::Dequeue(pOutValue);
}

size_t AsicWorkRingBuffer::GetLength() NN_NOEXCEPT
{
    NN_DETAIL_RING_BUFFER_LOCK_GUARD;
    return RingBuffer<AsicWorkInfo>::GetLength();
}

bool AsicWorkRingBuffer::IsEmpty() NN_NOEXCEPT
{
    NN_DETAIL_RING_BUFFER_LOCK_GUARD;
    return RingBuffer<AsicWorkInfo>::IsEmpty();
}


// *** AsicHandler

namespace {

// 復号処理（複数スレッドで読んでいた場合、パイプライン的に AsicHandler スレッドとは並列に行われる）
// 注：複数スレッドから呼ばれるため、この中に static な変数を入れないこと
void DecryptReadBuffer(char* outDataBuffer, char* preservedCtrCounterBuffer, size_t preservedCtrCounterBufferLength, size_t readSize)
{
    GcCrypto& gcCrypto = GcCrypto::GetInstance();

    Counter128Bit aesCtrCounter;
    aesCtrCounter.UseInverseCounter();
    aesCtrCounter.ResetCounterWithInitialValue(preservedCtrCounterBuffer, preservedCtrCounterBufferLength);

    const auto pCurrentThread = nn::os::GetCurrentThread();
    const auto currentPriority = nn::os::GetThreadPriority(pCurrentThread);
    nn::os::ChangeThreadPriority(pCurrentThread, currentPriority + 1);

    aesCtrCounter.OutputCounterValue(preservedCtrCounterBuffer, preservedCtrCounterBufferLength);
    gcCrypto.DecryptWithAesCtrWithCounter(outDataBuffer, readSize, outDataBuffer, readSize, preservedCtrCounterBuffer, preservedCtrCounterBufferLength);

    nn::os::ChangeThreadPriority(pCurrentThread, currentPriority);
}

}  // namespace

AsicHandler::AsicHandler() NN_NOEXCEPT : ThreadNamePtr(NN_SYSTEM_THREAD_NAME(gc, AsicHandler)), m_AsicWorkQueue(s_AsicWorkList, GcAsicWorkQueueLength), m_AsicLockerMutex(true), m_AsicHandlerCore(AsicHandlerCore::GetInstance())
{
    // gc ライブラリユーザにカードイベントを通知する用のコールバック
    InitializeGcCallback(&m_CardDetectionEventCallback);
}

AsicHandler& AsicHandler::GetInstance() NN_NOEXCEPT
{
    static AsicHandler s_Instance;
    return s_Instance;
}

void AsicHandler::InitializeThread() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    ThreadInterface::InitializeThread(m_ThreadStack, ThreadStackSize, ThreadNamePtr, NN_SYSTEM_THREAD_PRIORITY(gc, AsicHandler));
}

// NOTE: Initialize 内で GPIO 処理が行われる（Initialize->AsicHandlerCore->Reset->DeactivateIo）
nn::Result AsicHandler::Initialize() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // FW をセット（サイズ固定）
    StateMachine::GetInstanceForAsicHandler().SelectAsicFirmware(AsicFirmwareType_Read);

    // gpio 周りの初期化（DeviceDetector のコンストラクタが呼ばれるまでに初期化が必要）
    GeneralIo::GetInstance().Initialize();

    // DeviceDetector にカード抜去発生時に呼んでもらう用のコールバックを登録
    GcCallbackStruct cardRemovalCallback = { AsicHandlerCore::ProcessCardRemovalForCallback, &m_AsicHandlerCore };
    DeviceDetector::GetInstance().RegisterCardRemovalEventCallback(&cardRemovalCallback);

    // 挿抜スレッドの初期化・スタート
    DeviceDetector::GetInstance().InitializeThread();

    // リセット
    m_AsicHandlerCore.Reset();

    // スレッドの開始
    nn::gc::detail::AsicHandler::GetInstance().InitializeThread();

    NN_RESULT_SUCCESS;
}

void AsicHandler::Finalize() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    AsicHandler::GetInstance().FinalizeThread();
    DeviceDetector::GetInstance().FinalizeThread();
    DeviceDetector::GetInstance().UnregisterCardRemovalEventCallback();
    GeneralIo::GetInstance().Finalize();
    //   ~ AsicHandler::Initialize()

    AsicOperation::GetInstance().Finalize();
    DataIo::GetInstance().UnregisterDeviceVirtualAddress(m_WorkBuffer, m_WorkBufferLength, m_WorkBufferDeviceVirtualAddress);
    DataIo::GetInstance().Finalize();
    //   ~ AsicHandler::SetBuffer()
    // ~ gc::Initialize()
}

void AsicHandler::InitializeWaitEvents() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    ThreadInterface::InitializeWaitEvents();

    // イベントの初期化
    m_pGpioDetectReceiveEvent = DeviceDetector::GetInstance().GetDetectEvent();
    nn::os::InitializeEvent(&m_WorkRequestReceiveEvent, false, nn::os::EventClearMode_ManualClear);

    // GPIO Detect 端子
    nn::os::InitializeMultiWaitHolder(&m_GpioDetectEventHolder, m_pGpioDetectReceiveEvent);
    nn::os::LinkMultiWaitHolder(&m_Waiter, &m_GpioDetectEventHolder);
    // 処理要求
    nn::os::InitializeMultiWaitHolder(&m_WorkRequestEventHolder, &m_WorkRequestReceiveEvent);
    nn::os::LinkMultiWaitHolder(&m_Waiter, &m_WorkRequestEventHolder);
}

void AsicHandler::FinalizeWaitEvents() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // 処理要求
    nn::os::UnlinkMultiWaitHolder(&m_WorkRequestEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_WorkRequestEventHolder);
    // GPIO Detect 端子
    nn::os::UnlinkMultiWaitHolder(&m_GpioDetectEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_GpioDetectEventHolder);

    // イベントの終了処理
    nn::os::FinalizeEvent(&m_WorkRequestReceiveEvent);

    ThreadInterface::FinalizeWaitEvents();
}

nn::Result AsicHandler::OrderWork(AsicWork workIndex) NN_NOEXCEPT
{
    return OrderWorkInternalWithBlocking(workIndex, nullptr);
}

nn::Result AsicHandler::OrderWorkWithNonBlocking(AsicWork workIndex) NN_NOEXCEPT
{
    return OrderWorkInternal(workIndex, nullptr, true);
}

nn::Result AsicHandler::OrderWorkInternalWithBlocking(AsicWork workIndex, ReadInfo* readInfo) NN_NOEXCEPT
{
    return OrderWorkInternal(workIndex, readInfo, false);
}

nn::Result AsicHandler::OrderWorkInternal(AsicWork workIndex, ReadInfo* readInfo, bool isNonBlocking) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    WorkClientLocker* pLocker;
    int lockerIndex = 0;

    // ロッカーを取得する
    NN_RESULT_DO(AllocateLocker(&lockerIndex, &pLocker));

    // キューに処理要求を追加し、キュースレッドにイベントを通知する
    AsicWorkInfo info = {
        workIndex,
        lockerIndex,
        (isNonBlocking == false),
        readInfo
    };

    // キューへ追加し、イベントをシグナルする
    m_AsicWorkQueue.Enqueue(info);
    nn::os::SignalEvent(&m_WorkRequestReceiveEvent);

    // 通常は処理が終わるまでブロックさせる
    if(isNonBlocking == false)
    {
        // Asic 初期化はキューの処理後回し条件ではない他、たとえ失敗してもそのまま後段に流れていくこともあり、初期化無限ループは生じないので TimedWait でなくてよい
        pLocker->WaitEvent();
    }

    // ロッカーを返却する
    nn::Result result = ReturnLocker(lockerIndex);
    return result;
}

nn::Result AsicHandler::OrderWorkRead(char* outDataBuffer, const size_t bufferLength, const uint32_t pageAddress, const uint32_t pageCount) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_SDK_REQUIRES(outDataBuffer != nullptr);

    // 必要な変数定義
    size_t readSize = GcPageSize * pageCount;
    NN_DETAIL_GC_SDK_REQUIRES(readSize <= bufferLength);
    char preservedCtrCounterBuffer[16];

    // 実際の処理を依頼
    ReadInfo rinfo = {preservedCtrCounterBuffer, outDataBuffer, sizeof(preservedCtrCounterBuffer), bufferLength, pageAddress, pageCount};
    NN_RESULT_DO( OrderWorkInternalWithBlocking(AsicWork_Read, &rinfo) );

    // 復号化（誤ってメンバを参照しないよう分離）
    DecryptReadBuffer(outDataBuffer, preservedCtrCounterBuffer, sizeof(preservedCtrCounterBuffer), readSize);

    // 通信に成功しているが、Read 中・直後にカードが抜けている場合はカード抜けに置き換える
    bool isGpioRemovedSignal = m_AsicHandlerCore.IsRemoved() || (! StateMachine::GetInstance().IsCardInserted());
    nn::Result lastDeviceStatusAfterRead = m_AsicHandlerCore.GetLastDeviceStatusAfterRead();
    if(isGpioRemovedSignal || lastDeviceStatusAfterRead.IsFailure())
    {
        NN_DETAIL_GC_WARNING_LOG("Detected card removal without communication error (gpio: %d, cmd13: 0x%x)\n",
            isGpioRemovedSignal, lastDeviceStatusAfterRead.GetInnerValueForDebug());
        return nn::fs::ResultGameCardCardNotInserted();
    }

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandler::OrderWorkGetCardStatus(GameCardStatus* pOutValue) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_SDK_REQUIRES(pOutValue != nullptr);

    // 実際の処理を依頼：ReadInfo はバッファを渡すだけに使用、他の使わない値は nullptr か 0 で埋める
    ReadInfo rinfo = {nullptr, reinterpret_cast<char*>(pOutValue), 0, sizeof(GameCardStatus), 0, 0};
    NN_RESULT_DO( OrderWorkInternalWithBlocking(AsicWork_GetCardStatus, &rinfo) );

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandler::OrderWorkGetCardDeviceId(char* outBuffer, const size_t outBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_SDK_REQUIRES(outBuffer != nullptr);
    NN_DETAIL_GC_SDK_REQUIRES(outBufferSize >= GcCardDeviceIdSize);

    // 実際の処理を依頼：ReadInfo はバッファを渡すだけに使用、他の使わない値は nullptr か 0 で埋める
    ReadInfo rinfo = {nullptr, outBuffer, 0, GcCardDeviceIdSize, 0, 0};
    NN_RESULT_DO( OrderWorkInternalWithBlocking(AsicWork_GetDeviceId, &rinfo) );

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandler::OrderWorkGetCardDeviceCertificate(char* outBuffer, const size_t outBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_SDK_REQUIRES(outBuffer != nullptr);
    NN_DETAIL_GC_SDK_REQUIRES(outBufferSize >= GcDeviceCertificateSize);

    // 実際の処理を依頼：ReadInfo はバッファを渡すだけに使用、他の使わない値は nullptr か 0 で埋める
    ReadInfo rinfo = {nullptr, outBuffer, 0, GcDeviceCertificateSize, 0, 0};
    NN_RESULT_DO( OrderWorkInternalWithBlocking(AsicWork_GetDeviceCertificate, &rinfo) );

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandler::OrderWorkGetCardImageHash(char* outBuffer, const size_t outBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_SDK_REQUIRES(outBuffer != nullptr);
    NN_DETAIL_GC_SDK_REQUIRES(outBufferSize == GcCardImageHashSize);

    // 実際の処理を依頼：ReadInfo はバッファを渡すだけに使用、他の使わない値は nullptr か 0 で埋める
    ReadInfo rinfo = {nullptr, outBuffer, 0, GcCardImageHashSize, 0, 0};
    NN_RESULT_DO( OrderWorkInternalWithBlocking(AsicWork_GetCardImageHash, &rinfo) );

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandler::OrderWorkGetCardIdSet(GameCardIdSet* pOutValue) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_SDK_REQUIRES(pOutValue != nullptr);

    // 実際の処理を依頼：ReadInfo はバッファを渡すだけに使用、他の使わない値は nullptr か 0 で埋める
    ReadInfo rinfo = {nullptr, reinterpret_cast<char*>(pOutValue), 0, sizeof(GameCardIdSet), 0, 0};
    NN_RESULT_DO( OrderWorkInternalWithBlocking(AsicWork_GetCardIdSet, &rinfo) );

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandler::OrderWorkGetCardHeader(char* outBuffer, const size_t outBufferSize) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_SDK_REQUIRES(outBuffer != nullptr);
    NN_DETAIL_GC_SDK_REQUIRES(outBufferSize >= GcPageSize * GcCardHeaderPageCount);

    // 実際の処理を依頼：ReadInfo はバッファを渡すだけに使用、他の使わない値は nullptr か 0 で埋める
    ReadInfo rinfo = {nullptr, outBuffer, 0, GcPageSize * GcCardHeaderPageCount, 0, 0};
    NN_RESULT_DO( OrderWorkInternalWithBlocking(AsicWork_GetCardHeader, &rinfo) );

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandler::OrderWorkGetErrorInfo(nn::fs::GameCardErrorReportInfo* pOutGameCardErrorInfo) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_SDK_REQUIRES(pOutGameCardErrorInfo != nullptr);

    // 実際の処理を依頼：ReadInfo はバッファを渡すだけに使用、他の使わない値は nullptr か 0 で埋める
    ReadInfo rinfo = {nullptr, reinterpret_cast<char*>(pOutGameCardErrorInfo), 0, sizeof(nn::fs::GameCardErrorReportInfo), 0, 0};
    NN_RESULT_DO( OrderWorkInternalWithBlocking(AsicWork_GetErrorInfo, &rinfo) );

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandler::OrderWorkIsCardActivationValid(bool* pOutIsCardActivationValid) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    ReadInfo rinfo = {nullptr, reinterpret_cast<char*>(pOutIsCardActivationValid), 0, sizeof(bool), 0, 0};
    NN_RESULT_DO( OrderWorkInternalWithBlocking(AsicWork_IsCardActivationValid, &rinfo) );

    NN_RESULT_SUCCESS;
}

// ロッカーを探してホールドして返す
nn::Result AsicHandler::AllocateLocker(int* pOutIndex, WorkClientLocker** pOutLockerPtr) NN_NOEXCEPT
{
    NN_DETAIL_LOCKER_ACCESS_LOCK_GUARD;
    for(int i=0; i<GcAsicWorkQueueLength; i++)
    {
        if(m_AsicWorkClientLocker[i].IsAvailable())
        {
            *pOutIndex = i;
            *pOutLockerPtr = &(m_AsicWorkClientLocker[i]);
            (*pOutLockerPtr)->Hold();
            NN_RESULT_SUCCESS;
        }
    }
    NN_DETAIL_GC_WARNING_LOG("asic handler queue full\n");
    return nn::fs::ResultGameCardQueueFullFailure();
}

// 結果を取り出し、使い終わった locker を available にする
nn::Result AsicHandler::ReturnLocker(int lockerIndex) NN_NOEXCEPT
{
    NN_DETAIL_GC_SDK_REQUIRES(IsLockerIndexInRange(lockerIndex));
    NN_DETAIL_LOCKER_ACCESS_LOCK_GUARD;
    return m_AsicWorkClientLocker[lockerIndex].Release();
}

bool AsicHandler::IsLockerIndexInRange(int index) NN_NOEXCEPT
{
    return (0 <= index) && (index <= GcAsicWorkQueueLength);
}

void AsicHandler::RunThreadFunctionImpl() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

#if defined(NN_DETAIL_TEST_GC_SUPPRESS_INITIALIZATION)
    return;
#endif

    NN_DETAIL_GC_LOG("AsicHandler Thread started.\n");

    DeviceDetector& deviceDetector = DeviceDetector::GetInstance();
    bool isLastInserted = deviceDetector.GetGpioDetectPin();

    // セッション構築されていることを確定させる
    m_AsicHandlerCore.InitializeAsicIfNeeded();

    // 挿入された状態での起動のケア
    if(isLastInserted)
    {
        NN_DETAIL_GC_LOG("Insert (Initial)\n");
        m_AsicHandlerCore.ProcessInsert();
    }

    while(NN_STATIC_CONDITION(true))
    {
        // セッション構築されていることを確定させる
        m_AsicHandlerCore.InitializeAsicIfNeeded();

        // イベントが溜まっていないかチェック
        NN_DETAIL_GC_THREAD_DEBUG_LOG_EVENT_TRY_WAIT("AnyEvent");
        nn::os::MultiWaitHolderType* signaledHolder = nn::os::TryWaitAny(&m_Waiter);

        // イベントもキューも溜まっていなければ...
        if( signaledHolder == nullptr && m_AsicWorkQueue.GetLength() == 0 )
        {
            NN_DETAIL_GC_THREAD_DEBUG_LOG_LOG("no event");

            // イベント待ち
            NN_DETAIL_GC_THREAD_DEBUG_LOG_EVENT_WAIT("AnyEvent");
            signaledHolder = WaitAny(&m_Waiter);

            // 受け取ったイベントのログを流す
#ifdef GC_DETAIL_LOG_ENABLE
            if(signaledHolder == &m_ThreadEndEventHolder)
            {
                NN_DETAIL_GC_THREAD_DEBUG_LOG_EVENT_WAIT_END("m_ThreadEndEventHolder");
            }
            else if(signaledHolder == &m_GpioDetectEventHolder)
            {
                NN_DETAIL_GC_THREAD_DEBUG_LOG_EVENT_WAIT_END("m_GpioDetectEventHolder");
            }
#endif
        }
        // WaitAny で受け取ったイベントの順序がわからないので、明示的にイベントをチェックする
        bool isGpioEvent = nn::os::TryWaitEvent(m_pGpioDetectReceiveEvent);
        bool isThreadEndEvent = nn::os::TryWaitEvent(&m_ThreadEndEvent);
        bool isWorkRequestEvent = nn::os::TryWaitEvent(&m_WorkRequestReceiveEvent);

        // *** 以下、イベントを受け取ってからの処理

        // スレッドの終了処理
        if (isThreadEndEvent)
        {
            NN_DETAIL_GC_CLEAR_EVENT(&m_ThreadEndEvent);
            break;
        }
        // GPIO Detect 端子の割り込み
        else if ( isGpioEvent )
        {
            // GPIO DETECT を読む
            bool isCurrentInserted = deviceDetector.GetGpioDetectPin();
            NN_DETAIL_GC_LOG("card state: %d -> %d\n", isLastInserted, isCurrentInserted);

            // GPIO イベントのリセット
            NN_DETAIL_GC_THREAD_DEBUG_LOG_EVENT_CLEAR("GpioDetectEvent");
            deviceDetector.ClearGpioDetectEvent();

            // 上のレイヤに挿抜があったことを伝える
            NN_DETAIL_GC_LOG("Got event gpio detect\n");
            CallGcCallback(&m_CardDetectionEventCallback);

            // メインの挿抜処理
            if ((!isLastInserted) && (!isCurrentInserted))
            {
                // 抜 -> ... -> 抜 の場合
                NN_DETAIL_GC_LOG("Insert (LOST)\n");
                NN_DETAIL_GC_LOG("Remove (explicit)\n");
                // 本来は何もしなくてもよいが、明示的に呼んでおく
                m_AsicHandlerCore.ProcessRemove();
            }
            else if ((!isLastInserted) && isCurrentInserted)
            {
                // 抜 -> ... -> 挿 の場合
                NN_DETAIL_GC_LOG("Insert\n");
                m_AsicHandlerCore.ProcessInsert();
            }
            else if (isLastInserted && (!isCurrentInserted))
            {
                // 挿 -> ... -> 抜 の場合
                NN_DETAIL_GC_LOG("Remove\n");
                m_AsicHandlerCore.ProcessRemove();
            }
            else // (isLastInserted && isCurrentInserted)
            {
                // 挿 -> ... -> 挿 の場合
                NN_DETAIL_GC_LOG("Remove (LOST)\n");
                m_AsicHandlerCore.ProcessRemove();

                NN_DETAIL_GC_LOG("Insert\n");
                m_AsicHandlerCore.ProcessInsert();
            }
            isLastInserted = isCurrentInserted;
            continue;
        }
        // 通常イベント以外はキューに溜まった処理を実行していく
        else
        {
            do
            {
                //（キュー追加イベントが起こりたてでここに来る場合と、キューに溜まりっぱなしでイベントを受け取っていない時がある）
                if(isWorkRequestEvent)
                {
                    nn::os::ClearEvent(&m_WorkRequestReceiveEvent);
                }

                // キューに溜まった仕事を処理
                if(m_AsicWorkQueue.GetLength() > 0)
                {
                    AsicWorkInfo info;
                    m_AsicWorkQueue.Dequeue(&info);
                    HandleWorkRequest(info);

                    //（InitializeAsicIfNeeded 内部で IsForceReset フラグが立っている時は Deactivate から始める）
                }

                // スリープ中である限り、キューの処理を回し続ける：PutToSleep も Awaken も同じスレッドで仕事しているので、 IsAsleep は確かな値が入っている
                if(m_AsicHandlerCore.IsAsleep())
                {
                    isWorkRequestEvent = nn::os::TryWaitEvent(&m_WorkRequestReceiveEvent);
                    if(m_AsicWorkQueue.IsEmpty() && isWorkRequestEvent == false)
                    {
                        nn::os::WaitEvent(&m_WorkRequestReceiveEvent);
                    }
                    isWorkRequestEvent = true;
                    continue;
                }
            } while(NN_STATIC_CONDITION(false));
        }
    }
    NN_DETAIL_GC_LOG("AsicHandler Thread Finished.\n");
} // NOLINT

nn::Result AsicHandler::HandleWorkRequest(AsicWorkInfo workInfo) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // 初期化に失敗していたらその Result を返す
    nn::Result result = m_AsicHandlerCore.GetInitializationResult();

    // スリープ中なら AsicWork_Awaken しか受け付けない
    if(result.IsSuccess())
    {
        if(m_AsicHandlerCore.IsAsleep())
        {
            if(workInfo.workIndex != AsicWork_Awaken)
            {
                result = nn::fs::ResultGameCardNotAwakened();
            }
        }
    }

    // カード関連の仕事であれば抜去・Activate済みかをチェックする
    if(result.IsSuccess())
    {
        // 事故を避けるためデフォルトでチェックするようにしておき、する必要がないものリストをつくる
        static AsicWork CardStatusNoCheckWorkList[] = {
            AsicWork_None,
            AsicWork_Activate,
            AsicWork_Deactivate,
            AsicWork_PutToSleep,
            AsicWork_Awaken,
            AsicWork_GetCardHeader,
            AsicWork_IsCardActivationValid,
            AsicWork_GetErrorInfo,
            AsicWork_GetCardIdSet// Activate されていなくても ID は取得できてよい(エラーレポート用)
        };
        bool isNoCheck = false;
        for (int i = 0; i < sizeof(CardStatusNoCheckWorkList) / sizeof(AsicWork); i++)
        {
            isNoCheck |= (CardStatusNoCheckWorkList[i] == workInfo.workIndex);
        }
        if(isNoCheck == false)
        {
            result = m_AsicHandlerCore.CheckCardReady();
        }
    }

    // メインの処理分岐
    if(result.IsSuccess())
    {
        switch(workInfo.workIndex)
        {
        case AsicWork_None:
            {
                result = nn::ResultSuccess();
                break;
            }
        case AsicWork_Activate:
            {
                result = m_AsicHandlerCore.ActivateCard();
                break;
            }
        case AsicWork_Deactivate:
            {
                m_AsicHandlerCore.DeactivateCard();
                break;
            }
        case AsicWork_SetCardToSecureMode:
            {
                result = m_AsicHandlerCore.SetCardToSecureMode();
                break;
            }
        case AsicWork_PutToSleep:
            {
                result = m_AsicHandlerCore.PutToSleep();
                break;
            }
        case AsicWork_Awaken:
            {
                result = m_AsicHandlerCore.Awaken();
                // Awaken に失敗したら、外からカード抜けをシグナルさせる
                if(result.IsFailure())
                {
                    // カード抜けで失敗した場合は DeviceDetector がカード抜けを2回連続で signal することがあるが、抜 → 抜 の時は何もしないハズ
                    DeviceDetector::GetInstance().ManualSignalGpioDetectEventForGcCore();
                    m_AsicHandlerCore.IncrementAwakenFailureCount();
                }
                break;
            }
        case AsicWork_Read:
            {
                result = m_AsicHandlerCore.Read(workInfo.pReadInfo);
                break;
            }
        case AsicWork_GetCardStatus:
            {
                result = m_AsicHandlerCore.GetCardStatus(workInfo.pReadInfo);
                break;
            }
        case AsicWork_GetDeviceId:
            {
                result = m_AsicHandlerCore.GetCardDeviceId(workInfo.pReadInfo);
                break;
            }
        case AsicWork_GetDeviceCertificate:
            {
                result = m_AsicHandlerCore.GetCardDeviceCertificate(workInfo.pReadInfo);
                break;
            }
        case AsicWork_GetCardImageHash:
            {
                result = m_AsicHandlerCore.GetCardImageHash(workInfo.pReadInfo);
                break;
            }
        case AsicWork_GetCardIdSet:
            {
                result = m_AsicHandlerCore.GetCardIdSet(workInfo.pReadInfo);
                break;
            }
        case AsicWork_GetCardHeader:
            {
                result = m_AsicHandlerCore.GetCardHeader(workInfo.pReadInfo);
                break;
            }
        case AsicWork_IsCardActivationValid:
            {
                *reinterpret_cast<bool*>(workInfo.pReadInfo->outDataBuffer) = m_AsicHandlerCore.IsActivationValid();
                result = nn::ResultSuccess();
                break;
            }
        case AsicWork_GetErrorInfo:
            {
                result = m_AsicHandlerCore.GetErrorInfo(workInfo.pReadInfo);
                break;
            }
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    // sdmmc のエラー Result の場合、内部 Result に置換して返す（ResultCommunicationError を一時的に使用しているが、Module だけ取得できればそれでよい）
    if(result.GetModule() == nn::sdmmc::ResultCommunicationError().GetModule())
    {
        result = nn::fs::ResultGameCardCommunicationFailure();
    }

    // ノンブロッキングならロッカーは存在しない
    if(workInfo.hasLocker)
    {
        // 結果をロッカーに格納
        if(IsLockerIndexInRange(workInfo.lockerIndex) == false)
        {
            return nn::fs::ResultGameCardLockerOutOfRange();
        }
        WorkClientLocker& locker = m_AsicWorkClientLocker[workInfo.lockerIndex];
        locker.SetResult(result);
        locker.SignalEvent();
    }

    if(result.IsFailure())
    {
        NN_DETAIL_GC_LOG("HandleWorkRequest Failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }
    return result;
} // NOLINT




// *** public API

void AsicHandler::RegisterDetectionEventCallback(const GcCallbackStruct* pDetectionEventCallback) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    m_CardDetectionEventCallback = *(pDetectionEventCallback);
}

void AsicHandler::UnregisterDetectionEventCallback() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    InitializeGcCallback(&m_CardDetectionEventCallback);
}

void AsicHandler::SetBuffer(void* workBuffer, const size_t workBufferLength, const nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress)  NN_NOEXCEPT
{
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(workBufferLength >= GcWorkBufferSize);
    NN_DETAIL_GC_LOG("[GC] GcCore Initialization For Sdmmc\n");
    DataIo::Initialize();
    m_WorkBuffer = reinterpret_cast<uintptr_t>(workBuffer);
    m_WorkBufferLength = workBufferLength;
    m_WorkBufferDeviceVirtualAddress = workBufferDeviceVirtualAddress;
    DataIo::GetInstance().RegisterDeviceVirtualAddress(m_WorkBuffer, m_WorkBufferLength, m_WorkBufferDeviceVirtualAddress);
    // AsicOperation で使うためのバッファを用意する
    AsicOperation::Initialize(reinterpret_cast<char*>(workBuffer), SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(GcAsicFirmwareSize));
}



} } }
