﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/fs_BucketTreeBuilder.h>
#include <nn/fssystem/fs_IndirectStorageImpl.h>
#include <nn/fssystem/utilTool/fs_BucketTreeAnalyzer.h>
#include <nn/fssystem/utilTool/fs_SparseStorageBuilder.h>

namespace nn { namespace fssystem { namespace utilTool {

NN_DEFINE_STATIC_CONSTANT(const size_t SparseStorageBuilder::BlockSizeMin);

class SparseStorageBuilder::RangeContainer
{
public:
    explicit RangeContainer(int capacity) NN_NOEXCEPT
        : m_RangeArray(capacity)
        , m_EntryCount(0)
    {
    }

    Result Build(const BucketTree& indirectTable, int64_t originalSize, size_t blockSize) NN_NOEXCEPT;

    int64_t Optimize(int64_t originalSize, int64_t fragmentSize) NN_NOEXCEPT;

    int64_t Intersect(RangeArray otherArray, int64_t originalSize) NN_NOEXCEPT;

    bool IsValid() const NN_NOEXCEPT
    {
        return m_RangeArray.IsValid();
    }

    bool IsEmpty() const NN_NOEXCEPT
    {
        return m_RangeArray.empty();
    }

    int GetEntryCount() const NN_NOEXCEPT
    {
        return m_EntryCount;
    }

    RangeArray GetRangeArray() NN_NOEXCEPT
    {
        return std::move(m_RangeArray);
    }

private:
    RangeArray m_RangeArray;
    int m_EntryCount;
};

// IndirectStorage のテーブル情報からオリジナル側の必要な領域を表す Range の配列を生成
Result SparseStorageBuilder::RangeContainer::Build(const BucketTree& indirectTable, int64_t originalSize, size_t blockSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(originalSize, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(blockSize, BlockSizeMin);

    BucketTree::Visitor visitor;
    NN_RESULT_DO(indirectTable.Find(&visitor, 0));

    const auto endIndirectOffset = indirectTable.GetEnd();
    auto indirectOffset = visitor.Get<IndirectStorage::Entry>()->GetVirtualOffset();

    while( indirectOffset < endIndirectOffset )
    {
        IndirectStorage::EntryData indirectEntry;
        indirectEntry.Set(*visitor.Get<IndirectStorage::Entry>());

        // 次エントリの仮想オフセットを取得
        auto nextIndirectOffset = endIndirectOffset;
        if( visitor.CanMoveNext() )
        {
            NN_RESULT_DO(visitor.MoveNext());

            nextIndirectOffset = visitor.Get<IndirectStorage::Entry>()->GetVirtualOffset();

            NN_RESULT_THROW_UNLESS(
                indirectTable.IsInclude(nextIndirectOffset) && (indirectOffset < nextIndirectOffset),
                fs::ResultInvalidIndirectEntryOffset()
            );
        }

        // オリジナル側の必要な領域を切り出す
        if( indirectEntry.storageIndex == 0 )
        {
            const auto size = nextIndirectOffset - indirectOffset;
            NN_RESULT_THROW_UNLESS(indirectEntry.physicalOffset + size <= originalSize, fs::ResultInvalidArgument());

            // 必要な領域よりブロック単位で大きく取る
            const auto virtualOffset = util::align_down(indirectEntry.physicalOffset, blockSize);
            const auto endVirtualOffset = util::align_up(indirectEntry.physicalOffset + size, blockSize);

            m_RangeArray.emplace_back(virtualOffset, 0, std::min(endVirtualOffset, originalSize) - virtualOffset);
        }

        indirectOffset = nextIndirectOffset;
    }

    NN_RESULT_SUCCESS;
}

// Rnage 配列の最適化
int64_t SparseStorageBuilder::RangeContainer::Optimize(int64_t originalSize, int64_t eraseSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(originalSize, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(eraseSize, int64_t(BlockSizeMin));
    NN_SDK_REQUIRES(!m_RangeArray.empty());

    // virtualOffset の昇順、サイズの降順にソート
    std::sort(
        m_RangeArray.begin(),
        m_RangeArray.end(),
        [](const Range& lhs, const Range& rhs) NN_NOEXCEPT
        {
            if( lhs.virtualOffset == rhs.virtualOffset )
            {
                if( lhs.size == rhs.size )
                {
                    return &lhs < &rhs;
                }
                return lhs.size > rhs.size;
            }
            return lhs.virtualOffset < rhs.virtualOffset;
        }
    );

    int rangeCount = 0;
    int entryCount = 0;
    int64_t physicalOffset = 0;

    // 先頭の削除領域を調査
    {
        auto& front = m_RangeArray[0];

        // 削除サイズより小さければ front とつなげる
        if( front.virtualOffset < eraseSize )
        {
            front.size += front.virtualOffset;
            front.virtualOffset = 0;
        }
        else
        {
            ++entryCount; // 削除領域をカウント
        }
    }

    // 領域のマージと物理オフセットの割り当て
    for( int i = 1; i < m_RangeArray.size(); ++i )
    {
        auto& previous = m_RangeArray[rangeCount];
        const auto& current = m_RangeArray[i];

        // 領域が被らない
        if( previous.virtualOffset + previous.size < current.virtualOffset )
        {
            const auto distance = current.virtualOffset - (previous.virtualOffset + previous.size);

            // 削除サイズより小さければ previous ～ current をつなげる
            if( distance < eraseSize )
            {
                previous.size = current.virtualOffset + current.size - previous.virtualOffset;
            }
            else
            {
                previous.physicalOffset = physicalOffset;
                physicalOffset += previous.size;

                ++rangeCount;

                auto& entry = m_RangeArray[rangeCount];
                NN_SDK_ASSERT_EQUAL(entry.physicalOffset, 0);
                entry.virtualOffset = current.virtualOffset;
                entry.size = current.size;

                entryCount += 2; // previous と 削除領域をカウント
            }
        }
        // 領域が一部被るのでマージ
        else if( previous.virtualOffset + previous.size < current.virtualOffset + current.size )
        {
            previous.size = current.virtualOffset + current.size - previous.virtualOffset;
        }
        // 領域が全部被るのでスキップ
        // else {}
    }

    // 末尾の削除領域を調査
    NN_SDK_ASSERT_EQUAL(m_RangeArray[rangeCount].physicalOffset, 0);
    {
        auto& last = m_RangeArray[rangeCount];

        const auto distance = originalSize - (last.virtualOffset + last.size);
        NN_SDK_ASSERT_GREATER_EQUAL(distance, 0);

        // 削除サイズより小さければ last とつなげる
        if( distance < eraseSize )
        {
            last.size = originalSize - last.virtualOffset;
        }
        else
        {
            ++entryCount; // 削除領域をカウント
        }

        last.physicalOffset = physicalOffset;
        physicalOffset += last.size;

        ++rangeCount;
        ++entryCount; // last をカウント
    }

    m_EntryCount = entryCount;
    m_RangeArray.resize(rangeCount);

    return physicalOffset;
}

// 引数の Range 配列と共有する領域を生成
int64_t SparseStorageBuilder::RangeContainer::Intersect(RangeArray otherArray, int64_t originalSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!otherArray.empty());
    NN_SDK_REQUIRES_GREATER(originalSize, 0);
    NN_SDK_REQUIRES(!m_RangeArray.empty());

    RangeArray rangeArray(m_RangeArray.size() + otherArray.size());

    std::swap(m_RangeArray, rangeArray);

    if( m_RangeArray.IsValid() )
    {
        int entryCount = 0;
        int64_t virtualOffset = 0;
        int64_t physicalOffset = 0;
        auto otherBegin = otherArray.begin();
        const auto otherEnd = otherArray.end();

        for( int rangeIndex = 0; rangeIndex < rangeArray.size(); ++rangeIndex )
        {
            const auto& range = rangeArray[rangeIndex];

            // range の領域前方と other 側の領域後方が被っている場所を検索
            const auto intersectBegin = std::upper_bound(
                otherBegin,
                otherEnd,
                range.virtualOffset,
                [](int64_t offset, const Range& value) NN_NOEXCEPT
                {
                    return offset < (value.virtualOffset + value.size);
                }
            );

            // other 側がすべて range より前にあるので終了
            if( intersectBegin == otherEnd )
            {
                break;
            }

            // range の領域より後方で other 側の領域と被らない場所を検索
            const auto intersectEnd = std::lower_bound(
                otherBegin,
                otherEnd,
                range.virtualOffset + range.size,
                [](const Range& value, int64_t offset) NN_NOEXCEPT
                {
                    return value.virtualOffset < offset;
                }
            );

            // range と被っている領域を抽出
            for( auto iter = intersectBegin; iter < intersectEnd; ++iter )
            {
                const auto& other = *iter;
                NN_SDK_ASSERT_LESS(range.virtualOffset, other.virtualOffset + other.size);
                NN_SDK_ASSERT_LESS(other.virtualOffset, range.virtualOffset + range.size);

                const auto offset = std::max(range.virtualOffset, other.virtualOffset);
                const auto endOffset = std::min(range.virtualOffset + range.size, other.virtualOffset + other.size);
                NN_SDK_ASSERT_LESS(offset, endOffset);

                m_RangeArray.emplace_back(offset, physicalOffset, endOffset - offset);

                if( virtualOffset < offset )
                {
                    ++entryCount;
                }
                ++entryCount;

                virtualOffset = endOffset;
                physicalOffset += endOffset - offset;
            }

            if( otherBegin < intersectEnd )
            {
                otherBegin = intersectEnd - 1;
            }
        }

        if( m_RangeArray.empty() )
        {
            m_EntryCount = 0;
        }
        else
        {
            if( virtualOffset < originalSize )
            {
                ++entryCount;
            }

            m_EntryCount = entryCount;
        }

        return physicalOffset;
    }
    return 0;
}

/**
 * @brief   コンストラクタです。
 */
SparseStorageBuilder::SparseStorageBuilder() NN_NOEXCEPT
    : m_BlockSize(BlockSizeMin)
    , m_EraseSize(int64_t(BlockSizeMin))
    , m_EntryCount(0)
    , m_RangeArray()
    , m_DataStorage()
    , m_DataStorageSize(0)
    , m_VirtualStorageSize(0)
    , m_PhysicalStorageSize(0)
    , m_pStorageHolder()
{
}

/**
 * @brief   コンストラクタです。
 *
 * @param[in]   blockSize   ストレージの分割サイズ
 * @param[in]   eraseSize   ストレージの最小断片サイズ
 *
 * @pre
 *      - 512 <= blockSize
 *      - ispow2(blockSize) != false
 *      - blockSize <= eraseSize
 *      - eraseSize % blockSize == 0
 */
SparseStorageBuilder::SparseStorageBuilder(size_t blockSize, size_t eraseSize) NN_NOEXCEPT
    : m_BlockSize(blockSize)
    , m_EraseSize(int64_t(eraseSize))
    , m_EntryCount(0)
    , m_RangeArray()
    , m_DataStorage()
    , m_DataStorageSize(0)
    , m_VirtualStorageSize(0)
    , m_PhysicalStorageSize(0)
    , m_pStorageHolder()
{
    NN_SDK_REQUIRES((BlockSizeMin <= blockSize) && util::ispow2(blockSize));
    NN_SDK_REQUIRES((blockSize <= eraseSize) && (eraseSize % blockSize == 0));
}

/**
 * @brief   IndirectStorage を解析してテーブルを作成します。
 *
 * @param[in]   indirectTable   IndirectStorage のテーブル情報
 * @param[in]   originalSize    元データのサイズ
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - 0 <= originalSize
 */
Result SparseStorageBuilder::Build(const BucketTree& indirectTable, int64_t originalSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(originalSize, 0);
    NN_SDK_REQUIRES_EQUAL(m_VirtualStorageSize, 0);

    NN_RESULT_THROW_UNLESS(indirectTable.IsInitialized(), fs::ResultNotInitialized());

    if( 0 < indirectTable.GetEntryCount() )
    {
        NN_RESULT_THROW_UNLESS(0 < originalSize, fs::ResultInvalidSize());

        RangeContainer ranges(indirectTable.GetEntryCount());
        NN_RESULT_THROW_UNLESS(ranges.IsValid(), fs::ResultAllocationMemoryFailedNew());

        NN_RESULT_DO(ranges.Build(indirectTable, originalSize, m_BlockSize));
        if( !ranges.IsEmpty() )
        {
            const auto physicalOffset = ranges.Optimize(originalSize, m_EraseSize);

            m_EntryCount = ranges.GetEntryCount();
            m_RangeArray = ranges.GetRangeArray();
            m_PhysicalStorageSize = physicalOffset;
        }
    }

    m_VirtualStorageSize = originalSize;

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
 * @brief   IndirectStorage を解析してテーブルを再構築します。
 *
 * @param[in]   indirectTable   IndirectStorage のテーブル情報
 * @param[in]   originalSize    元データのサイズ
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - 0 <= originalSize
 */
Result SparseStorageBuilder::Rebuild(const BucketTree& indirectTable, int64_t originalSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(originalSize, 0);

    NN_RESULT_THROW_UNLESS(indirectTable.IsInitialized(), fs::ResultNotInitialized());
    NN_RESULT_THROW_UNLESS(m_VirtualStorageSize == originalSize, fs::ResultInvalidSize());

    if( !m_RangeArray.empty() )
    {
        NN_RESULT_THROW_UNLESS(0 < originalSize, fs::ResultInvalidSize());

        if( 0 < indirectTable.GetEntryCount() )
        {
            RangeContainer ranges(indirectTable.GetEntryCount());
            NN_RESULT_THROW_UNLESS(ranges.IsValid(), fs::ResultAllocationMemoryFailedNew());

            NN_RESULT_DO(ranges.Build(indirectTable, originalSize, m_BlockSize));
            if( !ranges.IsEmpty() )
            {
                ranges.Optimize(originalSize, m_EraseSize);
                if( !ranges.IsEmpty() )
                {
                    const auto physicalOffset = ranges.Intersect(std::move(m_RangeArray), originalSize);
                    NN_RESULT_THROW_UNLESS(ranges.IsValid(), fs::ResultAllocationMemoryFailedNew());

                    // 既存の物理領域のサイズ以下になるはず
                    NN_SDK_ASSERT_LESS_EQUAL(physicalOffset, m_PhysicalStorageSize);

                    m_EntryCount = ranges.GetEntryCount();
                    m_RangeArray = ranges.GetRangeArray();
                    m_PhysicalStorageSize = physicalOffset;

                    NN_RESULT_SUCCESS;
                }
            }
        }

        m_EntryCount = 0;
        m_RangeArray.clear();
        m_PhysicalStorageSize = 0;
    }

    NN_RESULT_SUCCESS;
}

/**
 * @brief   作成済みの SparseStorage を解析します。
 *
 * @param[in]   pStorage    構築済みのストレージのポインタ
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - pStorage != nullptr
 */
Result SparseStorageBuilder::Import(SparseStorage* pStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStorage);
    NN_SDK_REQUIRES_EQUAL(m_VirtualStorageSize, 0);

    NN_RESULT_THROW_UNLESS(pStorage->IsInitialized(), fs::ResultNotInitialized());

    auto& table = pStorage->GetEntryTable();
    const auto storageSize = table.GetEnd();

    const auto entryCount = pStorage->GetEntryTable().GetEntryCount();
    if( 0 < entryCount )
    {
        m_RangeArray = RangeArray((entryCount + 1) / 2);
        NN_RESULT_THROW_UNLESS(m_RangeArray.IsValid(), fs::ResultAllocationMemoryFailedNew());

        const auto pDataStorage = &pStorage->GetDataStorage(0);

        // Entry の配列から Range の配列にコンバート
        NN_RESULT_DO(pStorage->OperatePerEntry<false>(
            0,
            storageSize,
            [=](fs::SubStorage* pTargetStorage, int64_t dataOffset, int64_t processOffset, int64_t processSize) NN_NOEXCEPT -> Result
            {
                if( pTargetStorage == pDataStorage )
                {
                    m_RangeArray.emplace_back(processOffset, dataOffset, processSize);

                    m_PhysicalStorageSize = processOffset + processSize;
                }
                NN_RESULT_SUCCESS;
            }
        ));

        m_EntryCount = entryCount;
    }
    m_VirtualStorageSize = storageSize;

    NN_RESULT_SUCCESS;
}

/**
 * @brief   元データを含むストレージを設定します。
 *
 * @param[in]   storage     ストレージ
 *
 * @return  実行結果を返します。
 */
Result SparseStorageBuilder::SetDataStorage(fs::SubStorage storage) NN_NOEXCEPT
{
    int64_t storageSize = 0;
    NN_RESULT_DO(storage.GetSize(&storageSize));

    m_DataStorage = storage;
    m_DataStorageSize = storageSize;

    NN_RESULT_SUCCESS;
}

/**
 * @brief   スパース化済みのデータストレージを設定します。
 *
 * @param[in]   pStorage    ストレージのポインタ
 * @param[in]   offset      ストレージのオフセット
 * @param[in]   size        ストレージのサイズ
 */
void SparseStorageBuilder::SetDataStorage(std::unique_ptr<fs::IStorage>&& pStorage, int64_t offset, int64_t size) NN_NOEXCEPT
{
    // 事前検証は SubStorage に任せる
    m_DataStorage = fs::SubStorage(pStorage.get(), offset, size);
    m_DataStorageSize = size;

    m_pStorageHolder = std::move(pStorage); // SubStorage に pStorage を持たせられないので Builder 側に用意
}

/**
 * @brief   テーブルを書き出します。
 *
 * @param[in]   pAllocator      アロケータのポインタ
 * @param[in]   headerStorage   テーブルヘッダのストレージ
 * @param[in]   nodeStorage     テーブルノードのストレージ
 * @param[in]   entryStorage    テーブルエントリのストレージ
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - pAllocator != nullptr
 */
Result SparseStorageBuilder::WriteTable(
                                 IAllocator* pAllocator,
                                 fs::SubStorage headerStorage,
                                 fs::SubStorage nodeStorage,
                                 fs::SubStorage entryStorage
                             ) const NN_NOEXCEPT
{
    BucketTreeBuilder builder;
    NN_RESULT_DO(
        builder.Initialize(
            pAllocator,
            headerStorage,
            nodeStorage,
            entryStorage,
            SparseStorage::NodeSize,
            sizeof(SparseStorage::Entry),
            m_EntryCount
        )
    );

    if( 0 < m_EntryCount )
    {
        int64_t virtualOffset = 0;

        // IndirectStorage のエントリ情報にコンバート
        for( int i = 0; i < m_RangeArray.size(); ++i )
        {
            const auto& range = m_RangeArray[i];
            NN_SDK_ASSERT_LESS_EQUAL(virtualOffset, range.virtualOffset);

            // ZeroStorage 側の追加
            if( virtualOffset < range.virtualOffset )
            {
                SparseStorage::Entry entry;
                entry.SetVirtualOffset(virtualOffset);
                entry.SetPhysicalOffset(0);
                entry.storageIndex = 1;

                NN_RESULT_DO(builder.Write(entry));
            }

            // オリジナル側の追加
            {
                SparseStorage::Entry entry;
                entry.SetVirtualOffset(range.virtualOffset);
                entry.SetPhysicalOffset(range.physicalOffset);
                entry.storageIndex = 0;

                NN_RESULT_DO(builder.Write(entry));
            }

            virtualOffset = range.virtualOffset + range.size;
        }

        // 末尾の ZeroStorage 側の追加
        if( virtualOffset < m_VirtualStorageSize )
        {
            SparseStorage::Entry entry;
            entry.SetVirtualOffset(virtualOffset);
            entry.SetPhysicalOffset(0);
            entry.storageIndex = 1;

            NN_RESULT_DO(builder.Write(entry));
        }

        NN_RESULT_DO(builder.Commit(m_VirtualStorageSize));
    }

    NN_RESULT_SUCCESS;
}

/**
 * @brief   SparseStorage のビルド情報を出力します。
 *
 * @param[in]   pAllocator          アロケータのポインタ
 * @param[in]   pTableFileStorage   テーブル情報を書き出すストレージのポインタ
 * @param[in]   pNodeFileStorage    ノード情報を書き出すストレージのポインタ
 * @param[in]   pEntryFileStorage   エントリ情報を書き出すストレージのポインタ
 *
 * @pre
 *      - pAllocator != nullptr
 *      - pTableFileStorage != nullptr
 *      - pNodeFileStorage != nullptr
 *      - pEntryFileStorage != nullptr
 */
Result SparseStorageBuilder::OutputBuildLog(
           IAllocator* pAllocator,
           fs::IStorage* pTableFileStorage,
           fs::IStorage* pNodeFileStorage,
           fs::IStorage* pEntryFileStorage
       ) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pAllocator != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pTableFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pNodeFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pEntryFileStorage != nullptr, fs::ResultNullptrArgument());

    typedef SparseStorage::Entry Entry;
    typedef SparseStorage::EntryData EntryData;

    static const int64_t StorageSizeMax = 0x80000000;

    fs::SubStorage storages[utilTool::BucketTreeAnalyzer::StorageIndex_Max - 1] =
    {
        fs::SubStorage(pTableFileStorage, 0, StorageSizeMax),
        fs::SubStorage(pNodeFileStorage, 0, StorageSizeMax),
        fs::SubStorage(pEntryFileStorage, 0, StorageSizeMax),
    };

    auto formatEntry = [&](char* outText, size_t outSize, const void* pData) NN_NOEXCEPT -> size_t
    {
        // その他の事前検証は TextFormatter に任せる
        NN_SDK_REQUIRES_NOT_NULL(pData);

        utilTool::BucketTreeAnalyzer::TextFormatter formatter(outText, outSize);

        EntryData entry;
        entry.Set(*reinterpret_cast<const Entry*>(pData));

        // エントリ情報の出力
        if( entry.storageIndex == 0 )
        {
            if( !formatter.Append("0x%016llX 0x%016llX", entry.virtualOffset, entry.physicalOffset) )
            {
                return formatter.GetLength();
            }
        }
        else
        {
            if( !formatter.Append("0x%016llX * * *", entry.virtualOffset) )
            {
                return formatter.GetLength();
            }
        }
        return formatter.End();
    };

    utilTool::BucketTreeAnalyzer analyzer;
    NN_RESULT_DO(analyzer.Initialize(
        storages,
        SparseStorage::NodeSize,
        sizeof(Entry),
        m_EntryCount,
        0,
        "     VirtualOffset/    PhysicalOffset",
        formatEntry,
        nullptr
    ));

    char headerBuffer[sizeof(BucketTree::Header)];
    NN_SDK_ASSERT_EQUAL(QueryTableHeaderStorageSize(), int64_t(sizeof(headerBuffer)));
    fs::MemoryStorage headerStorage(headerBuffer, sizeof(headerBuffer));

    // BucketTreeBuilder からの書き出しをハックする
    NN_RESULT_DO(WriteTable(
        pAllocator,
        fs::SubStorage(&headerStorage, 0, sizeof(headerBuffer)),
        fs::SubStorage(analyzer.GetNodeStorage(), 0, QueryTableNodeStorageSize()),
        fs::SubStorage(analyzer.GetEntryStorage(), 0, QueryTableEntryStorageSize())
    ));

    NN_RESULT_SUCCESS;
}

}}}
