﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/utilTool/fs_IndirectStorageBuilderImpl.h>
#include <nn/fssystem/utilTool/fs_RelocatedIndirectStorageBuilder.h>
#include <nn/fssystem/utilTool/fs_RelocatedIndirectStorageUtils.h>

namespace nn { namespace fssystem { namespace utilTool {

/**
 * @brief   コンストラクタです。
 */
RelocatedIndirectStorageBuilder::RelocatedIndirectStorageBuilder() NN_NOEXCEPT
    : m_IsInitialized(false)
    , m_IsBuilt(false)
    , m_OldPatchInfo()
    , m_NewPatchInfo()
    , m_BinaryRegion()
    , m_OutputDataStorageOffset(0)
    , m_OldDataStorageSize(0)
    , m_NewDataStorageSize(0)
    , m_RelocatedDataSize(0)
    , m_IndirectTableSize(0)
    , m_pInvertedRelocationTable()
    , m_RelocatedEntries()
    , m_pProgress()
{
}

/**
 * @brief   デストラクタです。
 */
RelocatedIndirectStorageBuilder::~RelocatedIndirectStorageBuilder() NN_NOEXCEPT
{
}

/**
 * @brief   初期化をします。
 *
 * @param[in]   oldPatchInfo    古いデータを指すパッチの各種情報
 * @param[in]   newPatchInfo    新しいデータを指すパッチの各種情報
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - IsInitialize() == false
 *      - IsBuilt() == false
 */
Result RelocatedIndirectStorageBuilder::Initialize(
                                            const PatchInfo& oldPatchInfo,
                                            const PatchInfo& newPatchInfo,
                                            int64_t outputDataStorageOffset
                                        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    NN_SDK_REQUIRES(!IsBuilt());
    NN_SDK_REQUIRES_GREATER_EQUAL(outputDataStorageOffset, 0);

    {
        auto mutableStorage = oldPatchInfo.GetDataStorage();
        NN_RESULT_DO(mutableStorage.GetSize(&m_OldDataStorageSize));
    }
    {
        auto mutableStorage = newPatchInfo.GetDataStorage();
        NN_RESULT_DO(mutableStorage.GetSize(&m_NewDataStorageSize));
    }

    m_pInvertedRelocationTable.reset(new RelocationTable());
    NN_RESULT_THROW_UNLESS(
        m_pInvertedRelocationTable != nullptr, fs::ResultAllocationMemoryFailedNew());

    m_OldPatchInfo = oldPatchInfo;
    m_NewPatchInfo = newPatchInfo;
    m_IndirectTableSize = 0;
    m_OutputDataStorageOffset = outputDataStorageOffset;

    m_IsInitialized = true;

    NN_RESULT_SUCCESS;
}

/**
 * @brief   終了処理をします。
 */
void RelocatedIndirectStorageBuilder::Finalize() NN_NOEXCEPT
{
    m_OldPatchInfo = PatchInfo();
    m_NewPatchInfo = PatchInfo();
    m_BinaryRegion = decltype(m_BinaryRegion)();
    m_pInvertedRelocationTable.reset();
    m_RelocatedEntries.clear();
    m_pProgress.reset();

    m_IsInitialized = false;
    m_IsBuilt = false;
}

/**
 * @brief   リージョンハッシュの開始オフセットを取得します。
 *
 * @param[out]  pOutValue   オフセットの格納先
 * @param[in]   blockSize   ハッシュ化するデータのサイズ
 *
 * @return  実行結果を返します。
 */
Result RelocatedIndirectStorageBuilder::QueryBinaryRegionOffset(int64_t* pOutValue, size_t blockSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_GREATER(blockSize, static_cast<size_t>(0));
    NN_SDK_REQUIRES(IsInitialized());

    // shiftSize, regionSize, matchSize は使用しないので適当な値
    RelocationTable::BuildInfo info(
        m_OldPatchInfo.GetStorageOffset(), m_OutputDataStorageOffset, blockSize, blockSize, blockSize * 2, 0);

    NN_RESULT_DO(info.Initialize(m_OldPatchInfo.GetDataStorage(), m_NewPatchInfo.GetDataStorage()));

    if( info.IsOverlappedStorage() )
    {
        *pOutValue = info.GetOverlapStorageRange().first;
    }
    else
    {
        *pOutValue = 0;
    }

    NN_RESULT_SUCCESS;
}

/**
 * @brief   ストレージを解析してテーブルを作成します。
 *
 * @param[in]   pAllocator      アロケータのポインタ
 * @param[in]   shiftSize       ハッシュ化するデータの移動量
 * @param[in]   blockSize       ハッシュ化するデータのサイズ
 * @param[in]   regionSize      比較するデータの最小単位
 * @param[in]   matchSize       マッチするデータの最小単位
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - pAllocator != nullptr
 *      - 0 < shiftSize
 *      - shiftSize <= blockSize
 *      - ispow2(shiftSize) != false
 *      - 8 <= blockSize
 *      - ispow2(blockSize) != false
 *      - blockSize < regionSize
 *      - ispow2(regionSize) != false
 *      - 0 <= matchSize
 *      - IsInitialized() != false
 *      - IsBuilt() == false
 */
Result RelocatedIndirectStorageBuilder::Build(
                                            IAllocator* pAllocator,
                                            size_t shiftSize,
                                            size_t blockSize,
                                            size_t regionSize,
                                            int64_t matchSize
                                        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(!IsBuilt());

    NN_RESULT_THROW_UNLESS(pAllocator != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(
        sizeof(int64_t) <= blockSize && util::ispow2(blockSize), fs::ResultInvalidSize());
    NN_RESULT_THROW_UNLESS(
        0 < shiftSize && shiftSize <= blockSize && util::ispow2(shiftSize), fs::ResultInvalidSize());
    NN_RESULT_THROW_UNLESS(
        blockSize < regionSize && util::ispow2(regionSize), fs::ResultInvalidSize());
    NN_RESULT_THROW_UNLESS(0 <= matchSize, fs::ResultInvalidSize());

    // ストレージサイズのチェック
    NN_RESULT_THROW_UNLESS(
        util::is_aligned(m_OldDataStorageSize, blockSize) &&
        util::is_aligned(m_NewDataStorageSize, blockSize),
        fs::ResultInvalidSize()
    );

    RelocationTable::BuildInfo info(
        m_OldPatchInfo.GetStorageOffset(),
        m_OutputDataStorageOffset,
        shiftSize,
        blockSize,
        regionSize,
        matchSize
    );
    NN_RESULT_DO(info.Initialize(m_OldPatchInfo.GetDataStorage(), m_NewPatchInfo.GetDataStorage()));

    RelocationTable relocationTable;
    relocationTable.SetRegion(m_BinaryRegion.first);

    NN_RESULT_DO(relocationTable.Build(&m_pProgress, info));

    NN_RESULT_DO(relocationTable.Invert(
        m_pInvertedRelocationTable.get(),
        &m_RelocatedDataSize,
        m_NewPatchInfo.GetStorageOffset(),
        m_OldPatchInfo.GetStorageOffset()
    ));

    // IndirectStorage に RelocationTable を適用したエントリーを得る
    NN_RESULT_DO(BuildImpl(pAllocator, relocationTable));

    m_BinaryRegion = relocationTable.GetRegionInfo();

    NN_RESULT_SUCCESS;
}

// RelocationTable をストレージに適用します。
Result RelocatedIndirectStorageBuilder::BuildImpl(
                                            IAllocator* pAllocator,
                                            const RelocationTable& relocationTable
                                        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(!IsBuilt());

    BucketTree bucketTree;
    NN_RESULT_DO(bucketTree.Initialize(
        pAllocator,
        m_NewPatchInfo.GetNodeStorage(),
        m_NewPatchInfo.GetEntryStorage(),
        IndirectStorage::NodeSize,
        sizeof(IndirectStorage::Entry),
        m_NewPatchInfo.GetEntryCount()
    ));

    BucketTree::Visitor visitor;
    NN_RESULT_DO(bucketTree.Find(&visitor, 0));

    struct SimpleEntry
    {
        int64_t virtualOffset;
        int64_t physicalOffset;
        int storageIndex;

        bool CanMerge(const SimpleEntry& next) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_GREATER(next.virtualOffset, this->virtualOffset);

            return (this->storageIndex == next.storageIndex)
                && (this->physicalOffset + (next.virtualOffset - this->virtualOffset) == next.physicalOffset);
        }
    };

    const auto endOffset = bucketTree.GetEnd();
    auto entry1 = *visitor.Get<IndirectStorage::Entry>();
    auto entry1Offset = entry1.GetVirtualOffset();
    SimpleEntry previousEntry = { -1, -1, -1 };
    std::vector<IndirectStorage::Entry> entries;

    while( entry1Offset < endOffset )
    {
        IndirectStorage::Entry entry2 = {};
        int64_t entry2Offset;

        if( visitor.CanMoveNext() )
        {
            NN_RESULT_DO(visitor.MoveNext());

            entry2 = *visitor.Get<IndirectStorage::Entry>();
            entry2Offset = entry2.GetVirtualOffset();
        }
        else
        {
            entry2Offset = endOffset;
        }

        // 元データ以外のストレージにのみ RelocationTable を適用
        if( entry1.storageIndex != 0 )
        {
            const auto physicalOffset = entry1.GetPhysicalOffset();
            const auto physicalRange =
                std::make_pair(physicalOffset, physicalOffset + (entry2Offset - entry1Offset));

            std::vector<RelocationTable::Range> relocatedRanges;
            NN_RESULT_DO(relocationTable.ApplyTo(&relocatedRanges, physicalRange));

            int64_t offset = entry1Offset;

            // 再配置された各範囲を IndirectStorage の Entry として再構成する
            for( const auto range : relocatedRanges )
            {
                SimpleEntry currentEntry = { offset, range.first, entry1.storageIndex };

                // エントリが連続していないので追加する
                if( !previousEntry.CanMerge(currentEntry) )
                {
                    IndirectStorage::Entry entry;
                    entry.SetVirtualOffset(currentEntry.virtualOffset);
                    entry.SetPhysicalOffset(currentEntry.physicalOffset);
                    entry.storageIndex = currentEntry.storageIndex;

                    entries.push_back(entry);

                    previousEntry = currentEntry;
                }

                offset += (range.second - range.first);
            }
            NN_SDK_ASSERT(offset == entry2Offset);
        }
        else
        {
            SimpleEntry currentEntry = { entry1.GetVirtualOffset(), entry1.GetPhysicalOffset(), entry1.storageIndex };

            // エントリが連続していないので追加する
            if( !previousEntry.CanMerge(currentEntry) )
            {
                entries.push_back(entry1);

                previousEntry = currentEntry;
            }
        }

        entry1 = entry2;
        entry1Offset = entry2Offset;
    }

    m_RelocatedEntries = std::move(entries);
    m_IndirectTableSize = endOffset;
    m_IsBuilt = true;

    NN_RESULT_SUCCESS;
}

/**
 * @brief   テーブルを書き出します。
 *
 * @param[in]   pAllocator      アロケータのポインタ
 * @param[in]   headerStorage   テーブルヘッダのストレージ
 * @param[in]   nodeStorage     テーブルノードのストレージ
 * @param[in]   entryStorage    テーブルエントリのストレージ
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - pAllocator != nullptr
 *      - IsBuilt() != false
 */
Result RelocatedIndirectStorageBuilder::WriteTable(
                                            IAllocator* pAllocator,
                                            fs::SubStorage headerStorage,
                                            fs::SubStorage nodeStorage,
                                            fs::SubStorage entryStorage
                                        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsBuilt());

    // Relocation 適用済みのエントリーを使って BucketTree を作り直す
    BucketTreeBuilder builder;
    NN_RESULT_DO(builder.Initialize(
        pAllocator,
        headerStorage,
        nodeStorage,
        entryStorage,
        IndirectStorage::NodeSize,
        sizeof(IndirectStorage::Entry),
        GetEntryCount()
    ));

    for( const auto& entry : m_RelocatedEntries )
    {
        NN_RESULT_DO(builder.Write(entry));
    }

    return builder.Commit(m_IndirectTableSize);
}

/**
 * @brief   テーブルを元に必要なデータを読み込みます。
 *
 * @param[in]   offset  読み込むデータのオフセット
 * @param[in]   buffer  データを読み込むバッファのアドレス
 * @param[in]   size    読み込むデータのサイズ
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - IsBuilt() != false
 *      - size == 0 || buffer != nullptr
 *      - nn::fs::IStorage::CheckOffsetAndSize(offset, size) != false
 */
Result RelocatedIndirectStorageBuilder::ReadData(
                                            int64_t offset,
                                            void* buffer,
                                            size_t size
                                        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsBuilt());

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW_UNLESS(
        buffer != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(
        nn::fs::IStorage::CheckOffsetAndSize(offset, size), fs::ResultInvalidOffset());

    // RelocationTable を適用し、読み取り範囲を得る
    std::vector<utilTool::RelocationTable::PaddingRange> relocatedRanges;
    {
        const auto beginOffset = offset;
        const auto endOffset =
            std::min(offset + static_cast<int64_t>(size), m_RelocatedDataSize);

        NN_RESULT_DO(m_pInvertedRelocationTable->ApplyTo(
            &relocatedRanges, std::make_pair(beginOffset, endOffset)));
    }

    const auto oldStorageSize = m_OldDataStorageSize;
    auto& oldStorage = m_OldPatchInfo.GetDataStorage();
    auto& newStorage = m_NewPatchInfo.GetDataStorage();
    char* ptr = reinterpret_cast<char*>(buffer);

    // 各読み取り範囲から読み取る
    for( const auto paddingRange : relocatedRanges )
    {
        const auto& range = paddingRange.second;
        NN_SDK_ASSERT_LESS(range.first, range.second);

        const auto rangeSize = static_cast<size_t>(range.second - range.first);

        // oldStorage からコピー
        if( paddingRange.first )
        {
            // 全域がストレージの範囲外
            if( range.second < 0 || oldStorageSize <= range.first )
            {
                std::memset(ptr, 0, rangeSize);
            }
            // 前方の一部がストレージの範囲外
            else if( range.first < 0 )
            {
                const auto fillSize = static_cast<size_t>(-range.first);

                std::memset(ptr, 0, fillSize);

                const auto readSize = static_cast<size_t>(std::min(range.second, oldStorageSize));

                NN_RESULT_DO(oldStorage.Read(0, ptr + fillSize, readSize));

                // 後方の一部もストレージの範囲外
                if( oldStorageSize < range.second )
                {
                    const auto restSize = static_cast<size_t>(range.second - oldStorageSize);

                    std::memset(ptr + (fillSize + readSize), 0, restSize);
                }
            }
            // 後方の一部がストレージの範囲外
            else if( oldStorageSize < range.second )
            {
                const auto readSize = static_cast<size_t>(oldStorageSize - range.first);

                NN_RESULT_DO(oldStorage.Read(range.first, ptr, readSize));

                const auto fillSize = static_cast<size_t>(range.second - oldStorageSize);

                std::memset(ptr + readSize, 0, fillSize);
            }
            // 全域がストレージの範囲内
            else
            {
                NN_RESULT_DO(oldStorage.Read(range.first, ptr, rangeSize));
            }
        }
        // newStorage を入れ替え
        else
        {
            NN_RESULT_DO(newStorage.Read(range.first, ptr, rangeSize));
        }

        ptr += rangeSize;
    }

    // 残りの領域はゼロ埋め
    const auto stride = static_cast<size_t>(ptr - reinterpret_cast<char*>(buffer));
    if( stride < size )
    {
        std::memset(ptr, 0, size - stride);
    }

    NN_RESULT_SUCCESS;
}

/**
 * @brief   RelocatedIndirectStorage のビルド情報を出力します。
 *
 * @param[in]   pAllocator      アロケータのポインタ
 * @param[in]   pTableStorage   テーブル情報を書き出すストレージのポインタ
 * @param[in]   pNodeStorage    ノード情報を書き出すストレージのポインタ
 * @param[in]   pEntryStorage   エントリ情報を書き出すストレージのポインタ
 * @param[in]   pDataStorage    データ情報を書き出すストレージのポインタ
 * @param[in]   startOffset     IndirectStorage の開始オフセット
 *
 * @pre
 *      - pAllocator != nullptr
 *      - pTableStorage != nullptr
 *      - pNodeStorage != nullptr
 *      - pEntryStorage != nullptr
 *      - pDataStorage != nullptr
 *      - 0 <= startOffset
 */
Result RelocatedIndirectStorageBuilder::OutputBuildLog(
                                            IAllocator* pAllocator,
                                            fs::IStorage* pTableFileStorage,
                                            fs::IStorage* pNodeFileStorage,
                                            fs::IStorage* pEntryFileStorage,
                                            fs::IStorage* pDataFileStorage,
                                            int64_t startOffset
                                        ) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pAllocator != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pTableFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pNodeFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pEntryFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pDataFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(0 <= startOffset, fs::ResultInvalidOffset());

    return OutputBucketTreeBuildLog(
        pTableFileStorage,
        pNodeFileStorage,
        pEntryFileStorage,
        pDataFileStorage,
        GetEntryCount(),
        startOffset,
        startOffset + m_IndirectTableSize,
        // BucketTreeBuilder からの書き出しをハックする
        [=](const BucketTreeAnalyzer& analyzer) NN_NOEXCEPT -> Result
        {
            char headerBuffer[sizeof(BucketTree::Header)];
            NN_SDK_ASSERT_EQUAL(
                QueryTableHeaderStorageSize(), static_cast<int64_t>(sizeof(headerBuffer)));
            fs::MemoryStorage headerStorage(headerBuffer, sizeof(headerBuffer));

            BucketTreeBuilder builder;
            NN_RESULT_DO(
                builder.Initialize(
                    pAllocator,
                    fs::SubStorage(&headerStorage, 0, sizeof(headerBuffer)),
                    fs::SubStorage(analyzer.GetNodeStorage(), 0, QueryTableNodeStorageSize()),
                    fs::SubStorage(analyzer.GetEntryStorage(), 0, QueryTableEntryStorageSize()),
                    IndirectStorage::NodeSize,
                    sizeof(IndirectStorage::Entry),
                    GetEntryCount()
                )
            );

            for( const auto& entry : m_RelocatedEntries )
            {
                NN_RESULT_DO(builder.Write(entry));
            }

            return builder.Commit(m_IndirectTableSize);
        }
    );
}

/**
 * @brief   作成された 逆 RelocationTable を取得します。
 *
 * @return  RelocationTable を返します。
 *
 * @pre
 *      - IsBuilt() != false
 */
const RelocationTable&
    RelocatedIndirectStorageBuilder::GetInvertedRelocationTable() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsBuilt());

    return *m_pInvertedRelocationTable;
}

}}}
