﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/fs_NcaFileSystemDriver.h>
#include <nn/fssystem/utilTool/fs_NcaSparseStorage.h>
#include <nn/util/util_IntUtil.h>

namespace nn { namespace fssystem { namespace utilTool {

NN_DEFINE_STATIC_CONSTANT(const int NcaSparseStorage::Partition::CountMax);
NN_DEFINE_STATIC_CONSTANT(const uint32_t NcaSparseStorage::Header::Signature);

// コンストラクタ
NcaSparseStorage::NcaSparseStorage() NN_NOEXCEPT
    : m_Header()
    , m_Buffers()
{
    std::memset(&m_Header, 0, sizeof(m_Header));
}

// コンストラクタ
NcaSparseStorage::NcaSparseStorage(const NcaReader& reader) NN_NOEXCEPT
    : m_Header()
    , m_Buffers()
{
    std::memset(&m_Header, 0, sizeof(m_Header));

    // NcaHeader の必要な部分をコピー
    m_Header.signature = Header::Signature;
    m_Header.sdkAddonVersion = reader.GetSdkAddonVersion();
    m_Header.programId = reader.GetProgramId();

    for( int i = 0; i < Partition::CountMax; ++i )
    {
        // NcaFsHeader の必要な部分をコピー
        if( reader.HasFsInfo(i) )
        {
            NcaHeader::FsInfo fsInfo;
            reader.GetFsInfo(&fsInfo, i);

            auto& partition = m_Header.partition[i];
            partition.startSector = fsInfo.startSector;
            partition.endSector = fsInfo.endSector;
        }
    }
}

// パーティションの設定
Result NcaSparseStorage::SetPartition(MemoryResource* pAllocator, SparseStorageBuilder* pBuilder, const NcaFsHeaderReader& header) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAllocator);
    NN_SDK_REQUIRES_NOT_NULL(pBuilder);

    const auto nodeSize = pBuilder->QueryTableNodeStorageSize();
    const auto entrySize = pBuilder->QueryTableEntryStorageSize();

    NN_RESULT_THROW_UNLESS(header.GetSparseInfo().bucket.size == nodeSize + entrySize, fs::ResultInvalidSize());

    std::unique_ptr<char[]> buffer;
    if( 0 < nodeSize + entrySize )
    {
        NN_RESULT_THROW_UNLESS(util::IsIntValueRepresentable<size_t>(nodeSize + entrySize), fs::ResultInvalidSize());

        buffer.reset(new char[size_t(nodeSize + entrySize)]);
        NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultAllocationMemoryFailedNew());

        // WriteTable() の範囲外は何も書き込まれないので、あらかじめ 0 埋めしておく
        std::memset(buffer.get(), 0, size_t(nodeSize + entrySize));

        BucketTree::Header dummy;
        fs::MemoryStorage headerStorage(&dummy, sizeof(dummy));
        fs::MemoryStorage nodeStorage(buffer.get(), nodeSize);
        fs::MemoryStorage entryStorage(buffer.get() + nodeSize, entrySize);

        NN_RESULT_DO(pBuilder->WriteTable(
            pAllocator,
            fs::SubStorage(&headerStorage, 0, sizeof(dummy)),
            fs::SubStorage(&nodeStorage, 0, nodeSize),
            fs::SubStorage(&entryStorage, 0, entrySize)
        ));
    }

    const auto index = header.GetFsIndex();
    m_Buffers[index] = std::move(buffer);
    m_Header.partition[index].sparseInfo = header.GetSparseInfo();

    NN_RESULT_SUCCESS;
}

// 外部ファイルを読み込む
Result NcaSparseStorage::Load(const void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(size, size_t(0));

    NN_RESULT_THROW_UNLESS(sizeof(Header) <= size, fs::ResultInvalidSize());

    // ヘッダコピー
    std::memcpy(&m_Header, buffer, sizeof(Header));

    size_t bufferOffset = sizeof(Header);

    for( int i = 0; i < Partition::CountMax; ++i )
    {
        const auto& partition = m_Header.partition[i];

        // SparseStorage 情報をコピー
        const auto bufferSize = size_t(partition.sparseInfo.bucket.size);
        if( 0 < bufferSize )
        {
            NN_RESULT_THROW_UNLESS(bufferOffset + bufferSize <= size, fs::ResultInvalidSize());

            m_Buffers[i].reset(new char[bufferSize]);
            NN_RESULT_THROW_UNLESS(m_Buffers[i] != nullptr, fs::ResultAllocationMemoryFailedNew());

            std::memcpy(m_Buffers[i].get(), reinterpret_cast<const char*>(buffer) + bufferOffset, bufferSize);

            bufferOffset += bufferSize;
        }
    }

    NN_RESULT_SUCCESS;
}

// 外部ファイルのデータを取得
void NcaSparseStorage::Read(int64_t offset, void* buffer, size_t size) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);

    if( 0 < size )
    {
        NN_SDK_REQUIRES_NOT_NULL(buffer);

        int64_t copyOffset = offset;
        size_t copiedSize = 0;

        // ヘッダをコピー
        {
            const int64_t endOffset = sizeof(Header);

            if( copyOffset < endOffset )
            {
                const void* const pSrc = &m_Header;
                const auto copySize = std::min(size - copiedSize, size_t(endOffset - copyOffset));

                std::memcpy(buffer, reinterpret_cast<const char*>(pSrc) + copyOffset, copySize);

                copyOffset = 0;
                copiedSize += copySize;
            }
            else
            {
                copyOffset -= endOffset;
            }
        }

        if( copiedSize < size )
        {
            // m_Ranges をコピー
            for( int i = 0; i < Partition::CountMax; ++i )
            {
                const auto bufferSize = m_Header.partition[i].sparseInfo.bucket.size;
                if( copyOffset < bufferSize )
                {
                    const auto pDst = reinterpret_cast<char*>(buffer) + copiedSize;
                    const auto pSrc = m_Buffers[i].get() + copyOffset;
                    const auto copySize = std::min(size - copiedSize, size_t(bufferSize - copyOffset));

                    std::memcpy(pDst, pSrc, copySize);

                    copiedSize += copySize;
                    if( copiedSize == size )
                    {
                        break;
                    }

                    copyOffset = 0;
                }
                else
                {
                    copyOffset -= bufferSize;
                }
            }
        }

        // 残りの領域は 0 埋め
        if( copiedSize < size )
        {
            std::memset(reinterpret_cast<char*>(buffer) + copiedSize, 0, size - copiedSize);
        }
    }
}

// 外部ファイルのサイズを取得
int64_t NcaSparseStorage::GetSize() const NN_NOEXCEPT
{
    int64_t size = 0;

    for( int i = 0; i < Partition::CountMax; ++i )
    {
        size += m_Header.partition[i].sparseInfo.bucket.size;
    }

    return sizeof(Header) + size;
}

}}}
