﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssystem/utilTool/fs_BlockSignatureContainer.h>
#include <nn/fssystem/save/fs_HierarchicalIntegrityVerificationStorage.h>

namespace nn { namespace fssystem { namespace utilTool {

namespace {

const int HashSize = static_cast<int>(nn::crypto::Sha256Generator::HashSize);

}

NN_DEFINE_STATIC_CONSTANT(const int BlockSignatureContainer::LayerCountMin);
NN_DEFINE_STATIC_CONSTANT(const int BlockSignatureContainer::LayerCountMax);


BlockSignatureContainer::BlockSignatureContainer() NN_NOEXCEPT
    : m_Layers()
    , m_LayerCount(0)
    , m_IsOptimized(false)
    , m_Blocks()
{
    NN_STATIC_ASSERT(LayerCountMax == save::IntegrityMaxLayerCount - 1);
}

template< typename T >
void BlockSignatureContainer::InitializeImpl(const T& info) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(
        static_cast<int>(info.maxLayers) - 1, LayerCountMin, LayerCountMax);

    // マスターハッシュを除く階層数（データ部含む）
    m_LayerCount = static_cast<int>(info.maxLayers) - 1;

    for( int i = 0; i < m_LayerCount; ++i )
    {
        const auto& layer = info.info[i];
        NN_SDK_ASSERT_LESS(layer.orderBlock, NN_BITSIZEOF(int) - 1);

        m_Layers[i].start = layer.offset;
        m_Layers[i].width = layer.size;
        m_Layers[i].hashBlockSize = 1 << layer.orderBlock;
    }

    m_IsOptimized = false;
    m_Blocks.clear();
}

void BlockSignatureContainer::Initialize(const SaveDataLayerInfo& info) NN_NOEXCEPT
{
    InitializeImpl(info);
}

void BlockSignatureContainer::Initialize(const NcaRomFsLayerInfo& info) NN_NOEXCEPT
{
    InitializeImpl(info);
}

void BlockSignatureContainer::Collect(int64_t dataOffset, int64_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(dataOffset, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(dataSize, 0);
    NN_SDK_REQUIRES_GREATER(m_LayerCount, 0);
    NN_SDK_REQUIRES(!m_IsOptimized);

    const auto dataLayerIdx = m_LayerCount - 1;
    const auto& dataLayer = m_Layers[dataLayerIdx];

    if( dataSize == 0 || dataLayer.width <= dataOffset )
    {
        // データ範囲外なのでチェックしない
        return;
    }

    const Block dataBlock =
    {
        // データ範囲内に丸める
        dataLayer.width < dataOffset + dataSize ? dataLayer.width - dataOffset : dataSize,
        dataLayer.start + dataOffset,
        dataOffset,
        dataLayerIdx
    };

    if( AddDataBlock(dataBlock) )
    {
        // NOTE: AddDataBlock() で push_back() されるので iter は使えない
        AddHashBlock(dataBlock);
    }
}

bool BlockSignatureContainer::AddDataBlock(const Block& block) NN_NOEXCEPT
{
    const auto dataLayerIdx = m_LayerCount - 1;

    const auto end = m_Blocks.end();
    const auto begin = std::partition_point(
        m_Blocks.begin(),
        end,
        [=](const Block& dataBlock) NN_NOEXCEPT
        {
            return dataBlock.layerIndex < dataLayerIdx;
        }
    );

    auto pos = std::upper_bound(
        begin,
        end,
        block.position,
        [](int64_t value, const Block& target) NN_NOEXCEPT
        {
            return value < target.position;
        }
    );

    auto mergePos = pos;

    // 挿入位置より前のデータと被っているかチェック
    if( pos != begin )
    {
        auto& front = *(pos - 1);
        NN_SDK_ASSERT_GREATER_EQUAL(block.layerOffset, front.layerOffset);

        // 被っているか接しているのでマージする
        if( block.layerOffset <= front.layerOffset + front.size )
        {
            // block が front の中に納まっているので何もしない
            if( block.layerOffset + block.size <= front.layerOffset + front.size )
            {
                return false;
            }

            const auto blockSize = block.layerOffset + block.size - front.layerOffset;
            NN_SDK_ASSERT_GREATER(blockSize, front.size);

            // front を後方に伸長
            front.size = blockSize;

            --mergePos;
        }
    }

    // 挿入位置より後ろのデータと被っているかチェック
    if( pos != end )
    {
        const auto base = pos;

        // 被らない位置まで pos を進める
        while( pos->layerOffset <= block.layerOffset + block.size )
        {
            ++pos;
            if( pos == end )
            {
                break;
            }
        }

        // 被っているのでマージする
        if( pos != base )
        {
            const auto back = *(pos - 1);
            NN_SDK_ASSERT_LESS(block.layerOffset, back.layerOffset);

            auto& merge = *mergePos;

            const auto startOffset =
                std::min(merge.layerOffset, block.layerOffset);
            const auto endOffset =
                std::max(back.layerOffset + back.size, block.layerOffset + block.size);

            NN_SDK_ASSERT_GREATER_EQUAL(endOffset - startOffset, merge.size);

            merge.size = endOffset - startOffset;
            merge.position = block.position - block.layerOffset + startOffset;
            merge.layerOffset = startOffset;
        }
    }

    const auto mergeCount = std::distance(mergePos, pos);
    switch( mergeCount )
    {
    // マージされなかった
    case 0:
        m_Blocks.insert(pos, block);
        break;

    // いずれかの要素に吸収された
    case 1:
        break;

    // block がいくつかの要素をまとめた
    default:
        for( ; end != pos; ++pos )
        {
            ++mergePos;
            *mergePos = *pos;
        }

        NN_SDK_ASSERT_GREATER_EQUAL(m_Blocks.size(), static_cast<size_t>(mergeCount));
        m_Blocks.resize(m_Blocks.size() - (mergeCount - 1));
        break;
    }

    return true;
}

void BlockSignatureContainer::AddHashBlock(const Block& dataBlock) NN_NOEXCEPT
{
    // NOTE: 処理簡略化のためハッシュ部分のブロックはマージしない
    //       （ハッシュ部分のブロックを固定長にして同一データの検索を容易にするため）

    const auto& dataLayer = m_Layers[dataBlock.layerIndex];

    const auto startOffset =
        nn::util::align_down(dataBlock.layerOffset, dataLayer.hashBlockSize);
    const auto endOffset =
        nn::util::align_up(dataBlock.layerOffset + dataBlock.size, dataLayer.hashBlockSize);

    for( int64_t offset = startOffset; offset < endOffset; offset += dataLayer.hashBlockSize )
    {
        const auto blockCount = m_Blocks.size();
        auto layerOffset = offset;

        // 上層のハッシュをたどっていく
        for( int index = dataBlock.layerIndex - 1; 0 <= index; --index )
        {
            layerOffset = layerOffset / m_Layers[index + 1].hashBlockSize * HashSize;
            NN_SDK_ASSERT_LESS(layerOffset, m_Layers[index].width);

            Block hashBlock =
            {
                HashSize,
                m_Layers[index].start + layerOffset,
                layerOffset,
                index
            };
            NN_SDK_ASSERT_RANGE(
                hashBlock.position, m_Layers[index].start, m_Layers[index + 1].start);

            if( std::binary_search(
                m_Blocks.begin(),
                m_Blocks.end(),
                hashBlock,
                [](const Block& lhs, const Block& rhs) NN_NOEXCEPT
                {
                    return lhs.position < rhs.position;
                }
            ) )
            {
                // 同じデータが既に登録されているので以降はスキップ
                break;
            }
            else
            {
                // 同じデータがないので追加
                m_Blocks.push_back(hashBlock);
            }
        }

        if( blockCount != m_Blocks.size() )
        {
            std::sort(
                m_Blocks.begin(),
                m_Blocks.end(),
                [](const Block& lhs, const Block& rhs) NN_NOEXCEPT
                {
                    return lhs.position < rhs.position;
                }
            );
        }
    }
}

void BlockSignatureContainer::Optimize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(m_LayerCount, 0);

    if( m_IsOptimized )
    {
        return;
    }

    if( 1 < m_Blocks.size() )
    {
        const auto dataLayerIdx = m_LayerCount - 1;
        const auto dataBlockPos = std::partition_point(
            m_Blocks.begin(),
            m_Blocks.end(),
            [=](const Block& dataBlock) NN_NOEXCEPT
            {
                return dataBlock.layerIndex < dataLayerIdx;
            }
        );

        auto mergePos = m_Blocks.begin();
        auto pos = mergePos;
        ++pos;

        // ハッシュ部分のブロックをマージ
        while( pos < dataBlockPos )
        {
            auto& merge = *mergePos;
            const auto& block = *pos;

            // ブロックが連続するのでマージ
            if( merge.layerIndex == block.layerIndex &&
                merge.layerOffset + merge.size == block.layerOffset )
            {
                merge.size += block.size;
            }
            else
            {
                ++mergePos;
                if( mergePos != pos )
                {
                    *mergePos = *pos;
                }
            }

            ++pos;
        }
        ++mergePos;

        // データ部分のブロックを移動
        if( mergePos < pos )
        {
            const auto end = m_Blocks.end();
            while( pos != end )
            {
                *mergePos = *pos;

                ++pos;
                ++mergePos;
            }

            m_Blocks.resize(m_Blocks.size() - std::distance(mergePos, pos));
        }
    }

    m_IsOptimized = true;
}

}}}
