﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssystem/utilTool/fs_BinaryMatchPrivate.h>

namespace nn { namespace fssystem { namespace utilTool {

void BinaryMatch::RegionSet::Initialize(Region* pRegion) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pRegion);
    NN_SDK_REQUIRES(IsValid());

    static const int BinValueMax = std::numeric_limits<BinType>::max();

    for( int i = 0; i < m_RegionCount; ++i )
    {
        m_pRegions[i] = &pRegion[i];
    }

    // リージョンハッシュをキーにソートする
    std::sort(
        m_pRegions.get(),
        m_pRegions.get() + m_RegionCount,
        [](const Region* ptr1, const Region* ptr2) NN_NOEXCEPT
        {
            return *ptr1 < *ptr2;
        }
    );

    // 同じリージョンハッシュを持つ範囲を計算する
    // ゼロの出現数毎の開始インデックスを記録する
    int regionIndex = 0;
    int countZeros = 0;
    while( regionIndex < m_RegionCount )
    {
        const auto firstIndex = regionIndex;
        auto lastIndex = regionIndex;

        while( (lastIndex + 1 < m_RegionCount) &&
               (m_pRegions[lastIndex]->hash == m_pRegions[lastIndex + 1]->hash) )
        {
            ++lastIndex;
        }

        // 重複数のチェック
        {
            const auto duplicateCount = lastIndex + 1 - firstIndex;
            if( m_DuplicateCountMax < duplicateCount )
            {
                m_DuplicateCountMax = duplicateCount;
            }
        }

        while( regionIndex <= lastIndex )
        {
            m_pRegions[regionIndex]->beginIndex = firstIndex;
            m_pRegions[regionIndex]->endIndex = lastIndex + 1;

            ++regionIndex;
        }

        while( countZeros <= m_pRegions[firstIndex]->hash.GetIndex() )
        {
            m_IndexTable[countZeros] = firstIndex;
            ++countZeros;
        }
    }
    while( countZeros <= BinValueMax + 1 )
    {
        m_IndexTable[countZeros] = m_RegionCount;
        ++countZeros;
    }
}

const BinaryMatch::RegionRange BinaryMatch::RegionSet::Find(const RegionHash& regionHash) const NN_NOEXCEPT
{
    if( m_IndexTable != nullptr )
    {
        auto beginIndex = m_IndexTable[regionHash.GetIndex()];
        auto endIndex = m_IndexTable[regionHash.GetIndex() + 1];

        while( beginIndex < endIndex )
        {
            const auto index = (beginIndex + endIndex) / 2;
            const auto order = regionHash.Compare(m_pRegions[index]->hash);

            if( order < 0 )
            {
                endIndex = index;
            }
            else if( 0 < order )
            {
                beginIndex = index + 1;
            }
            else
            {
                NN_SDK_ASSERT_RANGE(index, 0, m_RegionCount);
                const auto& range = *m_pRegions[index];
                return RegionRange(&m_pRegions[range.beginIndex], range.endIndex - range.beginIndex);
            }
        }
    }
    return RegionRange();
}

void BinaryMatch::HashQueue::Reset(const char* buffer, size_t hashSize) NN_NOEXCEPT
{
    std::memset(&m_RegionHash, 0, sizeof(RegionHash));
    m_Index = 0;

    for( int i = 0; i < m_BlockHashCount; ++i )
    {
        BlockHash blockHash;
        blockHash.Set(buffer + i * hashSize, hashSize);

        m_RegionHash.Add(blockHash);
        m_pBlockHash[i] = blockHash;
        m_pEqual[i] = false;

        if( 0 < i )
        {
            auto const hashBuffer = buffer + (i - 1) * hashSize;
            if( std::memcmp(hashBuffer, hashBuffer + hashSize, hashSize) == 0 )
            {
                m_pEqual[i] = true;
                ++m_EqualCount;
            }
        }
    }
}

void BinaryMatch::HashQueue::Push(const char* buffer, size_t hashSize) NN_NOEXCEPT
{
    BlockHash blockHash;
    blockHash.Set(buffer, hashSize);

    m_RegionHash.Add(blockHash);
    m_RegionHash.Remove(m_pBlockHash[m_Index]);
    m_pBlockHash[m_Index] = blockHash;

    if( m_pEqual[m_Index] )
    {
        --m_EqualCount;
    }
    // NOTE: 呼び出し元で buffer[-hashSize] の領域にも有効なデータがあることが保証されているので問題ない
    m_pEqual[m_Index] = std::memcmp(buffer - hashSize, buffer, hashSize) == 0;
    if( m_pEqual[m_Index] )
    {
        ++m_EqualCount;
    }

    m_Index = (m_Index + 1) % m_BlockHashCount;
}

BinaryMatch::HashQueueArray::HashQueueArray(int hashQueueCount, int blockHashCount) NN_NOEXCEPT
    : m_Memory(new char[hashQueueCount * HashQueue::QuerySize(blockHashCount) + sizeof(int)])
    , m_pBegin(nullptr)
    , m_QueueSize(HashQueue::QuerySize(blockHashCount))
    , m_QueueCount(hashQueueCount)
{
    NN_STATIC_ASSERT(sizeof(BinaryMatch::HashQueue) % sizeof(void*) == 0);
    NN_SDK_REQUIRES_GREATER(hashQueueCount, 0);
    NN_SDK_REQUIRES_GREATER(blockHashCount, 0);

    if( m_Memory != nullptr )
    {
        m_pBegin = reinterpret_cast<char*>(util::align_up(reinterpret_cast<uintptr_t>(m_Memory.get()), sizeof(void*)));

        for( int i = 0; i < hashQueueCount; ++i )
        {
            HashQueue::Create(m_pBegin + i * m_QueueSize, m_QueueSize, blockHashCount);
        }
    }
}

BinaryMatch::HashQueueArray::~HashQueueArray() NN_NOEXCEPT
{
    if( m_Memory != nullptr )
    {
        for( int i = 0; i < m_QueueCount; ++i )
        {
            HashQueue::Destroy(Get(i));
        }
    }
}

void BinaryMatch::CompareRange::MoveNext(int64_t offset) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(offset, 0);

    m_CurOffset = offset;

    for( ; m_ResultIndex < m_ResultCount; ++m_ResultIndex )
    {
        const auto& result = m_Results[m_ResultIndex];
        const auto size = result.newOffset - m_CurOffset;
        NN_SDK_ASSERT_GREATER_EQUAL(size, 0);

        if( m_CompareSize <= size )
        {
            m_EndOffset = result.newOffset;
            return;
        }

        // result.EndOffset ～ [result + 1].Offset が次の比較範囲
        m_CurOffset = result.newOffset + result.size;
    }

    m_EndOffset = m_CurOffset;
}

}}}
