﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/utilTool/fs_BucketTreeAnalyzer.h>
#include <nn/fssystem/utilTool/fs_AesCtrCounterExtendedStorageBuilder.h>
#include <nn/fssystem/utilTool/fs_RelocatedIndirectStorageUtils.h>

namespace nn { namespace fssystem { namespace utilTool {

namespace {

// BucketTree より beginOffset から endOffset の一部または全部のオフセットを含むエントリーを取得
Result MakeSubRangeAesCtrCounterExtendedEntry(
           std::vector<AesCtrCounterExtendedStorage::Entry>* pOutValue,
           int64_t beginOffset,
           int64_t endOffset,
           const BucketTree& bucketTree
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_GREATER_EQUAL(beginOffset, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(beginOffset, endOffset);

    BucketTree::Visitor visitor;
    NN_RESULT_DO(bucketTree.Find(&visitor, beginOffset));

    while( NN_STATIC_CONDITION(true) )
    {
        const auto& entry = *visitor.Get<const AesCtrCounterExtendedStorage::Entry>();

        if( endOffset <= entry.GetOffset() )
        {
            break;
        }

        pOutValue->push_back(entry);

        if( !visitor.CanMoveNext() )
        {
            break;
        }

        NN_RESULT_DO(visitor.MoveNext());
    }

    NN_RESULT_SUCCESS;
}

}

/**
 * @brief   コンストラクタです。
 */
AesCtrCounterExtendedStorageBuilder::AesCtrCounterExtendedStorageBuilder() NN_NOEXCEPT
    : m_IsInitialized(false)
    , m_IsBuilt(false)
    , m_OldTableInfo()
    , m_OldStorageOffset(0)
    , m_OldDataStorageSize(0)
    , m_NewStorageOffset(0)
    , m_NewDataStorageSize(0)
    , m_Entries()
    , m_EndOffset(0)
{
}

/**
 * @brief   デストラクタです。
 */
AesCtrCounterExtendedStorageBuilder::~AesCtrCounterExtendedStorageBuilder() NN_NOEXCEPT
{
}

/**
 * @brief   初期化をします。
 *
 * @param[in]   dataSize    データのサイズ
 * @param[in]   generation  全ての領域に設定する世代
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - IsInitialized() == false
 *      - IsBuilt() == false
 *      - 0 < dataSize
 */
void AesCtrCounterExtendedStorageBuilder::Initialize(
                                              int64_t dataSize,
                                              uint32_t generation
                                          ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(dataSize, 0);
    NN_SDK_REQUIRES(!IsInitialized());
    NN_SDK_REQUIRES(!IsBuilt());

    m_OldStorageOffset = 0;
    m_OldDataStorageSize = 0;
    m_NewStorageOffset = 0;
    m_NewDataStorageSize = dataSize;

    // AesCtrEx テーブルを生成する
    AesCtrCounterExtendedStorage::Entry entry;
    entry.SetOffset(0);
    entry.reserved = 0;
    entry.generation = generation;

    m_Entries.push_back(entry);

    // EndOffset 設定
    m_EndOffset = dataSize;

    m_IsInitialized = true;
    m_IsBuilt = true;
}

/**
 * @brief   初期化をします。
 *
 * @param[in]   oldTableInfo        古いストレージのテーブル関連の情報
 * @param[in]   oldStorageOffset    古いストレージまでのオフセット
 * @param[in]   oldDataStorageSize  古いデータを指すストレージのサイズ
 * @param[in]   newStorageOffset    新しいストレージまでのオフセット
 * @param[in]   newDataStorageSize  新しいデータを指すストレージのサイズ
 *
 * @pre
 *      - IsInitialized() == false
 *      - IsBuilt() == false
 *      - 0 <= oldStorageOffset
 *      - 0 <= oldDataStorageSize
 *      - 0 <= newStorageOffset
 *      - 0 <= newDataStorageSize
 */
void AesCtrCounterExtendedStorageBuilder::Initialize(
                                              const TableInfo& oldTableInfo,
                                              int64_t oldStorageOffset,
                                              int64_t oldDataStorageSize,
                                              int64_t newStorageOffset,
                                              int64_t newDataStorageSize
                                          ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    NN_SDK_REQUIRES(!IsBuilt());
    NN_SDK_REQUIRES_GREATER_EQUAL(oldStorageOffset, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(oldDataStorageSize, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(newStorageOffset, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(newDataStorageSize, 0);

    m_OldTableInfo = oldTableInfo;
    m_OldStorageOffset = oldStorageOffset;
    m_OldDataStorageSize = oldDataStorageSize;
    m_NewStorageOffset = newStorageOffset;
    m_NewDataStorageSize = newDataStorageSize;

    m_IsInitialized = true;
}

/**
 * @brief   終了処理をします。
 */
void AesCtrCounterExtendedStorageBuilder::Finalize() NN_NOEXCEPT
{
    m_OldTableInfo = TableInfo();
    m_Entries.clear();
    m_IsInitialized = false;
    m_IsBuilt = false;
}

/**
 * @brief   RelocationTable を解析してテーブルを作成します。
 *
 * @param[in]   pAllocator              アロケータのポインタ
 * @param[in]   invertedRelocationTable 逆リロケーションテーブル
 * @param[in]   indirectTableOffset     IndirectStorage テーブルのオフセット
 * @param[in]   indirectTableSize       IndirectStorage テーブルのサイズ
 * @param[in]   generation              更新された領域に設定する世代
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - IsInitialized() != false
 *      - IsBuilt() == false
 *      - pAllocator != nullptr
 *      - 0 <= indirectTableSize
 *      - 0 <= indirectTableOffset
 */
Result AesCtrCounterExtendedStorageBuilder::Build(
                                                IAllocator* pAllocator,
                                                const RelocationTable& invertedRelocationTable,
                                                int64_t indirectTableOffset,
                                                int64_t indirectTableSize,
                                                uint32_t generation
                                            ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(!IsBuilt());
    NN_SDK_REQUIRES_NOT_NULL(pAllocator);
    NN_SDK_REQUIRES_GREATER_EQUAL(indirectTableOffset, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(indirectTableSize, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(indirectTableOffset, m_NewDataStorageSize);

    const auto tableEndOffset = util::align_up(
        std::min(indirectTableOffset, m_NewDataStorageSize), AesCtrCounterExtendedStorage::BlockSize);

    if( !invertedRelocationTable.empty() )
    {
        // データ部の AesCtrEx テーブルを生成する
        NN_RESULT_DO(BuildImpl(pAllocator, invertedRelocationTable, tableEndOffset, generation));
    }

    // データ部の後ろに IndirectStorage のテーブルがある
    // (ただし、データ部が無いときは indirectTableOffset == 0)
    NN_RESULT_THROW_UNLESS(
        m_Entries.empty() ? indirectTableOffset == 0
                          : m_Entries.back().GetOffset() < indirectTableOffset,
        fs::ResultInvalidOffset()
    );

    // IndirectStorage のテーブル領域について追加
    if( 0 < indirectTableSize )
    {
        if( m_Entries.empty() || (m_Entries.back().generation != static_cast<int32_t>(generation)) )
        {
            AesCtrCounterExtendedStorage::Entry entry;
            entry.SetOffset(tableEndOffset);
            entry.reserved = 0;
            entry.generation = generation; // 更新された領域として鍵世代を設定

            m_Entries.push_back(entry);
        }
    }

    m_EndOffset = indirectTableOffset + indirectTableSize;

    m_IsBuilt = true;

    NN_RESULT_SUCCESS;
}

// データ領域のテーブルを作成します。
Result AesCtrCounterExtendedStorageBuilder::BuildImpl(
                                                IAllocator* pAllocator,
                                                const RelocationTable& relocationTable,
                                                int64_t tableEndOffset,
                                                uint32_t generation
                                            ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAllocator);
    NN_SDK_REQUIRES(!relocationTable.empty());

    // 前バージョンにテーブルがない
    if( m_OldTableInfo.GetEntryCount() <= 0 )
    {
        // 一様に generation を使う
        AesCtrCounterExtendedStorage::Entry entry;
        entry.SetOffset(0);
        entry.reserved = 0;
        entry.generation = generation;

        m_Entries.push_back(entry);

        NN_RESULT_SUCCESS;
    }

    // BucketTree を構成する
    BucketTree bucketTree;
    NN_RESULT_DO(bucketTree.Initialize(
        pAllocator,
        m_OldTableInfo.GetNodeStorage(),
        m_OldTableInfo.GetEntryStorage(),
        AesCtrCounterExtendedStorage::NodeSize,
        sizeof(AesCtrCounterExtendedStorage::Entry),
        m_OldTableInfo.GetEntryCount()
    ));

    std::vector<AesCtrCounterExtendedStorage::Entry> allEntries;

    // 前バージョンのパッチ領域の範囲外は世代を generation として追加
    if( m_NewStorageOffset < m_OldStorageOffset )
    {
        AesCtrCounterExtendedStorage::Entry entry;
        entry.SetOffset(0);
        entry.reserved = 0;
        entry.generation = generation;

        allEntries.push_back(entry);
    }

    const auto adjustOffset = m_OldStorageOffset - m_NewStorageOffset;
    {
        std::vector<AesCtrCounterExtendedStorage::Entry> oldEntries;

        for( const auto& relocationEntry : relocationTable )
        {
            // 新バージョン側の領域に対応する前バージョン側のエントリーを取得
            {
                const auto beginOffset = relocationEntry.GetSrcOffset();
                const auto endOffset = beginOffset + relocationEntry.GetSize();

                // 取得する範囲を前バージョンのパッチ領域に制限
                const auto overlapBeginOffset =
                    std::max<int64_t>(beginOffset - adjustOffset, 0);
                const auto overlapEndOffset =
                    std::min<int64_t>(endOffset - adjustOffset, m_OldDataStorageSize);

                if( overlapBeginOffset < overlapEndOffset )
                {
                    NN_RESULT_DO(MakeSubRangeAesCtrCounterExtendedEntry(
                        &oldEntries, overlapBeginOffset, overlapEndOffset, bucketTree));
                }
            }

            // 各エントリーについてオフセットと世代を更新
            for( const auto& oldEntry : oldEntries )
            {
                auto entry = oldEntry;

                // オフセットを new 側のものに補正
                entry.SetOffset(
                    std::max(entry.GetOffset() + adjustOffset, relocationEntry.GetSrcOffset()));
                entry.reserved = 0;

                // 世代を generation に更新
                if( !relocationEntry.IsMatched() && !relocationEntry.IsPadding() )
                {
                    entry.generation = generation;
                }

                NN_SDK_ASSERT(
                    allEntries.empty() || (allEntries.back().GetOffset() < entry.GetOffset()));

                allEntries.push_back(entry);
            }

            oldEntries.clear();
        }
    }

    // 前バージョンのパッチ領域の範囲外は世代を generation として追加
    if( m_OldDataStorageSize + adjustOffset < m_NewDataStorageSize )
    {
        AesCtrCounterExtendedStorage::Entry entry;
        entry.SetOffset(std::max<int64_t>(m_OldDataStorageSize + adjustOffset, 0));
        entry.reserved = 0;
        entry.generation = generation;

        NN_SDK_ASSERT(
            allEntries.empty() || (allEntries.back().GetOffset() < entry.GetOffset()));

        allEntries.push_back(entry);
    }

    NN_SDK_ASSERT(!allEntries.empty());

    // generation が連続する Entry をつなげる
    NN_RESULT_DO(OptimizeAesCtrCounterExtendedEntry(&allEntries, tableEndOffset, static_cast<int>(generation)));

    m_Entries = std::move(allEntries);

    NN_RESULT_SUCCESS;
}

/**
 * @brief   テーブルを書き出します。
 *
 * @param[in]   pAllocator        アロケータのポインタ
 * @param[in]   headerStorage   テーブルヘッダのストレージ
 * @param[in]   nodeStorage     テーブルノードのストレージ
 * @param[in]   entryStorage    テーブルエントリのストレージ
 *
 * @return  実行結果を返します。
 *
 * @pre
 *      - IsInitialized() != false
 *      - IsBuilt() != false
 *      - pAllocator != nullptr
 */
Result AesCtrCounterExtendedStorageBuilder::WriteTable(
                                                IAllocator* pAllocator,
                                                fs::SubStorage headerStorage,
                                                fs::SubStorage nodeStorage,
                                                fs::SubStorage entryStorage
                                            ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(IsBuilt());

    BucketTreeBuilder builder;
    NN_RESULT_DO(builder.Initialize(
        pAllocator,
        headerStorage,
        nodeStorage,
        entryStorage,
        AesCtrCounterExtendedStorage::NodeSize,
        sizeof(AesCtrCounterExtendedStorage::Entry),
        static_cast<int>(m_Entries.size())
    ));

    for( const auto& entry : m_Entries )
    {
        NN_RESULT_DO(builder.Write(entry));
    }

    return builder.Commit(m_EndOffset);
}

/**
 * @brief   AesCtrCounterExtendedStorageBuilder のビルド情報を出力します。
 *
 * @param[in]   pAllocator      アロケータのポインタ
 * @param[in]   pTableStorage   テーブル情報を書き出すストレージのポインタ
 * @param[in]   pNodeStorage    ノード情報を書き出すストレージのポインタ
 * @param[in]   pEntryStorage   エントリ情報を書き出すストレージのポインタ
 * @param[in]   startOffset     IndirectStorage の開始オフセット
 *
 * @pre
 *      - IsBuilt() != false
 *      - pAllocator != nullptr
 *      - pTableStorage != nullptr
 *      - pNodeStorage != nullptr
 *      - pEntryStorage != nullptr
 *      - 0 <= startOffset
 */
Result AesCtrCounterExtendedStorageBuilder::OutputBuildLog(
                                                IAllocator* pAllocator,
                                                fs::IStorage* pTableFileStorage,
                                                fs::IStorage* pNodeFileStorage,
                                                fs::IStorage* pEntryFileStorage,
                                                int64_t startOffset,
                                                const RelocationTable& invertedRelocationTable
                                            ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsBuilt);

    NN_RESULT_THROW_UNLESS(pAllocator != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pTableFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pNodeFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(pEntryFileStorage != nullptr, fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(0 <= startOffset, fs::ResultInvalidOffset());

    typedef AesCtrCounterExtendedStorage::Entry Entry;

    static const int64_t StorageSizeMax = 0x80000000;
    static const char* const StatusText[] =
    {
        "?",
        "M",
        "P",
        "A",
        "SF",
        "LS",
        "LJ",
        "SS",
        "FS",
        "RJ"
    };
    NN_STATIC_ASSERT(sizeof(StatusText) / sizeof(StatusText[0]) == BinaryMatchDetail_Max);

    fs::SubStorage storages[BucketTreeAnalyzer::StorageIndex_Max - 1] =
    {
        fs::SubStorage(pTableFileStorage, 0, StorageSizeMax),
        fs::SubStorage(pNodeFileStorage, 0, StorageSizeMax),
        fs::SubStorage(pEntryFileStorage, 0, StorageSizeMax),
    };

    auto formatEntry = [&](char* outText, size_t outSize, const void* pData) NN_NOEXCEPT -> size_t
    {
        const auto& entry = *reinterpret_cast<const Entry*>(pData);
        const auto offset = entry.GetOffset();

        BucketTreeAnalyzer::TextFormatter formatter(outText, outSize);

        // エントリ情報の出力
        if( !formatter.Append("0x%016llX %3d", offset, entry.generation) )
        {
            return formatter.GetLength();
        }

        if( !invertedRelocationTable.empty() )
        {
            int64_t nextOffset;
            {
                auto iter = std::lower_bound(
                    m_Entries.begin(),
                    m_Entries.end(),
                    entry,
                    [](const Entry& lhs, const Entry& rhs) NN_NOEXCEPT
                    {
                        return lhs.GetOffset() < rhs.GetOffset();
                    }
                );
                NN_SDK_ASSERT_NOT_EQUAL(iter, m_Entries.end());
                NN_SDK_ASSERT_EQUAL(iter->GetOffset(), offset);

                ++iter;

                nextOffset = iter != m_Entries.end() ? iter->GetOffset() : m_EndOffset;
            }

            // 詳細の出力
            const auto end = invertedRelocationTable.end();
            auto iter = invertedRelocationTable.Find(offset);
            if( iter < end )
            {
                if( !formatter.Append(" (") )
                {
                    return formatter.GetLength();
                }

                for( ; iter < end; ++iter )
                {
                    const auto srcOffset = iter->GetSrcOffset();
                    const auto overlapBegin = std::max(srcOffset, offset);
                    const auto overlapEnd = std::min(srcOffset + iter->GetSize(), nextOffset);
                    const auto overlapSize = overlapEnd - overlapBegin;

                    if( overlapSize <= 0 )
                    {
                        break;
                    }

                    const auto detail = iter->GetDetail();
                    const char* detailText = StatusText[BinaryMatchDetail_Unknown];
                    if( BinaryMatchDetail_Unknown <= detail && detail < BinaryMatchDetail_Max )
                    {
                        detailText = StatusText[detail];
                    }

                    if( detail == BinaryMatchDetail_LargeSplit ||
                        detail == BinaryMatchDetail_SoftSplit ||
                        detail == BinaryMatchDetail_ForceSplit )
                    {
                        if( !formatter.Append("%s%x:0x%llX, ",
                                        detailText, iter->GetDetailIndex(), overlapSize) )
                        {
                            return formatter.GetLength();
                        }
                    }
                    else
                    {
                        if( !formatter.Append("%s:0x%llX, ", detailText, overlapSize) )
                        {
                            return formatter.GetLength();
                        }
                    }
                }

                formatter.Back(2);
                formatter.Append(')');
            }
        }

        return formatter.End();
    };

    BucketTreeAnalyzer analyzer;
    NN_RESULT_DO(analyzer.Initialize(
        storages,
        AesCtrCounterExtendedStorage::NodeSize,
        sizeof(Entry),
        GetEntryCount(),
        startOffset,
        "            Offset/Generation (Detail)",
        formatEntry,
        nullptr
    ));

    char headerBuffer[sizeof(BucketTree::Header)];
    NN_SDK_ASSERT_EQUAL(QueryTableHeaderStorageSize(), static_cast<int64_t>(sizeof(headerBuffer)));
    fs::MemoryStorage headerStorage(headerBuffer, sizeof(headerBuffer));

    // BucketTreeBuilder からの書き出しをハックする
    NN_RESULT_DO(WriteTable(
        pAllocator,
        fs::SubStorage(&headerStorage, 0, sizeof(headerBuffer)),
        fs::SubStorage(analyzer.GetNodeStorage(), 0, QueryTableNodeStorageSize()),
        fs::SubStorage(analyzer.GetEntryStorage(), 0, QueryTableEntryStorageSize())
    ));

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

}}}
