﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <memory>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/save/fs_MappingTable.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>

namespace nn { namespace fssystem { namespace save {

static const uint32_t MappingEntryBufferCountMax = 8 * 1024;

//! コンストラクタ
MappingTable::MappingTable() NN_NOEXCEPT
{
    std::memset(&m_ControlArea, 0, sizeof(m_ControlArea));
}

//! デストラクタ
MappingTable::~MappingTable() NN_NOEXCEPT
{
}

/**
* @brief        マッピングテーブルのメタデータサイズを取得します。
*
* @param[in]    countMapEntry   マッピングテーブルのエントリ数
* @param[in]    countReserved   予約領域のエントリ数
*
* @return       マッピングテーブルのメタデータサイズを返します。
*/
void MappingTable::QueryMappingMetaSize(
         int64_t* outTableSize,
         int64_t* outBitmapSizeUpdatedPhysical,
         int64_t* outBitmapSizeUpdatedVirtual,
         int64_t* outBitmapSizeUnassigned,
         uint32_t countMapEntry,
         uint32_t countReserved
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outTableSize);
    NN_SDK_REQUIRES_NOT_NULL(outBitmapSizeUpdatedPhysical);
    NN_SDK_REQUIRES_NOT_NULL(outBitmapSizeUpdatedVirtual);
    NN_SDK_REQUIRES_NOT_NULL(outBitmapSizeUnassigned);

    const int64_t sizeDataList = dbm::Bitmap::QuerySize(countMapEntry);
    const int64_t sizeWholeList = dbm::Bitmap::QuerySize(countMapEntry + countReserved);
    *outTableSize = sizeof(MappingEntry) * countMapEntry;
    *outBitmapSizeUpdatedPhysical = sizeWholeList;
    *outBitmapSizeUpdatedVirtual = sizeDataList;
    *outBitmapSizeUnassigned = sizeWholeList;
}

/**
* @brief        マッピングテーブルメタデータ領域をフォーマットします。
*
* @param[in]    storageControlArea              管理領域を配置するレイヤー
* @param[in]    storageTable                    マッピングテーブルのメタデータを配置するレイヤー
* @param[in]    storageBitmapUpdatedPhysical    更新済みビットマップ(物理インデックス)を配置するレイヤー
* @param[in]    storageBitmapUpdatedVirtual     更新済みビットマップ(論理インデックス)を配置するレイヤー
* @param[in]    storageBitmapUnassigned         未割当ビットマップ(物理インデックス)を配置するレイヤー
* @param[in]    countMapEntry                   マッピングテーブルのエントリ数
* @param[in]    countReserved                   予約領域のエントリ数
*
* @return       関数の処理結果を返します。
*
* @details      マッピングテーブルが管理するブロック数を拡張します。
*               マウントしていない状態で実行してください。
*/
Result MappingTable::Format(
           fs::SubStorage storageControlArea,
           fs::SubStorage storageTable,
           fs::SubStorage storageBitmapUpdatedPhysical,
           fs::SubStorage storageBitmapUpdatedVirtual,
           fs::SubStorage storageBitmapUnassigned,
           uint32_t countMapEntry,
           uint32_t countReserved
       ) NN_NOEXCEPT
{
    const uint32_t bitCount = countMapEntry + countReserved;

    // 管理領域を初期化します。
    ControlArea header;
    std::memset(&header, 0, sizeof(header));
    header.version = CONTROL_AREA_VERSION;
    header.countMapEntry = countMapEntry;
    header.countReserved = countReserved;
    NN_RESULT_DO(storageControlArea.Write(0, &header, sizeof(header)));

    // マッピングテーブル領域を初期化します。
    {
        // ライトバッファを確保します。
        static const uint32_t WriteBufferCountMax = MappingEntryBufferCountMax;
        const uint32_t writeBufferCount = std::min(header.countMapEntry, WriteBufferCountMax);
        const auto sizeOfWorkBuffer = writeBufferCount * sizeof(MappingEntry);
        PooledBuffer pooledBuffer(sizeOfWorkBuffer, sizeOfWorkBuffer);
        NN_SDK_ASSERT_GREATER_EQUAL(pooledBuffer.GetSize(), sizeOfWorkBuffer);
        MappingEntry* writeBuffer = reinterpret_cast<MappingEntry*>(pooledBuffer.GetBuffer());

        for( uint32_t writeCount = 0; writeCount < header.countMapEntry; )
        {
            // フォーマット内容をライトバッファにまとめます。
            uint32_t bufferIndex = 0;
            for( ; bufferIndex + writeCount < header.countMapEntry
                    && bufferIndex < writeBufferCount;
                ++bufferIndex )
            {
                MappingEntry& entry = writeBuffer[bufferIndex];
                entry.indexPhysical = PhysicalIndex(bufferIndex + writeCount);
                entry.indexPhysicalPrev = 0;
            }
            const uint32_t useBufferCount = bufferIndex;

            // ライトバッファでストレージをフォーマットします。
            NN_RESULT_DO(
                storageTable.Write(
                    sizeof(MappingEntry) * writeCount,
                    writeBuffer,
                    sizeof(MappingEntry) * useBufferCount
                )
            );
            writeCount += useBufferCount;
        }
    }

    // 更新済みビットマップ(物理インデックス)を初期化します。
    NN_RESULT_DO(dbm::Bitmap::Format(bitCount, storageBitmapUpdatedPhysical));

    // 更新済みビットマップ(論理インデックス)を初期化します。
    NN_RESULT_DO(dbm::Bitmap::Format(countMapEntry, storageBitmapUpdatedVirtual));

    // 未割当ビットマップ(物理インデックス)を初期化します。
    NN_RESULT_DO(dbm::Bitmap::Format(bitCount, storageBitmapUnassigned));

    // フォーマット直後は予約領域はすべて未割当状態としておきます。
    dbm::Bitmap bitmapUnassigned;
    bitmapUnassigned.Initialize(bitCount, storageBitmapUnassigned);
    NN_RESULT_DO(bitmapUnassigned.Reverse(countMapEntry, countReserved));

    NN_RESULT_SUCCESS;
}

/**
* @brief        マッピングテーブルメタデータ領域を拡張します。
*
* @param[in]    storageControlArea              管理領域を配置するレイヤー
* @param[in]    storageTable                    マッピングテーブルを配置するレイヤー
* @param[in]    storageBitmapUpdatedPhysical    更新済みビットマップ(物理インデックス)を配置するレイヤー
* @param[in]    storageBitmapUpdatedVirtual     更新済みビットマップ(論理インデックス)を配置するレイヤー
* @param[in]    storageBitmapUnassigned         未割当ビットマップ(物理インデックス)を配置するレイヤー
* @param[in]    countMapEntryNew                拡張後のマッピングテーブルのエントリ数
* @param[in]    countReservedNew                拡張後の予約領域のエントリ数
*
* @return       関数の処理結果を返します。
*
* @details      エントリ数は以前の個数よりも大きくすることしかできません。
*               エントリ数が以前よりも小さい場合はエラーを返します。
*/
Result MappingTable::Expand(
           fs::SubStorage storageControlArea,
           fs::SubStorage storageTable,
           fs::SubStorage storageBitmapUpdatedPhysical,
           fs::SubStorage storageBitmapUpdatedVirtual,
           fs::SubStorage storageBitmapUnassigned,
           uint32_t countMapEntryNew,
           uint32_t countReservedNew
       ) NN_NOEXCEPT
{
    // 管理領域を読み込みます。
    ControlArea header;
    NN_RESULT_DO(storageControlArea.Read(0, &header, sizeof(header)));

    // 管理領域の正当性チェックを行ないます。
    if( header.version > CONTROL_AREA_VERSION )
    {
        return nn::fs::ResultUnsupportedVersion();
    }

    uint32_t countMapEntryOld = header.countMapEntry;
    uint32_t countReservedOld = header.countReserved;

    // 縮小することはできません。
    if( countMapEntryOld > countMapEntryNew )
    {
        return nn::fs::ResultInvalidSize();
    }
    if( countReservedOld > countReservedNew )
    {
        return nn::fs::ResultInvalidSize();
    }

    const uint32_t bitCountOld = countMapEntryOld + countReservedOld;
    const uint32_t bitCountNew = countMapEntryNew + countReservedNew;

    // 拡張された範囲のエントリを初期化します。
    for( uint32_t indexEntry = 0; indexEntry < countMapEntryNew - countMapEntryOld; ++indexEntry )
    {
        MappingEntry entry;
        entry.indexPhysical = PhysicalIndex(countMapEntryOld + countReservedOld + indexEntry);
        entry.indexPhysicalPrev = 0;
        NN_RESULT_DO(
            storageTable.Write(
                sizeof(MappingEntry) * (countMapEntryOld + indexEntry),
                &entry,
                sizeof(MappingEntry)
            )
        );
    }

    if( bitCountNew > bitCountOld )
    {
        // 更新済みビットマップ(物理インデックス)を拡張します。
        NN_RESULT_DO(dbm::Bitmap::Expand(bitCountOld, bitCountNew, storageBitmapUpdatedPhysical));

        // 更新済みビットマップ(論理インデックス)を拡張します。
        if( countMapEntryNew > countMapEntryOld )
        {
            NN_RESULT_DO(
                dbm::Bitmap::Expand(
                    countMapEntryOld,
                    countMapEntryNew,
                    storageBitmapUpdatedVirtual
                )
            );
        }

        // 未割当ビットマップ(物理インデックス)を拡張します。
        NN_RESULT_DO(dbm::Bitmap::Expand(bitCountOld, bitCountNew, storageBitmapUnassigned));

        // 予約領域はすべて未割当状態としておきます。
        if( countReservedNew > countReservedOld )
        {
            dbm::Bitmap bitmapUnassigned;
            bitmapUnassigned.Initialize(bitCountNew, storageBitmapUnassigned);
            NN_RESULT_DO(
                bitmapUnassigned.Reverse(
                    countMapEntryNew + countReservedOld,
                    countReservedNew - countReservedOld
                )
            );
        }
    }

    // 管理領域を更新します。
    header.countMapEntry = countMapEntryNew;
    header.countReserved = countReservedNew;
    NN_RESULT_DO(storageControlArea.Write(0, &header, sizeof(header)));

    NN_RESULT_SUCCESS;
}

/**
* @brief        マッピングテーブルレイヤーとしてマウントします。
*
* @param[in]    storageControlArea              管理領域を配置するレイヤー
* @param[in]    storageTable                    マッピングテーブルを配置するレイヤー
* @param[in]    storageBitmapUpdatedPhysical    更新済みビットマップ(物理インデックス)を配置するレイヤー
* @param[in]    storageBitmapUpdatedVirtual     更新済みビットマップ(論理インデックス)を配置するレイヤー
* @param[in]    storageBitmapUnassigned         未割当ビットマップ(物理インデックス)を配置するレイヤー
*
* @return       関数の処理結果を返します。
*/
Result MappingTable::Initialize(
           fs::SubStorage storageControlArea,
           fs::SubStorage storageTable,
           fs::SubStorage storageBitmapUpdatedPhysical,
           fs::SubStorage storageBitmapUpdatedVirtual,
           fs::SubStorage storageBitmapUnassigned
       ) NN_NOEXCEPT
{
    m_StorageTable = storageTable;
    m_StorageBitmapUpdatedPhysical = storageBitmapUpdatedPhysical;
    m_StorageBitmapUpdatedVirtual = storageBitmapUpdatedVirtual;
    m_StorageBitmapUnassigned = storageBitmapUnassigned;

    // 管理領域を読み込みます。
    NN_RESULT_DO(storageControlArea.Read(0, &m_ControlArea, sizeof(m_ControlArea)));

    // 管理領域の正当性チェックを行ないます。
    if( m_ControlArea.version > CONTROL_AREA_VERSION )
    {
        return nn::fs::ResultUnsupportedVersion();
    }

    // ビットマップをマウントします。
    const uint32_t countBit = m_ControlArea.countMapEntry + m_ControlArea.countReserved;
    m_BitmapUpdatedPhysical.Initialize(countBit, m_StorageBitmapUpdatedPhysical);
    m_BitmapUpdatedVirtual.Initialize(m_ControlArea.countMapEntry, m_StorageBitmapUpdatedVirtual);
    m_BitmapUnassigned.Initialize(countBit, m_StorageBitmapUnassigned);

    NN_RESULT_SUCCESS;
}

/**
* @brief        総ブロック数を取得します。
*
* @return       総ブロック数を返します。
*/
uint32_t MappingTable::GetTotalBlockCount() const NN_NOEXCEPT
{
    return m_ControlArea.countMapEntry + m_ControlArea.countReserved;
}

/**
* @brief        物理インデックスとして範囲内に収まっているかを確認します。
*
* @param[in]    index  物理インデックス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       物理インデックスとして正常な値
* @retval       ResultInvalidOffset 物理インデックスではない
* @retval       ResultInvalidOffset 範囲内に収まっていない
*/
Result MappingTable::CheckPhysicalIndex(uint32_t index) const NN_NOEXCEPT
{
    if( ! IsPhysicalIndex(index) )
    {
        return nn::fs::ResultInvalidOffset();
    }

    if( GetValidIndex(index) >= GetTotalBlockCount() )
    {
        return nn::fs::ResultInvalidOffset();
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        論理インデックスとして範囲内に収まっているかを確認します。
*
* @param[in]    index   論理インデックス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       物理インデックスとして正常な値
* @retval       ResultInvalidOffset 物理インデックスではない
* @retval       ResultInvalidOffset 範囲内に収まっていない
*/
Result MappingTable::CheckVirtualIndex(uint32_t index) const NN_NOEXCEPT
{
    if( IsPhysicalIndex(index) )
    {
        return nn::fs::ResultInvalidOffset();
    }

    if( index >= GetTotalBlockCount() )
    {
        return nn::fs::ResultInvalidOffset();
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        マッピングテーブルからエントリーを読み込みます。
*
* @param[out]   outValue    エントリー読込先
* @param[in]    index       論理インデックス
*
* @return       関数の処理結果を返します。
*
* @pre
*               - outValue != nullptr
*/
Result MappingTable::ReadMappingEntry(
           MappingEntry* outValue,
           uint32_t index
       ) NN_NOEXCEPT
{
    // パラメータチェック
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_RESULT_DO(CheckVirtualIndex(index));

    NN_RESULT_DO(
        m_StorageTable.Read(
           sizeof(MappingEntry) * index,
           outValue,
           sizeof(MappingEntry)
       )
    );

    // 読み込んだ値のチェック
    NN_RESULT_DO(CheckPhysicalIndex(outValue->indexPhysical));

    NN_RESULT_SUCCESS;
}

/**
* @brief    マッピングテーブルから複数のエントリーを読み込みます。
*
* @param[out] outValue      エントリー読込先
* @param[in] bufferCount    エントリー読込先バッファの個数
* @param[in] index          読み込むエントリー先頭の論理インデックス
* @param[in] count          読み込むエントリーの個数
*
* @return   関数の処理結果を返します。
*
* @pre
*               - outValue != nullptr
*               - bufferCount >= count
*/
Result MappingTable::ReadMappingEntries(
           MappingEntry* outValue,
           size_t bufferCount,
           uint32_t index,
           size_t count
       ) NN_NOEXCEPT
{
    // パラメータチェック
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_GREATER_EQUAL(bufferCount, count);
    NN_UNUSED(bufferCount);

    for( uint32_t i = index; i < index + count; ++i )
    {
        NN_RESULT_DO(CheckVirtualIndex(i));
    }

    NN_RESULT_DO(
        m_StorageTable.Read(
           sizeof(MappingEntry) * index,
           outValue,
           sizeof(MappingEntry) * count
       )
    );

    // 読み込んだ値のチェック
    for( uint32_t i = 0; i < count; ++i )
    {
        NN_RESULT_DO(CheckPhysicalIndex(outValue[i].indexPhysical));
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        マッピングテーブルにエントリーを保存します。
*
* @param[in]    entry   書き込むエントリー
* @param[in]    index   論理インデックス
*
* @return       関数の処理結果を返します。
*/
Result MappingTable::WriteMappingEntry(
           const MappingEntry& entry,
           uint32_t index
       ) NN_NOEXCEPT
{
    // パラメータチェック
    NN_RESULT_DO(CheckVirtualIndex(index));
    NN_RESULT_DO(CheckPhysicalIndex(entry.indexPhysical));

    NN_RESULT_DO(
        m_StorageTable.Write(
           sizeof(MappingEntry) * index,
           &entry,
           sizeof(MappingEntry)
        )
    );

    NN_RESULT_SUCCESS;
}

/**
* @brief        マッピングテーブルに複数のエントリーを保存します。
*
* @param[in]    pEntries    書き込むエントリー
* @param[in]    count       エントリーの個数
* @param[in]    index       書き込むエントリー先頭の論理インデックス
*
* @return       関数の処理結果を返します。
*
* @pre
*               - pEntries != nullptr
*               - 0 <= count
*/
Result MappingTable::WriteMappingEntries(
           const MappingEntry* pEntries,
           int count,
           uint32_t index
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEntries);
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

    // パラメータチェック
    for( auto indexCheck = 0; indexCheck < count; ++indexCheck )
    {
        NN_RESULT_DO(CheckVirtualIndex(index + indexCheck));
        NN_RESULT_DO(CheckPhysicalIndex(pEntries[indexCheck].indexPhysical));
    }

    return m_StorageTable.Write(
               sizeof(MappingEntry) * index,
               pEntries,
               sizeof(MappingEntry) * count
           );
}

/**
* @brief        論理/物理インデックスから物理インデックスに変換します。
*
* @param[out]   outValue    物理インデックス
* @param[in]    index       論理/物理インデックス
*
* @return       関数の処理結果を返します。
*
* @pre
*               - outValue != nullptr
*/
Result MappingTable::GetPhysicalIndex(
           uint32_t* outValue,
           uint32_t index
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);

    // 物理インデックスかどうか判定します。
    if( IsPhysicalIndex(index) )
    {
        *outValue = GetValidIndex(index);
        NN_RESULT_SUCCESS;
    }

    // 論理インデックスから物理インデックスに変換します。
    MappingEntry entry;
    NN_RESULT_DO(ReadMappingEntry(&entry, index));
    NN_RESULT_DO(CheckPhysicalIndex(entry.indexPhysical));

    *outValue = GetValidIndex(entry.indexPhysical);

    NN_RESULT_SUCCESS;
}

/**
* @brief        要求したブロック数に応じた連続ブロックを確保します。
*
* @param[out]   outIndexDst     確保した先頭物理インデックス
* @param[out]   outCountAssign  実際に確保できたブロック数
* @param[in]    indexStart      割り当てる論理インデックス
* @param[in]    countRequired   確保したいブロック数
*
* @return       関数の処理結果を返します。
*
* @details
*   実際に確保できたサイズは outCountAssign に格納されます。
*   指定したブロック数に届かない場合は、確保できたブロック数を
*   引いてから、再度この関数を実行してください。
*
*   未割当領域が足りなくなると、どこまで書き込んだかわからない状態になります。
*   その時には、エラーを示す Result を返します。
*
* @pre
*               - outIndexDst != nullptr
*               - outCountAssign != nullptr
*/
Result MappingTable::MarkAssignment(
           uint32_t* outIndexDst,
           uint32_t* outCountAssign,
           uint32_t indexStart,
           uint32_t countRequired
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndexDst);
    NN_SDK_REQUIRES_NOT_NULL(outCountAssign);
    NN_RESULT_DO(CheckVirtualIndex(indexStart));

    // 要求されたブロック数にふさわしい空き領域を探索します。
    // 現実装では一番最初に見つけた覆えるものを最優先で採用します。
    uint32_t indexCurr = 0;
    bool findMatch = false;
    bool findNear = false;
    uint32_t countMatch = 0;
    uint32_t indexNear = 0;
    uint32_t countNear = 0;
    dbm::Bitmap::Iterator it;
    NN_RESULT_DO(m_BitmapUnassigned.IterateBegin(&it, 0));
    while( indexCurr < GetTotalBlockCount() )
    {
        uint32_t countZero;
        uint32_t countOne;
        NN_RESULT_DO(
            m_BitmapUnassigned.LimitedIterateNext(&countZero, &countOne, &it, 0, countRequired));

        // イテレーション終了
        if( (countOne == 0) && (countZero == 0) )
        {
            break;
        }

        if( countOne >= countRequired )
        {
            findMatch = true;
            countMatch = countOne;
            break;
        }
        if( countOne > 0 )
        {
            findNear = true;
            if( countNear < countOne )
            {
                indexNear = indexCurr;
                countNear = countOne;
            }
        }

        indexCurr += countZero + countOne;
    }

    uint32_t indexVirtual;
    uint32_t countAssigned;
    if( findMatch )
    {
        // 確保成功
        indexVirtual = indexCurr;
        countAssigned = std::min(countMatch, countRequired);
    }
    else if( findNear )
    {
        // 部分確保成功
        indexVirtual = indexNear;
        countAssigned = countNear;
    }
    else
    {
        // 空きがなくなりました。
        return nn::fs::ResultMappingTableFull();
    }

    *outIndexDst = PhysicalIndex(indexVirtual);
    *outCountAssign = countAssigned;

    // 割当てを行なう範囲のビットマップを反転します。
    NN_RESULT_DO(m_BitmapUpdatedVirtual.Reverse(indexStart, countAssigned));
    NN_RESULT_DO(m_BitmapUpdatedPhysical.Reverse(indexVirtual, countAssigned));

    // マッピングテーブルを更新します。
    NN_RESULT_DO(
        UpdateMapTable(
            PhysicalIndex(indexVirtual),
            indexStart,
            countAssigned
        )
    );

    // 消費したブロック数削ります。
    NN_RESULT_DO(m_BitmapUnassigned.Reverse(indexVirtual, countAssigned));

    NN_RESULT_SUCCESS;
}

/**
* @brief        データ書き込み前にマッピングテーブルを更新します。
*
* @param[out]   outNeedCopyHead 先頭ブロックのコピーが必要かどうか
* @param[out]   outHeadSrc      書き込み前の先頭物理インデックス
* @param[out]   outHeadDst      書き込み先の先頭物理インデックス
* @param[out]   outNeedCopyTail 終端ブロックのコピーが必要かどうか
* @param[out]   outTailSrc      書き込み前の終端物理インデックス
* @param[out]   outTailDst      書き込み先の終端物理インデック
* @param[in]    indexStart      書き込み開始論理インデックス
* @param[in]    countContinue   確保する連続ブロック数
*
* @return       関数の処理結果を返します。
*
* @pre
*               - outNeedCopyHead != nullptr
*               - outHeadSrc != nullptr
*               - outHeadDst != nullptr
*               - outNeedCopyTail != nullptr
*               - outTailSrc != nullptr
*               - outTailDst != nullptr
*/
Result MappingTable::MarkUpdateRecursively(
           bool* outNeedCopyHead,
           uint32_t* outHeadSrc,
           uint32_t* outHeadDst,
           bool* outNeedCopyTail,
           uint32_t* outTailSrc,
           uint32_t* outTailDst,
           uint32_t indexStart,
           uint32_t countContinue
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outNeedCopyHead);
    NN_SDK_REQUIRES_NOT_NULL(outHeadSrc);
    NN_SDK_REQUIRES_NOT_NULL(outHeadDst);
    NN_SDK_REQUIRES_NOT_NULL(outNeedCopyTail);
    NN_SDK_REQUIRES_NOT_NULL(outTailSrc);
    NN_SDK_REQUIRES_NOT_NULL(outTailDst);

    // 更新範囲のうち、更新済みブロックを探索します。
    // 更新が必要な範囲については未割当ビットマップから取得します。
    uint32_t indexCheck = indexStart;
    uint32_t countCheck = countContinue;
    dbm::Bitmap::Iterator it;
    NN_RESULT_DO(m_BitmapUpdatedVirtual.IterateBegin(&it, indexStart));
    while( countCheck > 0 )
    {
        uint32_t countZero;
        uint32_t countOne;
        NN_RESULT_DO(
            m_BitmapUpdatedVirtual.LimitedIterateNext(&countZero, &countOne, &it, countCheck, 0)
        );

        if( countOne > 0 )
        {
            // 現在のブロック位置から連続する更新済み
            uint32_t countFind = std::min(countOne, countCheck);
            indexCheck += countFind;
            countCheck -= countFind;
        }
        else if( countZero > 0 )
        {
            // 最寄に更新済み領域が存在している場合は
            // その領域までを更新が必要な領域としてビットマップに追加する
            uint32_t indexRequiredStart = indexCheck;
            uint32_t countRequired = std::min(countCheck, countZero);
            while( countRequired > 0 )
            {
                // 先頭ブロックが新規確保ブロックに含まれているかどうかを確認します。
                // 含まれていない場合、コピー処理が必要な場合があります。
                // (実際に必要かどうかは上のレイヤーで判定します)
                if( indexRequiredStart == indexStart )
                {
                    *outNeedCopyHead = true;

                    // コピー元の物理インデックスを取得します。
                    MappingEntry entrySrc;
                    NN_RESULT_DO(ReadMappingEntry(&entrySrc, indexRequiredStart));
                    *outHeadSrc = entrySrc.indexPhysical;
                }
                // 終端ブロックが新規確保ブロックに含まれているかどうかを確認します。
                // 含まれていない場合、コピー処理が必要な場合があります。
                // (実際に必要かどうかは上のレイヤーで判定します)
                if( (countContinue > 1)
                 && (indexRequiredStart + countRequired == indexStart + countContinue) )
                {
                    // コピー元の物理インデックスを取得します。
                    MappingEntry entrySrc;
                    NN_RESULT_DO(ReadMappingEntry(&entrySrc, indexStart + countContinue - 1));
                    *outTailSrc = entrySrc.indexPhysical;
                }

                // 未割当ビットマップからブロック割当を行ないます。
                uint32_t indexDst;
                uint32_t countAssign;
                NN_RESULT_DO(
                    MarkAssignment(
                        &indexDst,
                        &countAssign,
                        indexRequiredStart,
                        countRequired
                    )
                );

                // 先頭ブロックが新規確保ブロックに含まれているかどうかを確認します。
                // 含まれていない場合、コピー処理が必要な場合があります。
                // (実際に必要かどうかは上のレイヤーで判定します)
                if( indexRequiredStart == indexStart )
                {
                    *outHeadDst = indexDst;
                }
                // 終端ブロックが新規確保ブロックに含まれているかどうかを確認します。
                // 含まれていない場合、コピー処理が必要な場合があります。
                // (実際に必要かどうかは上のレイヤーで判定します)
                if( (countContinue > 1)
                 && (indexRequiredStart + countAssign == indexStart + countContinue) )
                {
                    *outNeedCopyTail = true;
                    *outTailDst = indexDst + countAssign - 1;
                }

                indexRequiredStart += countAssign;
                countRequired -= countAssign;
            }

            // 更新が必要な範囲を確保し終わるまで再帰的に
            // 未割当ビットマップから取得します。
            NN_RESULT_DO(
                MarkUpdateRecursively(
                   outNeedCopyHead,
                   outHeadSrc,
                   outHeadDst,
                   outNeedCopyTail,
                   outTailSrc,
                   outTailDst,
                   indexStart,
                   countContinue
               )
            );

            break;
        }
        else
        {
            return nn::fs::ResultOutOfRange();
        }
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        データ書き込み前にマッピングテーブルを更新します。
*
* @param[out]   outNeedCopyHead 先頭ブロックのコピーが必要かどうか
* @param[out]   outHeadSrc      書き込み前の先頭物理インデックス
* @param[out]   outHeadDst      書き込み先の先頭物理インデックス
* @param[out]   outNeedCopyTail 終端ブロックのコピーが必要かどうか
* @param[out]   outTailSrc      書き込み前の終端物理インデックス
* @param[out]   outTailDst      書き込み先の終端物理インデック
* @param[in]    indexStart      書き込み開始論理インデックス
* @param[in]    countContinue   確保する連続ブロック数
*
* @return       関数の処理結果を返します。
*
* @pre
*               - outNeedCopyHead != nullptr
*               - outHeadSrc != nullptr
*               - outHeadDst != nullptr
*               - outNeedCopyTail != nullptr
*               - outTailSrc != nullptr
*               - outTailDst != nullptr
*/
Result MappingTable::MarkUpdate(
           bool* outNeedCopyHead,
           uint32_t* outHeadSrc,
           uint32_t* outHeadDst,
           bool* outNeedCopyTail,
           uint32_t* outTailSrc,
           uint32_t* outTailDst,
           uint32_t indexStart,
           uint32_t countContinue
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outNeedCopyHead);
    NN_SDK_REQUIRES_NOT_NULL(outHeadSrc);
    NN_SDK_REQUIRES_NOT_NULL(outHeadDst);
    NN_SDK_REQUIRES_NOT_NULL(outNeedCopyTail);
    NN_SDK_REQUIRES_NOT_NULL(outTailSrc);
    NN_SDK_REQUIRES_NOT_NULL(outTailDst);
    NN_RESULT_DO(CheckVirtualIndex(indexStart));

    *outNeedCopyHead = false;
    *outHeadSrc = 0;
    *outHeadDst = 0;
    *outNeedCopyTail = false;
    *outTailSrc = 0;
    *outTailDst = 0;

    NN_RESULT_DO(
        MarkUpdateRecursively(
            outNeedCopyHead,
            outHeadSrc,
            outHeadDst,
            outNeedCopyTail,
            outTailSrc,
            outTailDst,
            indexStart,
            countContinue
        )
    );

    if( *outNeedCopyHead )
    {
        NN_RESULT_DO(CheckPhysicalIndex(*outHeadSrc));
        NN_RESULT_DO(CheckPhysicalIndex(*outHeadDst));
    }
    if( *outNeedCopyTail )
    {
        NN_RESULT_DO(CheckPhysicalIndex(*outTailSrc));
        NN_RESULT_DO(CheckPhysicalIndex(*outTailDst));
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        イテレーションを開始します。
*
* @param[out]   it          イテレーター
* @param[in]    index       開始インデックス
* @param[in]    countMax    イテレーションするブロック数
*
* @return       関数の処理結果を返します。
*
* @pre
*               - it != nullptr
*/
Result MappingTable::MakeIterator(
           Iterator* it,
           uint32_t index,
           uint32_t countMax
       ) NN_NOEXCEPT
{
    // パラメータチェック
    NN_SDK_REQUIRES_NOT_NULL(it);
    if( index >= m_ControlArea.countMapEntry )
    {
        return fs::ResultOutOfRange();
    }

    MappingEntry entry;
    NN_RESULT_DO(ReadMappingEntry(&entry, index));

    it->m_IndexEntry = index;
    it->m_IndexPhysical = entry.indexPhysical;
    it->m_CountRemain = countMax;

    // 連続しているブロック数を求めます。
    MappingEntry entryHead;
    NN_RESULT_DO(ReadMappingEntry(&entryHead, index));
    uint32_t countContinue = 1;
    while( (index + countContinue < m_ControlArea.countMapEntry)
        && (countContinue < it->m_CountRemain) )
    {
        MappingEntry entryCurr;
        NN_RESULT_DO(ReadMappingEntry(&entryCurr, index + countContinue));
        if( entryCurr.indexPhysical != entryHead.indexPhysical + countContinue )
        {
            break;
        }
        countContinue++;
    }
    it->m_CountContinue = countContinue;

    NN_SDK_ASSERT(it->m_CountRemain >= it->m_CountContinue);
    it->m_CountRemain -= it->m_CountContinue;

    NN_RESULT_SUCCESS;
}

/**
* @brief        イテレーターが指す位置を更新します。
*
* @param[in]    it  イテレーター
*
* @return       関数の処理結果を返します。
*
* @pre
*               - it != nullptr
*/
Result MappingTable::UpdateIterator(Iterator* it) NN_NOEXCEPT
{
    // パラメータチェック
    NN_SDK_REQUIRES_NOT_NULL(it);
    if( it->m_IndexEntry >= m_ControlArea.countMapEntry )
    {
        return fs::ResultOutOfRange();
    }

    it->m_IndexEntry += it->m_CountContinue;

    MappingEntry entry;
    NN_RESULT_DO(ReadMappingEntry(&entry, it->m_IndexEntry));

    it->m_IndexPhysical = entry.indexPhysical;

    // 連続しているブロック数を求めます。
    uint32_t index = it->m_IndexEntry;
    MappingEntry entryHead;
    NN_RESULT_DO(ReadMappingEntry(&entryHead, index));
    uint32_t countContinue = 1;
    while( (index + countContinue < m_ControlArea.countMapEntry)
        && (countContinue < it->m_CountRemain) )
    {
        MappingEntry entryCurr;
        NN_RESULT_DO(ReadMappingEntry(&entryCurr, index + countContinue));
        if( entryCurr.indexPhysical != entryHead.indexPhysical + countContinue )
        {
            break;
        }
        countContinue++;
    }
    it->m_CountContinue = countContinue;

    NN_SDK_ASSERT(it->m_CountRemain >= it->m_CountContinue);
    it->m_CountRemain -= it->m_CountContinue;

    NN_RESULT_SUCCESS;
}

/**
* @brief       キャッシュを無効化します。
*/
Result MappingTable::InvalidateCache() NN_NOEXCEPT
{
    const auto InvalidateStorageCache = [](nn::fs::IStorage* pStorage) NN_NOEXCEPT -> nn::Result
    {
        int64_t sizeStorage = 0;
        NN_RESULT_DO(pStorage->GetSize(&sizeStorage));
        NN_RESULT_DO(
            pStorage->OperateRange(
                fs::OperationId::Invalidate,
                0,
                sizeStorage
            )
        );
        NN_RESULT_SUCCESS;
    };
    NN_RESULT_DO(InvalidateStorageCache(&m_StorageTable));
    NN_RESULT_DO(InvalidateStorageCache(&m_StorageBitmapUpdatedPhysical));
    NN_RESULT_DO(InvalidateStorageCache(&m_StorageBitmapUpdatedVirtual));
    NN_RESULT_DO(InvalidateStorageCache(&m_StorageBitmapUnassigned));
    NN_RESULT_SUCCESS;
}

/**
* @brief        マッピングテーブル上の論理インデックス、物理インデックスの変換測を更新します。
*
* @param[in]    indexPhysical   物理インデックス
* @param[in]    indexVirtual    論理インデックス
* @param[in]    countContinue   ブロック数(1以上)
*
* @return       関数の処理結果を返します。
*/
Result MappingTable::UpdateMapTable(
           uint32_t indexPhysical,
           uint32_t indexVirtual,
           uint32_t countContinue
       ) NN_NOEXCEPT
{
    // パラメータチェック
    NN_SDK_ASSERT(countContinue > 0);
    NN_RESULT_DO(CheckPhysicalIndex(indexPhysical));
    NN_RESULT_DO(CheckPhysicalIndex(indexPhysical + countContinue - 1));
    NN_RESULT_DO(CheckVirtualIndex(indexVirtual));
    NN_RESULT_DO(CheckVirtualIndex(indexVirtual + countContinue - 1));

    const auto countEntryBuffer = std::min(countContinue, MappingEntryBufferCountMax);
    const auto sizeOfWorkBuffer = countEntryBuffer * sizeof(MappingEntry);
    PooledBuffer pooledBuffer(sizeOfWorkBuffer, sizeOfWorkBuffer);
    NN_SDK_ASSERT_GREATER_EQUAL(pooledBuffer.GetSize(), sizeOfWorkBuffer);
    const auto entryBuffer = reinterpret_cast<MappingEntry*>(pooledBuffer.GetBuffer());

    // 更新範囲のマッピングテーブルを更新します。
    uint32_t countUpdated = 0;
    while( countUpdated < countContinue )
    {
        const auto countUpdate = std::min(countContinue - countUpdated, countEntryBuffer);
        NN_RESULT_DO(
            ReadMappingEntries(
                entryBuffer,
                countEntryBuffer,
                indexVirtual + countUpdated,
                countUpdate
            )
        );
        for( uint32_t indexUpdate = 0; indexUpdate < countUpdate; ++indexUpdate )
        {
            auto& entry = entryBuffer[indexUpdate];
            entry.indexPhysicalPrev = entry.indexPhysical;
            entry.indexPhysical = indexPhysical + countUpdated + indexUpdate;
        }
        NN_RESULT_DO(
            WriteMappingEntries(
                entryBuffer,
                countUpdate,
                indexVirtual + countUpdated
            )
        );
        countUpdated += countUpdate;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        未割当ビットマップに対して、全ての未使用になった範囲のビットを立てます。
*
* @param[in]    isCommit    コミットかどうか
*
* @return       関数の処理結果を返します。
*
* @details      コミットなら更新前に割り当てられていた領域を未使用にし、
*               ロールバックなら新たに割り当てられた領域を未使用にします。
*/
Result MappingTable::AddUnassignedListIteratively(
           bool isCommit
       ) NN_NOEXCEPT
{
    // 空いたエントリーを未割当ビットマップに移動します。
    uint32_t indexCurr = 0;
    dbm::Bitmap::Iterator it;

    NN_RESULT_DO(m_BitmapUpdatedVirtual.IterateBegin(&it, 0));

    while( indexCurr < GetTotalBlockCount() )
    {
        uint32_t countZero;
        uint32_t countOne;
        NN_RESULT_DO(m_BitmapUpdatedVirtual.IterateNext(&countZero, &countOne, &it));

        // イテレーション終了
        if( (countOne == 0) && (countZero == 0) )
        {
            break;
        }

        // 未割当ビットマップに対して有効ビットを立てます。
        if( countOne > 0 )
        {
            NN_RESULT_DO(AddUnassignedList(indexCurr, countOne, isCommit));
        }

        indexCurr += countZero + countOne;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        未割当ビットマップに対して、未使用になった範囲のビットを立てます。
*
* @param[in]    index       開始論理インデックス
* @param[in]    count       連続論理ブロック数
* @param[in]    isCommit    コミットかどうか
*
* @return       関数の処理結果を返します。
*
* @details      コミットなら更新前に割り当てられていた領域を未割当にし、
*               ロールバックなら新たに割り当てられた領域を未割当にします。
*/
Result MappingTable::AddUnassignedList(
           uint32_t index,
           uint32_t countOne,
           bool isCommit
       ) NN_NOEXCEPT
{
    const auto countEntryBuffer = std::min(countOne, MappingEntryBufferCountMax);
    const auto sizeOfWorkBuffer = countEntryBuffer * sizeof(MappingEntry);
    PooledBuffer pooledBuffer(sizeOfWorkBuffer, sizeOfWorkBuffer);
    NN_SDK_ASSERT_GREATER_EQUAL(pooledBuffer.GetSize(), sizeOfWorkBuffer);
    const auto entryBuffer = reinterpret_cast<MappingEntry*>(pooledBuffer.GetBuffer());

    const auto UnassignEntry = [isCommit](MappingEntry* pEntry) NN_NOEXCEPT
    {
        if( isCommit )
        {
            // コミットする場合 indexPhysicalPrev をリセットします。
            pEntry->indexPhysicalPrev = 0;
        }
        else
        {
            // ロールバックする場合 indexPhysical をリセットします。
            pEntry->indexPhysical = pEntry->indexPhysicalPrev;
            pEntry->indexPhysicalPrev = 0;
        }
    };

    // 連続している物理ブロック数を取得します。
    // 分断化している場合は複数回に分けて処理を行ないます。
    uint32_t count = 0;
    while( count < countOne )
    {
        MappingEntry entryHead;
        NN_RESULT_DO(ReadMappingEntry(&entryHead, index + count));
        uint32_t countCur = 1;
        while( count + countCur < countOne )
        {
            const auto countUnassign = std::min(countOne - (count + countCur), countEntryBuffer);
            uint32_t countUnassigned = 0;

            NN_RESULT_DO(
                ReadMappingEntries(
                    entryBuffer,
                    countEntryBuffer,
                    index + count + countCur,
                    countUnassign
                )
            );

            while( countUnassigned < countUnassign )
            {
                const auto pEntryCur = entryBuffer + countUnassigned;
                if( pEntryCur->indexPhysicalPrev
                    != entryHead.indexPhysicalPrev + countCur + countUnassigned )
                {
                    break;
                }
                UnassignEntry(pEntryCur);
                ++countUnassigned;
            }

            NN_RESULT_DO(
                WriteMappingEntries(
                    entryBuffer,
                    countUnassigned,
                    index + count + countCur
                )
            );

            countCur += countUnassigned;

            if( countUnassigned < countUnassign )
            {
                break;
            }
        }

        NN_RESULT_DO(
            m_BitmapUnassigned.Reverse(
                GetValidIndex(isCommit ? entryHead.indexPhysicalPrev : entryHead.indexPhysical),
                countCur
            )
        );

        UnassignEntry(&entryHead);
        NN_RESULT_DO(WriteMappingEntry(entryHead, index + count));

        count += countCur;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        コミット操作を実行します。
*
* @return       関数の処理結果を返します。
*/
Result MappingTable::Commit() NN_NOEXCEPT
{
    NN_RESULT_DO(AddUnassignedListIteratively(true));
    NN_RESULT_DO(m_BitmapUpdatedVirtual.Clear());
    NN_RESULT_DO(m_BitmapUpdatedPhysical.Clear());
    NN_RESULT_SUCCESS;
}

/**
* @brief        変更を巻き戻します。
*
* @return       関数の処理結果を返します。
*/
Result MappingTable::Rollback() NN_NOEXCEPT
{
    NN_RESULT_DO(AddUnassignedListIteratively(false));
    NN_RESULT_DO(m_BitmapUpdatedVirtual.Clear());
    NN_RESULT_DO(m_BitmapUpdatedPhysical.Clear());
    NN_RESULT_SUCCESS;
}

/*!
* @brief        未割当状態のエントリー数を取得します。
*
* @param[out]   outCount      未割当状態のエントリー数
*
* @return       関数の処理結果を返します。
*
* @details      デバッグ用の機能です。
*
* @pre
*               - outCount != nullptr
*/
Result MappingTable::GetUnassignedCount(uint32_t* outCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);

    uint32_t indexCurr = 0;
    uint32_t countUnassigned = 0;
    dbm::Bitmap::Iterator it;
    NN_RESULT_DO(m_BitmapUnassigned.IterateBegin(&it, 0));
    while( indexCurr < GetTotalBlockCount() )
    {
        uint32_t countZero;
        uint32_t countOne;
        NN_RESULT_DO(m_BitmapUnassigned.IterateNext(&countZero, &countOne, &it));

        // イテレーション終了
        if( (countOne == 0) && (countZero == 0) )
        {
            break;
        }

        countUnassigned += countOne;
        indexCurr += countZero + countOne;
    }

    *outCount = countUnassigned;

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::fssystem::save

