﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fssystem/save/fs_IInternalStorageFileSystemVisitor.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataStorage.h>

namespace nn { namespace fssystem { namespace save {

/**
* @brief        ストレージに必要なサイズを計算します。
*
* @param[out]   outSizeJournalTable                 ジャーナリング用テーブルに必要なサイズ
* @param[out]   outSizeJournalBitmapUpdatedPhysical 更新済み物理ビットマップに必要なサイズ
* @param[out]   outSizeJournalBitmapUpdatedVirtual  更新済み論理ビットマップに必要なサイズ
* @param[out]   outSizeJournalBitmapUnassigned      未割当ビットマップに必要なサイズ
* @param[out]   outIntegritySizeSet                 各階層ハッシュに必要なサイズ
* @param[in]    integrityInputParam                 階層ハッシュのブロックサイズ
* @param[in]    sizeJournalReservedArea             ジャーナリング用予約領域サイズ
* @param[in]    sizeJournalBlock                    ジャーナリング領域のブロックサイズ
* @param[in]    sizeData                            データ領域のサイズ
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataStorage::QuerySize(
           int64_t* outSizeJournalTable,
           int64_t* outSizeJournalBitmapUpdatedPhysical,
           int64_t* outSizeJournalBitmapUpdatedVirtual,
           int64_t* outSizeJournalBitmapUnassigned,
           HierarchicalIntegrityVerificationSizeSet* outIntegritySizeSet,
           const HierarchicalIntegrityVerificationStorageControlArea::InputParam& integrityInputParam,
           int64_t sizeJournalReservedArea,
           int64_t sizeJournalBlock,
           int64_t sizeData
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSizeJournalTable);
    NN_SDK_REQUIRES_NOT_NULL(outSizeJournalBitmapUpdatedPhysical);
    NN_SDK_REQUIRES_NOT_NULL(outSizeJournalBitmapUpdatedVirtual);
    NN_SDK_REQUIRES_NOT_NULL(outSizeJournalBitmapUnassigned);
    NN_SDK_REQUIRES_NOT_NULL(outIntegritySizeSet);

    // 階層ハッシュ検証レイヤーに関するサイズを取得します。
    HierarchicalIntegrityVerificationStorageControlArea::QuerySize(
        outIntegritySizeSet,
        integrityInputParam,
        IntegrityLayerCountSave,
        sizeData
    );

    // ジャーナリングレイヤーに関するサイズを取得します。
    JournalStorage::QueryMappingMetaSize(
        outSizeJournalTable,
        outSizeJournalBitmapUpdatedPhysical,
        outSizeJournalBitmapUpdatedVirtual,
        outSizeJournalBitmapUnassigned,
        sizeData + sizeJournalReservedArea,
        sizeJournalReservedArea,
        sizeJournalBlock
    );

    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージをフォーマットします。
*
* @param[in]    storageJournalControlArea           ジャーナリング管理領域を格納したストレージ
* @param[in]    storageJournalTable                 ジャーナリング用テーブルを格納したストレージ
* @param[in]    storageJournalBitmapUpdatedPhysical 更新済み物理ビットマップを格納したストレージ
* @param[in]    storageJournalBitmapUpdatedVirtual  更新済み論理ビットマップを格納したストレージ
* @param[in]    storageJournalBitmapUnassigned      未割当ビットマップを格納したストレージ
* @param[in]    storageIntegrityMeta                階層ハッシュメタデータを格納したストレージ
* @param[in]    storageIntegrityMasterHash          マスターハッシュを格納したストレージ
* @param[in]    sizeJournalReservedArea             ジャーナリング用予約領域サイズ
* @param[in]    sizeJournalBlock                    ジャーナリング領域のブロックサイズ
* @param[in]    integrityInputMeta                  階層ハッシュのブロックサイズ
* @param[in]    sizeData                            データ領域のサイズ
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataStorage::Format(
           fs::SubStorage storageJournalControlArea,
           fs::SubStorage storageJournalTable,
           fs::SubStorage storageJournalBitmapUpdatedPhysical,
           fs::SubStorage storageJournalBitmapUpdatedVirtual,
           fs::SubStorage storageJournalBitmapUnassigned,
           fs::SubStorage storageIntegrityMeta,
           fs::SubStorage storageIntegrityMasterHash,
           int64_t sizeJournalReservedArea,
           int64_t sizeJournalBlock,
           const HierarchicalIntegrityVerificationMetaInformation& integrityMeta,
           int64_t sizeData
       ) NN_NOEXCEPT
{
    // ジャーナリングレイヤーのメタデータをフォーマットします。
    NN_RESULT_DO(
        JournalStorage::Format(
            storageJournalControlArea,
            storageJournalTable,
            storageJournalBitmapUpdatedPhysical,
            storageJournalBitmapUpdatedVirtual,
            storageJournalBitmapUnassigned,
            sizeData + sizeJournalReservedArea,
            sizeJournalReservedArea,
            sizeJournalBlock
        )
    );

    // 階層ハッシュ検証レイヤーのメタデータをフォーマットします。
    NN_RESULT_DO(
        HierarchicalIntegrityVerificationStorageControlArea::Format(
            storageIntegrityMeta,
            integrityMeta
        )
    );

    // マスターハッシュを消去します。
    int64_t sizeMasterHash = 0;
    NN_RESULT_DO(storageIntegrityMasterHash.GetSize(&sizeMasterHash));
    std::unique_ptr<char[], nn::fs::detail::Deleter> bufMasterHash
        = nn::fs::detail::MakeUnique<char[]>(static_cast<size_t>(sizeMasterHash));
    NN_RESULT_THROW_UNLESS(bufMasterHash != nullptr, nn::fs::ResultAllocationMemoryFailedMakeUnique());
    std::memset(bufMasterHash.get(), 0, static_cast<size_t>(sizeMasterHash));
    NN_RESULT_DO(storageIntegrityMasterHash.Write(0, bufMasterHash.get(), static_cast<size_t>(sizeMasterHash)));

    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージを拡張します。
*
* @param[in]    storageJournalControlArea           ジャーナリング管理領域を格納したストレージ
* @param[in]    storageJournalTable                 ジャーナリング用テーブルを格納したストレージ
* @param[in]    storageJournalBitmapUpdatedPhysical 更新済み物理ビットマップを格納したストレージ
* @param[in]    storageJournalBitmapUpdatedVirtual  更新済み論理ビットマップを格納したストレージ
* @param[in]    storageJournalBitmapUnassigned      未割当ビットマップを格納したストレージ
* @param[in]    storageIntegrityMeta                階層ハッシュメタデータを格納したストレージ
* @param[in]    storageIntegrityMasterHash          マスターハッシュを格納したストレージ
* @param[in]    sizeJournalReservedAreaNew          ジャーナリング用予約領域サイズ
* @param[in]    integrityMeta                       階層ハッシュのパラメータ
* @param[in]    sizeDataNew                         データ領域のサイズ
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataStorage::Expand(
           fs::SubStorage storageJournalControlArea,
           fs::SubStorage storageJournalTable,
           fs::SubStorage storageJournalBitmapUpdatedPhysical,
           fs::SubStorage storageJournalBitmapUpdatedVirtual,
           fs::SubStorage storageJournalBitmapUnassigned,
           fs::SubStorage storageIntegrityMeta,
           fs::SubStorage storageIntegrityMasterHash,
           int64_t sizeJournalReservedAreaNew,
           const HierarchicalIntegrityVerificationMetaInformation& integrityMeta,
           int64_t sizeDataNew
       ) NN_NOEXCEPT
{
    // ジャーナリングレイヤーのメタデータを拡張します。
    NN_RESULT_DO(
        JournalStorage::Expand(
            storageJournalControlArea,
            storageJournalTable,
            storageJournalBitmapUpdatedPhysical,
            storageJournalBitmapUpdatedVirtual,
            storageJournalBitmapUnassigned,
            sizeDataNew + sizeJournalReservedAreaNew,
            sizeJournalReservedAreaNew
        )
    );

    // 階層ハッシュ検証レイヤーのメタデータを拡張します。
    NN_RESULT_DO(
        HierarchicalIntegrityVerificationStorageControlArea::Expand(
            storageIntegrityMeta,
            integrityMeta
        )
    );

    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージをフォーマットします。
*
* @param[in]    storageJournalControlArea           ジャーナリング管理領域を格納したストレージ
* @param[in]    storageJournalTable                 ジャーナリング用テーブルを格納したストレージ
* @param[in]    storageJournalBitmapUpdatedPhysical 更新済み物理ビットマップを格納したストレージ
* @param[in]    storageJournalBitmapUpdatedVirtual  更新済み論理ビットマップを格納したストレージ
* @param[in]    storageJournalBitmapUnassigned      未割当ビットマップを格納したストレージ
* @param[in]    storageIntegrityMasterHash          マスターハッシュを格納したストレージ
* @param[in]    storageIntegrityLayeredhashL1       L1 階層ハッシュを格納したストレージ
* @param[in]    storageIntegrityLayeredhashL2       L2 階層ハッシュを格納したストレージ
* @param[in]    storageIntegrityLayeredhashL3       L3 階層ハッシュを格納したストレージ
* @param[in]    storageData                         実データを格納したストレージ
* @param[in]    integrityInfo                       階層ハッシュのパラメータ
* @param[in]    pBufferManagerSet                   ハッシュ検証用キャッシュマネージャ
* @param[in]    pLocker                             キャッシュマネージャのスレッド間同期オブジェクト
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataStorage::Initialize(
           fs::SubStorage storageJournalControlArea,
           fs::SubStorage storageJournalMeta,
           fs::SubStorage storageJournalBitmapUpdatedPhysical,
           fs::SubStorage storageJournalBitmapUpdatedVirtual,
           fs::SubStorage storageJournalBitmapUnassigned,
           fs::SubStorage storageIntegrityMasterHash,
           fs::SubStorage storageIntegrityLayeredHashL1,
           fs::SubStorage storageIntegrityLayeredHashL2,
           fs::SubStorage storageIntegrityLayeredHashL3,
           fs::SubStorage storageData,
           const HierarchicalIntegrityVerificationInformation& integrityInfo,
           FilesystemBufferManagerSet* pBufferManagerSet,
           nn::os::Mutex* pLocker
       ) NN_NOEXCEPT
{
    // ジャーナリングレイヤーをマウントします。
    NN_RESULT_DO(
        m_JournalStorage.Initialize(
            storageJournalControlArea,
            storageJournalMeta,
            storageJournalBitmapUpdatedPhysical,
            storageJournalBitmapUpdatedVirtual,
            storageJournalBitmapUnassigned,
            storageData
        )
    );

    // 階層ハッシュ検証レイヤーをマウントします。
    HierarchicalIntegrityVerificationStorage::HierarchicalStorageInformation storageInfo;
    storageInfo.SetMasterHashStorage(storageIntegrityMasterHash);
    storageInfo.SetLayer1HashStorage(storageIntegrityLayeredHashL1);
    storageInfo.SetLayer2HashStorage(storageIntegrityLayeredHashL2);
    storageInfo.SetLayer3HashStorage(storageIntegrityLayeredHashL3);
    storageInfo.SetDataStorage(m_JournalStorage.CreateDataStorage());
    NN_RESULT_DO(
        m_IntegrityStorage.Initialize(
            integrityInfo,
            storageInfo,
            pBufferManagerSet,
            pLocker,
            fs::StorageType::StorageType_SaveData
        )
    );

    NN_RESULT_SUCCESS;
}

/**
* @brief        範囲を指定して操作を実行します。
*
* @param[out]   outBuffer       操作の結果を格納するバッファ
* @param[in]    outBufferSize   操作の結果を格納するバッファのサイズ
* @param[in]    operationId     実行する操作
* @param[in]    offset          操作を実行する範囲の先頭オフセット
* @param[in]    size            操作を実行する範囲のサイズ
* @param[in]    inBuffer        操作に渡すバッファ
* @param[in]    inBufferSize    操作に渡すバッファのサイズ
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataStorage::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           fs::OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    switch( operationId )
    {
    case fs::OperationId::FillZero:
    case fs::OperationId::DestroySignature:
    case fs::OperationId::QueryRange:
        return m_IntegrityStorage.OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize);
    case fs::OperationId::Invalidate:
    default:
        return nn::fs::ResultUnsupportedOperation();
    }
}

/**
* @brief        コミット操作を実行します。
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataStorage::Commit() NN_NOEXCEPT
{
    // ハッシュ検証 -> ジャーナリングの順でコミットを行います。
    NN_RESULT_DO(m_IntegrityStorage.Commit());
    NN_RESULT_DO(m_JournalStorage.Commit());
    NN_RESULT_SUCCESS;
}

/**
* @brief        ロールバック前処理を行います。
*
* @return       関数の処理結果を返します。
*/
Result JournalIntegritySaveDataStorage::OnRollback() NN_NOEXCEPT
{
    NN_RESULT_DO(m_IntegrityStorage.OnRollback());
    NN_RESULT_SUCCESS;
}

// セーブデータ内部ストレージアクセス用の Vistor を受け入れます。
nn::Result JournalIntegritySaveDataStorage::AcceptVisitor(IInternalStorageFileSystemVisitor* pVisitor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pVisitor);
    NN_RESULT_DO(pVisitor->VisitStorage(InternalStorageFileNameJournal, &m_JournalStorage));
    NN_RESULT_DO(pVisitor->VisitStorage(InternalStorageFileNameIntegrity, &m_IntegrityStorage));
    NN_RESULT_SUCCESS;
}

const char JournalIntegritySaveDataStorage::InternalStorageFileNameJournal[]               = "InternalStorageFileNameJournal";
const char JournalIntegritySaveDataStorage::InternalStorageFileNameIntegrity[]             = "InternalStorageFileNameIntegrity";
const char JournalIntegritySaveDataStorage::InternalStorageFileNameIntegrityWithZeroFree[] = "InternalStorageFileNameIntegrityWithZeroFree";

}}}

