﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <vector>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fssystem/save/fs_SaveStorageFile.h>
#include <nn/fssystem/save/fs_IntegritySaveDataFileSystemDriver.h>
#include <nn/fssystem/save/fs_UnionStorage.h>

namespace nn { namespace fssystem { namespace save {

IntegritySaveDataFileSystemDriver::IntegritySaveDataFileSystemDriver() NN_NOEXCEPT
    : m_Locker(true)
{
}

IntegritySaveDataFileSystemDriver::~IntegritySaveDataFileSystemDriver() NN_NOEXCEPT
{
    Finalize();
}

// セーブデータをフォーマットするのに必要なデータサイズを計算します。
Result IntegritySaveDataFileSystemDriver::QueryTotalSize(
           int64_t* outSizeTotal,
           size_t sizeBlock,
           const HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
           uint32_t countDataBlock
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSizeTotal);
    NN_RESULT_THROW(IntegritySaveDataFileSystem::QuerySize(
               outSizeTotal,
               paramIntegrity,
               static_cast<uint32_t>(sizeBlock),
               countDataBlock
    ));
}

// セーブデータ領域をフォーマットします。
Result IntegritySaveDataFileSystemDriver::Format(
           fs::SubStorage baseStorage,
           size_t sizeBlock,
           const HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
           uint32_t countDataBlock,
           IBufferManager* pBufferManager,
           IMacGenerator* pMacGenerator
       ) NN_NOEXCEPT
{
    // ファイルアクセスはこの関数内で完結するので、
    // ローカルの排他制御オブジェクトで構いません
    nn::os::Mutex locker(true);

    FilesystemBufferManagerSet bufferManagerSet;
    for( auto& pBuffer : bufferManagerSet.pBuffer )
    {
        pBuffer = pBufferManager;
    }

    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [&]() NN_NOEXCEPT -> Result
        {
            buffers::ScopedBufferManagerContextRegistration scopedRegistration;
            NN_RESULT_DO(IntegritySaveDataFileSystem::Format(
                baseStorage,
                paramIntegrity,
                static_cast<uint32_t>(sizeBlock),
                countDataBlock,
                &bufferManagerSet,
                pBufferManager,
                &locker,
                pMacGenerator
            ));
            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));

    NN_RESULT_SUCCESS;
}

//! 拡張データを読み込みます
Result IntegritySaveDataFileSystemDriver::ReadExtraData(
           IntegritySaveDataFileSystem::ExtraData* outData,
           fs::SubStorage baseStorage,
           IBufferManager* pBufferManager,
           IMacGenerator* pMacGenerator
       ) NN_NOEXCEPT
{
    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [=]() NN_NOEXCEPT -> Result
        {
            buffers::ScopedBufferManagerContextRegistration scopedRegistration;
            NN_RESULT_DO(IntegritySaveDataFileSystem::ReadExtraData(outData, baseStorage, pBufferManager, pMacGenerator));
            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));
    NN_RESULT_SUCCESS;
}

//! 一時揮発ファイルシステムをマウントします。
Result IntegritySaveDataFileSystemDriver::Initialize(fs::SubStorage baseStorage, IBufferManager* pBufferManager, IMacGenerator* pMacGenerator) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBufferManager);

    m_BaseStorage = baseStorage;

    m_pBufferManager = pBufferManager;
    for( auto& pBuffer : m_BufferManagerSet.pBuffer )
    {
        pBuffer = pBufferManager;
    }

    // m_SaveDataFileSystem.Initialize はリトライできないつくりなので、
    // 常にブロッキングでバッファを確保する
    buffers::ScopedBufferManagerContextRegistration scopedRegistration;
    buffers::EnableBlockingBufferManagerAllocation();

    // セーブデータをマウントします。
    NN_RESULT_DO(
        m_SaveDataFileSystem.Initialize(
            m_BaseStorage,
            &m_BufferManagerSet,
            m_pBufferManager,
            &m_Locker,
            pMacGenerator
        )
    );

    ProxyFileSystemWithRetryingBufferAllocation::Initialize(&m_SaveDataFileSystem);

    NN_RESULT_SUCCESS;
}

//! セーブデータをアンマウントします。
void IntegritySaveDataFileSystemDriver::Finalize() NN_NOEXCEPT
{
    // Result 返ってこないので、常に末端でリトライ
    buffers::ScopedBufferManagerContextRegistration scopedRegistration;
    buffers::EnableBlockingBufferManagerAllocation();

    m_SaveDataFileSystem.Finalize();
    ProxyFileSystemWithRetryingBufferAllocation::Finalize();
}

//! 拡張データを更新します
Result IntegritySaveDataFileSystemDriver::WriteExtraData(
           const IntegritySaveDataFileSystem::ExtraData& extraData
       ) NN_NOEXCEPT
{
    NN_RESULT_THROW(m_SaveDataFileSystem.WriteExtraData(extraData));
}

//! 拡張データを取得します
void IntegritySaveDataFileSystemDriver::ReadExtraData(
         IntegritySaveDataFileSystem::ExtraData* outData
     ) NN_NOEXCEPT
{
    m_SaveDataFileSystem.ReadExtraData(outData);
}

//! セーブデータへの変更をコミットします。
Result IntegritySaveDataFileSystemDriver::Commit() NN_NOEXCEPT
{
    NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
        [=]() NN_NOEXCEPT -> Result
        {
            buffers::ScopedBufferManagerContextRegistration scopedRegistration;
            buffers::EnableBlockingBufferManagerAllocation();
            NN_RESULT_DO(m_SaveDataFileSystem.CommitFileSystem());
            NN_RESULT_SUCCESS;
        },
        NN_CURRENT_FUNCTION_NAME
    ));
    NN_RESULT_SUCCESS;
}

//! セーブデータ管理領域のパラメータを自動設定します。
IntegritySaveDataParameters IntegritySaveDataFileSystemDriver::SetUpSaveDataParameters(int32_t blockSize, int64_t size) NN_NOEXCEPT
{
    IntegritySaveDataParameters params = {};

    params.paramIntegrity.sizeBlockLevel[0] = blockSize;
    params.paramIntegrity.sizeBlockLevel[1] = blockSize;
    params.paramIntegrity.sizeBlockLevel[2] = blockSize;
    params.paramIntegrity.sizeBlockLevel[3] = blockSize;

    // size = 利用可能サイズとする
    params.countDataBlock = static_cast<uint32_t>((size + (blockSize - 1)) / blockSize);
    params.blockSize = blockSize;

    return params;
}

}}}
