﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/os.h>
#include <nn/crypto.h>
#include <nn/fs/fs_Result.h>
#include <nn/fssystem/save/fs_HierarchicalIntegrityVerificationStorage.h>

namespace nn { namespace fssystem { namespace save {

namespace
{
    // マジックコード
    const uint32_t IVFS_MAGIC_CODE = 0x43465649;  // 'IVFS'

    // バージョン
    const uint32_t IVFS_VERSION_CODE = 0x00020000;
    const uint32_t IVFS_VERSION_CODE_MASK = 0xFFFF0000;

    const auto CountMaxCacheEntriesSaveDataFsData = 32;
    const auto CountMaxCacheEntriesSaveDataFsHash = 4;
    const auto CountMaxCacheEntriesRomFsData = 24;
    const auto CountMaxCacheEntriesRomFsHash = 8;

    const auto AccessCountMax = 5;
    const auto AccessTimeout = nn::TimeSpan::FromMilliSeconds(10);
    nn::os::Semaphore g_ReadSemaphore(AccessCountMax, AccessCountMax);
    nn::os::Semaphore g_WriteSemaphore(AccessCountMax, AccessCountMax);
}

//! メタデータ領域を初期化します。
void HierarchicalIntegrityVerificationMetaInformation::Format() NN_NOEXCEPT
{
    nn::fs::SaveDataHashSalt hashSalt;
    Format(hashSalt);
}

void HierarchicalIntegrityVerificationMetaInformation::Format(const nn::fs::SaveDataHashSalt& hashSalt) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(HierarchicalIntegrityVerificationStorage::m_pGenerateRandom);

    this->magic = IVFS_MAGIC_CODE;
    this->version = IVFS_VERSION_CODE;
    this->sizeMasterHash = 0;
    std::memset(&this->infoLevelHash, 0, sizeof(this->infoLevelHash));

    // 混ぜ物
    if( hashSalt )
    {
        std::memcpy(this->infoLevelHash.seed.value, hashSalt.value().value, sizeof(this->infoLevelHash.seed));
    }
    else
    {
        HierarchicalIntegrityVerificationStorage::m_pGenerateRandom(
            this->infoLevelHash.seed.value, sizeof(this->infoLevelHash.seed));
    }
}

/**
* @brief        完全性検証ストレージの総データサイズを取得します。
*
* @param[out]   outSizeSet          サイズ
* @param[in]    inputParam          入力パラメータ
* @param[in]    sizeData            実データサイズ
*
* @return       関数の処理結果を返します。
*
* @details      レイヤーハッシュと本体のサイズを別計算します
*/
Result HierarchicalIntegrityVerificationStorageControlArea::QuerySize(
           HierarchicalIntegrityVerificationSizeSet* outSizeSet,
           const InputParam& inputParam,
           int32_t countLayer,
           int64_t sizeData
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outSizeSet);
    NN_SDK_ASSERT((countLayer >= static_cast<int32_t>(IntegrityMinLayerCount))
               && (countLayer <= static_cast<int32_t>(IntegrityMaxLayerCount)));
    for( int32_t level = 0; level < (countLayer - 1); ++level )
    {
        NN_SDK_ASSERT(
            (inputParam.sizeBlockLevel[level] > 0)
         && IsPower2(static_cast<int>(inputParam.sizeBlockLevel[level]))
        );
    }

    // 管理領域サイズ
    outSizeSet->controlSize = sizeof(HierarchicalIntegrityVerificationMetaInformation);

    // それぞれのデータサイズを求ます。
    int64_t sizeLevel[IntegrityMaxLayerCount];

    int32_t level = countLayer - 1;

    // 実データ層
    sizeLevel[level] = nn::util::align_up(sizeData, inputParam.sizeBlockLevel[level - 1]);
    level--;

    for( ; level > 0; --level )
    {
        sizeLevel[level] = nn::util::align_up(sizeLevel[level + 1] / inputParam.sizeBlockLevel[level] * HashSize, inputParam.sizeBlockLevel[level - 1]);
    }

    // Master
    sizeLevel[0] = sizeLevel[1] / inputParam.sizeBlockLevel[0] * HashSize;

    // ルートハッシュのサイズが確定しました。
    outSizeSet->masterHashSize = sizeLevel[0];

    // レイヤーハッシュサイズ
    for( level = 1; level < countLayer - 1; ++level )
    {
        outSizeSet->layeredHashSizes[level - 1] = sizeLevel[level];
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        完全性検証ストレージ用メタデータ領域をフォーマットします。
*
* @param[in]    storageMeta メタデータをフォーマットするストレージ
* @param[in]    meta        メタデータ
*
* @return       関数の処理結果を返します。
*
* @details      この関数では以下の処理を行います。
*
*               ・管理領域の生成。
*               ・あらかじめ計算されたハッシュ領域、実データ領域の、それぞれのオフセット、サイズを管理領域に覚えます。
*                 オフセット情報はフォーマット後は基本的に読み取り専用のデータですが、拡張時に上書きされる可能性があります。
*/
Result HierarchicalIntegrityVerificationStorageControlArea::Format(
           fs::SubStorage storageMeta,
           const HierarchicalIntegrityVerificationMetaInformation& meta
       ) NN_NOEXCEPT
{
    {
        int64_t sizeMeta = 0;
        NN_RESULT_DO(storageMeta.GetSize(&sizeMeta));
        if( sizeMeta < static_cast<int64_t>(sizeof(meta)) )
        {
            return nn::fs::ResultInvalidSize();
        }
    }

    if( meta.magic != IVFS_MAGIC_CODE )
    {
        return nn::fs::ResultIncorrectIntegrityVerificationMagicCode();
    }

    if( meta.version != IVFS_VERSION_CODE )
    {
        return nn::fs::ResultUnsupportedVersion();
    }

    // メタデータを書き込みます。
    NN_RESULT_DO(storageMeta.Write(0, &meta, sizeof(meta)));
    NN_RESULT_DO(storageMeta.Flush());

    // オプションデータ領域には何も書き込みません。

    NN_RESULT_SUCCESS;
}

/**
* @brief        完全性検証ストレージ用メタデータ領域を拡張します。
*
* @param[in]    storageMeta メタデータを再フォーマットするストレージ
* @param[in]    meta        拡張後のメタデータ
*
* @return       関数の処理結果を返します。
*
* @details      この関数では以下の処理を行います。
*
*               ・あらかじめ計算されたハッシュ領域、実データ領域の、それぞれのオフセット、サイズを管理領域に覚え直します。
*/
Result HierarchicalIntegrityVerificationStorageControlArea::Expand(
           fs::SubStorage storageMeta,
           const HierarchicalIntegrityVerificationMetaInformation& meta
       ) NN_NOEXCEPT
{
    {
        int64_t sizeMeta = 0;
        NN_RESULT_DO(storageMeta.GetSize(&sizeMeta));
        if( sizeMeta < static_cast<int64_t>(sizeof(meta)) )
        {
            return nn::fs::ResultInvalidSize();
        }
    }

    {
        // 拡張前のメタデータを読み込みます。
        HierarchicalIntegrityVerificationMetaInformation metaPrevious = {};
        NN_RESULT_DO(storageMeta.Read(0, &metaPrevious, sizeof(metaPrevious)));

        // メタデータを検査します
        if( meta.magic != IVFS_MAGIC_CODE || metaPrevious.magic != meta.magic )
        {
            return nn::fs::ResultIncorrectIntegrityVerificationMagicCode();
        }

        if( meta.version != IVFS_VERSION_CODE || metaPrevious.version != meta.version )
        {
            return nn::fs::ResultUnsupportedVersion();
        }
    }

    // メタデータを書き込みます。
    NN_RESULT_DO(storageMeta.Write(0, &meta, sizeof(meta)));
    NN_RESULT_DO(storageMeta.Flush());

    // オプションデータ領域には何も書き込みません。

    NN_RESULT_SUCCESS;
}

/**
* @brief        完全性検証ストレージ用メタデータをマウントします。
*
* @param[in]    storageMeta メタデータストレージ
*
* @return       関数の処理結果を返します。
*/
Result HierarchicalIntegrityVerificationStorageControlArea::Initialize(
           fs::SubStorage storageMeta
       ) NN_NOEXCEPT
{
    {
        int64_t sizeMeta = 0;
        NN_RESULT_DO(storageMeta.GetSize(&sizeMeta));
        if( sizeMeta < static_cast<int64_t>(sizeof(m_Meta)) )
        {
            return nn::fs::ResultInvalidSize();
        }
    }

    m_Storage = storageMeta;

    // メタデータを読み込みます。
    NN_RESULT_DO(m_Storage.Read(0, &m_Meta, sizeof(m_Meta)));

    // メタデータのバージョンとマジックコードを取得します。
    if( m_Meta.magic != IVFS_MAGIC_CODE )
    {
        return nn::fs::ResultIncorrectIntegrityVerificationMagicCode();
    }
    if( (m_Meta.version & IVFS_VERSION_CODE_MASK) != (IVFS_VERSION_CODE & IVFS_VERSION_CODE_MASK) )
    {
        return nn::fs::ResultUnsupportedVersion();
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        完全性検証ストレージ用メタデータをアンマウントします。
*/
void HierarchicalIntegrityVerificationStorageControlArea::Finalize() NN_NOEXCEPT
{
    m_Storage = fs::SubStorage();
}

/**
* @brief        コンストラクタ
*/
HierarchicalIntegrityVerificationStorage::HierarchicalIntegrityVerificationStorage() NN_NOEXCEPT
    : m_pBuffers(nullptr),
      m_pLocker(nullptr),
      m_SizeData(-1),
      m_IsWrittenForRollback(false)
{
}

/**
* @brief        デストラクタ
*/
HierarchicalIntegrityVerificationStorage::~HierarchicalIntegrityVerificationStorage() NN_NOEXCEPT
{
    Finalize();
}

/**
* @brief        完全性検証ストレージをマウントします。
*
* @param[in]    info                    L1, L2, L3, 実データの情報
* @param[in]    storage                 マスターハッシュ、L1 ～ Ln ハッシュ、実データストレージ
* @param[in]    pBuffer                 キャッシュバッファマネージャ
* @param[in]    pLocker                 排他制御用のロックオブジェクト
* @param[in]    storageType             使用するストレージのタイプ
*
* @return      関数の処理結果を返します。
*
* @details     完全性検証ストレージレイヤーをフォーマットする、という操作は不要です。
*              使用しているうちに、自動的にフォーマット相当の処理が進行していきます。
*/
Result HierarchicalIntegrityVerificationStorage::Initialize(
           const HierarchicalIntegrityVerificationInformation& info,
           HierarchicalStorageInformation storage,
           FilesystemBufferManagerSet* pBuffers,
           os::Mutex* pLocker,
           fs::StorageType storageType
       ) NN_NOEXCEPT
{
    Result result;

    NN_SDK_ASSERT_NOT_NULL(pBuffers);
    NN_SDK_ASSERT((info.maxLayers >= IntegrityMinLayerCount) && (info.maxLayers <= IntegrityMaxLayerCount));

    // 最大レイヤー
    m_MaxLayers = info.maxLayers;

    // 階層化キャッシュをコピー
    m_pBuffers = pBuffers;

    // 同期オブジェクトをコピー
    m_pLocker = pLocker;

    static const char MasterKey[] = "HierarchicalIntegrityVerificationStorage::Master";
    static const char L1Key[] = "HierarchicalIntegrityVerificationStorage::L1";
    static const char L2Key[] = "HierarchicalIntegrityVerificationStorage::L2";
    static const char L3Key[] = "HierarchicalIntegrityVerificationStorage::L3";
    static const char L4Key[] = "HierarchicalIntegrityVerificationStorage::L4";
    static const char L5Key[] = "HierarchicalIntegrityVerificationStorage::L5";
    struct {
        const char* key;
        size_t size;
    } const KeyArray[] = {
        { MasterKey, sizeof(MasterKey) },
        { L1Key, sizeof(L1Key) },
        { L2Key, sizeof(L2Key) },
        { L3Key, sizeof(L3Key) },
        { L4Key, sizeof(L4Key) },
        { L5Key, sizeof(L5Key) }
    };

    const auto countMaxCacheEntriesData
        = storageType == fs::StorageType_SaveData
        ? CountMaxCacheEntriesSaveDataFsData : CountMaxCacheEntriesRomFsData;
    const auto countMaxCacheEntriesHash
        = storageType == fs::StorageType_SaveData
        ? CountMaxCacheEntriesSaveDataFsHash : CountMaxCacheEntriesRomFsHash;

    // 階層的に初期化します。
    {
        nn::fs::HashSalt mac;
        nn::crypto::GenerateHmacSha256Mac(
            mac.value, sizeof(mac),
            info.seed.value, sizeof(info.seed),
            KeyArray[0].key, KeyArray[0].size
        );
        m_StorageVerify[0].Initialize(
            storage[HierarchicalStorageInformation::MasterStorage],
            storage[HierarchicalStorageInformation::Layer1Storage],
            static_cast<int64_t>(1) << info.info[0].orderBlock,
            HASH_SIZE,
            m_pBuffers->pBuffer[m_MaxLayers - 2],
            mac,
            false,
            storageType
        );
    }
    result = m_StorageBuffer[0].Initialize(
                 m_pBuffers->pBuffer[0],
                 m_pLocker,
                 &m_StorageVerify[0],
                 info.info[0].size,
                 static_cast<int64_t>(1) << info.info[0].orderBlock,
                 countMaxCacheEntriesHash,
                 false,                // isReadDataCache
                 0x10,                 // bufferLevel
                 false,                // isKeepBurstMode
                 storageType
             );
    if( !result.IsFailure() )
    {
        int32_t level = 0;
        for( ; level < (m_MaxLayers - 3); ++level)
        {
            // 1 階層目以降はオフセット使用しません。
            {
                fs::SubStorage storageBuffer(&m_StorageBuffer[level], 0, info.info[level].size);
                nn::fs::HashSalt mac;
                nn::crypto::GenerateHmacSha256Mac(
                    mac.value, sizeof(mac),
                    info.seed.value, sizeof(info.seed),
                    KeyArray[level + 1].key, KeyArray[level + 1].size
                );
                m_StorageVerify[level + 1].Initialize(
                    storageBuffer,
                    storage[level + 2],
                    static_cast<int64_t>(1) << info.info[level + 1].orderBlock,
                    static_cast<int64_t>(1) << info.info[level].orderBlock,
                    m_pBuffers->pBuffer[m_MaxLayers - 2],
                    mac,
                    false,                // isReadDataCache
                    storageType
                );
            }
            result = m_StorageBuffer[level + 1].Initialize(
                         m_pBuffers->pBuffer[level + 1],
                         m_pLocker,
                         &m_StorageVerify[level + 1],
                         info.info[level + 1].size,
                         static_cast<int64_t>(1) << info.info[level + 1].orderBlock,
                         countMaxCacheEntriesHash,
                         false,                             // isReadDataCache
                         0x11 + static_cast<int8_t>(level), // bufferLevel
                         false,                             // isKeepBurstMode
                         storageType
                     );
            if( result.IsFailure() )
            {
                m_StorageVerify[level + 1].Finalize();
                for( ; level > 0; --level )
                {
                    m_StorageBuffer[level].Finalize();
                    m_StorageVerify[level].Finalize();
                }
                break;
            }
        }
        if( !result.IsFailure() )
        {
            // 1 階層目以降はオフセット使用しません。
            {
                fs::SubStorage storageBuffer(&m_StorageBuffer[level], 0, info.info[level].size);
                nn::fs::HashSalt mac;
                nn::crypto::GenerateHmacSha256Mac(
                    mac.value, sizeof(mac),
                    info.seed.value, sizeof(info.seed),
                    KeyArray[level + 1].key, KeyArray[level + 1].size
                );
                m_StorageVerify[level + 1].Initialize(
                    storageBuffer,
                    storage[HierarchicalStorageInformation::DataStorage],
                    static_cast<int64_t>(1) << info.info[level + 1].orderBlock,
                    static_cast<int64_t>(1) << info.info[level].orderBlock,
                    m_pBuffers->pBuffer[m_MaxLayers - 2],
                    mac,
                    true,           // isReadDataCache
                    storageType
                );
            }
            result = m_StorageBuffer[level + 1].Initialize(
                         m_pBuffers->pBuffer[level + 1],
                         m_pLocker,
                         &m_StorageVerify[level + 1],
                         info.info[level + 1].size,
                         static_cast<int64_t>(1) << info.info[level + 1].orderBlock,
                         countMaxCacheEntriesData,
                         true,                              // isReadDataCache
                         0x11 + static_cast<int8_t>(level), // bufferLevel
                         true,                              // isKeepBurstMode
                         storageType
                     );
            if( !result.IsFailure() )
            {
                m_SizeData = info.info[level + 1].size;
                NN_RESULT_SUCCESS;
            }
            m_StorageVerify[level + 1].Finalize();
            for( ; level > 0; --level )
            {
                m_StorageBuffer[level].Finalize();
                m_StorageVerify[level].Finalize();
            }
        }
        m_StorageBuffer[0].Finalize();
    }
    m_StorageVerify[0].Finalize();

    m_SizeData = -1;

    m_pBuffers = nullptr;
    m_pLocker = nullptr;

    // こちらはエラー終了
    return result;
} // NOLINT(impl/function_size)

/**
* @brief        完全検証ストレージを閉じます。
*/
void HierarchicalIntegrityVerificationStorage::Finalize() NN_NOEXCEPT
{
    // 初期化済みの場合のみ閉じる処理を行ないます。
    if( m_SizeData >= 0LL )
    {
        m_SizeData = 0;

        m_pBuffers = nullptr;
        m_pLocker = nullptr;

        // ストレージを閉じる順番に注意
        for( int32_t level = m_MaxLayers - 2; level >= 0; --level )
        {
            m_StorageBuffer[level].Finalize();
            m_StorageVerify[level].Finalize();
        }

        m_SizeData = -1;
    }
}

/**
* @brief        実データの総バイトサイズを取得します。
*
* @param[out]   outValue    実データの総バイトサイズ
*
* @return       関数の処理結果を返します。
*/
Result HierarchicalIntegrityVerificationStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outValue);
    NN_SDK_ASSERT(m_SizeData >= 0);
    *outValue = m_SizeData;
    NN_RESULT_SUCCESS;
}

/**
* @brief        ハッシュ検証つきで実データ領域からデータを読み込みます。
*
* @param[in]    offset  読み込み開始位置
* @param[out]   buffer  読み込んだ内容をコピーするバッファ
* @param[in]    size    読み込むデータサイズ
*
* @return       関数の処理結果を返します。
*
* @details      読み込もうとしたデータに対するハッシュが不正だった場合には、0 フィルされたデータが
*               読込先バッファに書き込まれ、nn::fs::ResultIntegrityVerificationStorageCorrupted を返します。
*/
Result HierarchicalIntegrityVerificationStorage::Read(
           int64_t offset,
           void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_SizeData >= 0);

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());

    if( !g_ReadSemaphore.TimedAcquire(AccessTimeout) )
    {
        for( auto level = m_MaxLayers - 2; 0 <= level; --level )
        {
            NN_RESULT_DO(m_StorageBuffer[level].Flush());
        }
        g_ReadSemaphore.Acquire();
    }
    NN_UTIL_SCOPE_EXIT
    {
        g_ReadSemaphore.Release();
    };
    NN_RESULT_DO(m_StorageBuffer[m_MaxLayers - 2].Read(offset, buffer, size));
    NN_RESULT_SUCCESS;
}

/**
* @brief        実データ領域にデータを書き込み、ハッシュを更新します。
*
* @param[in]    offset  書き込み開始オフセット
* @param[in]    buffer  書き込むデータ
* @param[in]    size    書き込むサイズ
*
* @return       関数の処理結果を返します。
*/
Result HierarchicalIntegrityVerificationStorage::Write(
           int64_t offset,
           const void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_SizeData >= 0);

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());

    if( !g_WriteSemaphore.TimedAcquire(AccessTimeout) )
    {
        for( auto level = m_MaxLayers - 2; 0 <= level; --level )
        {
            NN_RESULT_DO(m_StorageBuffer[level].Flush());
        }
        g_WriteSemaphore.Acquire();
    }
    NN_UTIL_SCOPE_EXIT
    {
        g_WriteSemaphore.Release();
    };
    NN_RESULT_DO(m_StorageBuffer[m_MaxLayers - 2].Write(offset, buffer, size));
    m_IsWrittenForRollback = true;
    NN_RESULT_SUCCESS;
}

/**
* @brief        フラッシュします。
*
* @return       関数の処理結果を返します。
*/
Result HierarchicalIntegrityVerificationStorage::Flush() NN_NOEXCEPT
{
    // 過度なストレージへのアクセスを防ぐため、Commit による Flush 処理に任せる。
    // ここで Flush しても Commit するまで書き込みは保証されないので、
    // ここでは Flush は実行しなくてもよい。
    NN_RESULT_SUCCESS;
}

/**
* @brief       範囲指定処理を行います。
*
* @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
* @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
* @param[in]   operationId      範囲指定処理の種類
* @param[in]   offset           範囲指定処理開始位置
* @param[in]   size             範囲指定処理を行うデータサイズ
* @param[in]   inBuffer         範囲指定処理に渡すバッファ
* @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
*
* @return      関数の処理結果を返します。
*/
Result HierarchicalIntegrityVerificationStorage::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           fs::OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    switch( operationId )
    {
    case fs::OperationId::FillZero:
    case fs::OperationId::DestroySignature:
        {
            NN_RESULT_DO(
                m_StorageBuffer[m_MaxLayers - 2].OperateRange(
                    outBuffer,
                    outBufferSize,
                    operationId,
                    offset,
                    size,
                    inBuffer,
                    inBufferSize
                )
            );
            // 書き込みを伴う OperateRange では書き込みフラグを有効にします。
            m_IsWrittenForRollback = true;
            NN_RESULT_SUCCESS;
        }
    case fs::OperationId::Invalidate:
    case fs::OperationId::QueryRange:
        {
            NN_RESULT_DO(
                m_StorageBuffer[m_MaxLayers - 2].OperateRange(
                    outBuffer,
                    outBufferSize,
                    operationId,
                    offset,
                    size,
                    inBuffer,
                    inBufferSize
                )
            );
            NN_RESULT_SUCCESS;
        }
    default:
        return nn::fs::ResultUnsupportedOperation();
    }
}

/**
* @brief 完全性検証ストレージをコミットします。
*
* @return 関数の処理結果を返します。
*/
Result HierarchicalIntegrityVerificationStorage::Commit() NN_NOEXCEPT
{
    for( int32_t level = m_MaxLayers - 2; level >= 0; --level )
    {
        NN_RESULT_DO(m_StorageBuffer[level].Commit());
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        完全性検証ストレージへの変更を破棄します。
*
* @return       関数の処理結果を返します。
*/
Result HierarchicalIntegrityVerificationStorage::OnRollback() NN_NOEXCEPT
{
    for( int32_t level = m_MaxLayers - 2; level >= 0; --level )
    {
        NN_RESULT_DO(m_StorageBuffer[level].OnRollback());
    }
    m_IsWrittenForRollback = false;
    NN_RESULT_SUCCESS;
}

/**
* @brief        乱数列生成関数をセットします。
*
* @return       なし
*/
void HierarchicalIntegrityVerificationStorage::SetGenerateRandomFunction(
            GenerateRandomFunction function
        ) NN_NOEXCEPT
{
    m_pGenerateRandom = function;
}
HierarchicalIntegrityVerificationStorage::GenerateRandomFunction HierarchicalIntegrityVerificationStorage::m_pGenerateRandom = nullptr;

}}}

